/* lux.c,
 *
 * Luces are used to simply brighten up an area by adding some extra
 * colour things, which compensates for 'game_darkening_magic'.
 *
 * Lights in the 'manual list' are updated by the creator.  After the
 * creator dies, the light should be moved to the 'automatic list'
 * where it will fade away.
 */

#include <alleggl.h>
#include <assert.h>
#include <math.h>
#include <stdbool.h>
#include <stdlib.h>
#include "common.h"
#include "lux.h"
#include "list.h"
#include "universal.h"


static int lux_count;
static list_head_t lux_automatic_list;	/* Client only */
static list_head_t lux_manual_list;	/* Client only */

/*--------------------------------------------------------------*/

static bool lux_update_unit(lux_t *l);
static void lux_update_colour(lux_t *l);
static void lux_free(lux_t *l);


void lux_update(void)
{
    lux_t *l, *nx;

    list_for_each_next(l, nx, &lux_automatic_list) {
	if (!lux_update_unit(l))
	    lux_free(l);
    }
}


static bool lux_update_unit(lux_t *l)
{
    assert(l);

    l->life -= 1000.0*SECONDS_PER_TICK;
    if (l->life <= 0)
	return false;

    lux_update_colour(l);
    return true;
}


static void lux_update_colour(lux_t *l)
{
    int i;
    assert(l);

    if (l->life <= 0) {
	for (i = 0; i < 4; i++)
	    l->colour[i] = 0.0;
    }
    else {
	double progress = (double)l->life/l->_spawn_life;

	for (i = 0; i < 4; i++)
	    l->colour[i] = l->_spawn_colour[i] * progress;
    }
}

/*--------------------------------------------------------------*/

static void lux_draw_unit(const lux_t *l);


void lux_draw(void)
{
    lux_t *l;

    if (list_empty(lux_automatic_list) &&
	list_empty(lux_manual_list))
	return;

    glDisable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_ALPHA_TEST);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);
    glBlendEquation(GL_FUNC_ADD);

    list_for_each(l, &lux_automatic_list) {
	lux_draw_unit(l);
    }

    list_for_each(l, &lux_manual_list) {
	lux_draw_unit(l);
    }

    glDisable(GL_BLEND);
    glEnable(GL_ALPHA_TEST);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glColor3fv(glWhite);
}


static void lux_draw_unit(const lux_t *l)
{
    int i;
    assert(l);

    glPushMatrix();
    glTranslated(l->_spawn_x, l->_spawn_y, 0.0);
    glScaled(l->r, l->r, l->r);
    glBegin(GL_TRIANGLE_FAN);
    glColor4fv(l->colour);

    glVertex2d(0.0, 0.0);
    for (i = 0; i <= 32; i++)
	glVertex2d(cos(2*M_PI*i/32), sin(2*M_PI*i/32));

    glEnd();			/* glBegin(GL_TRIANGLE_FAN) */
    glPopMatrix();
}

/*--------------------------------------------------------------*/

/* lux_new:
 *
 * Create a new lux with various parameters.  The light is added to
 * the manual list.
 */
lux_t *lux_new(const double x, const double y, const int life,
	       const GLfloat col[4])
{
    int i;
    lux_t *l = malloc(sizeof(*l));
    assert(l);

    /* Spawn state. */
    for (i = 0; i < 4; i++)
	l->_spawn_colour[i] = col[i];

    l->_spawn_x = x;
    l->_spawn_y = y;
    l->_spawn_life = life;

    /* Current state. */
    l->life = life;
    l->r = 0.0;
    lux_update_colour(l);

    list_add(lux_manual_list, l);
    lux_count++;
    return l;
}


/* lux_automate:
 *
 * Move a lux from the manual list into the automated list.
 */
void lux_automate(lux_t *l)
{
    assert(l);

    list_remove(l);
    list_add(lux_automatic_list, l);
}


static void lux_free(lux_t *l)
{
    assert(l);

    list_remove(l);
    free(l);
    lux_count--;
}

/*--------------------------------------------------------------*/

void lux_start(void)
{
    list_init(lux_automatic_list);
    list_init(lux_manual_list);
    lux_count = 0;
}


void lux_stop(void)
{
    list_free(lux_automatic_list, lux_free);
    list_free(lux_manual_list, lux_free);
    assert(lux_count == 0);
}
