/* input.c,
 *
 * Handle input for the local player.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "angle.h"
#include "bullet.h"
#include "input.h"
#include "loft.h"
#include "map.h"
#include "network.h"
#include "player.h"
#include "score.h"
#include "server.h"
#include "start-loc.h"
#include "universal.h"
#include "weapon-pick.h"
#include "weapon.h"


static bool skip_ammo = true;

/*--------------------------------------------------------------*/

static void input_do_climb(player_t *p, const int sign);
static void input_do_jump(player_t *p);


static void input_process_up(player_t *p)
{
    switch (p->movement_mode) {
      case MOVEMENT_NORMAL:
      case MOVEMENT_ABOVE_LADDER:
	  input_do_jump(p);
	  break;
      case MOVEMENT_IN_LADDER:
	  input_do_climb(p, 1);
	  break;
      default:
	  assert(false);
    }
}


static void input_process_down(player_t *p)
{
    p->hold_keys |= INPUT_MOVE_DOWN;

    switch (p->movement_mode) {
      case MOVEMENT_NORMAL:
	  break;
      case MOVEMENT_IN_LADDER:
      case MOVEMENT_ABOVE_LADDER:
	  input_do_climb(p, -1);
	  break;
      default:
	  assert(false);
    }
}


static void input_do_climb(player_t *p, const int sign)
{
    p->vy += sign * 0.65;
}


static void input_do_jump(player_t *p)
{
    const double jump_vy[MAX_JUMP_FRAMES]= {
	0.03, 0.03, 0.03, 0.05, 0.05,
	0.10, 0.10, 0.15, 0.15, 0.20,
	0.20, 0.25, 0.25, 0.30, 0.30,
	0.00, 0.00, 0.05, 0.10, 0.15,
	0.20, 0.25, 0.30, 0.40, 0.50
    };

    if (!p->can_jump && !(p->hold_keys & INPUT_MOVE_UP))
	return;

    p->can_jump = false;

    p->jump_frame--;
    if (p->jump_frame <= 0) {
	p->jump_frame = 0;
	p->hold_keys &=~INPUT_MOVE_UP;
    }
    else
	p->hold_keys |= INPUT_MOVE_UP;

    assert(p->jump_frame < MAX_JUMP_FRAMES);
    p->vy += jump_vy[p->jump_frame] + GRAVITY;
}

/*--------------------------------------------------------------*/

static bool input_can_fire(const player_t *p);


static void input_do_fire(player_t *p, game_state_t *state)
{
    assert(state);

    if (state->as_server) {
	struct packet_bullet_new packet;
	static bullet_id id;
	double spray, rx, ry;
	int tick;
	assert(p->weapon < NUM_WEAPON_CLASSES);

	if (!input_can_fire(p))
	    return;

	if (weapon[p->weapon].spray == 0) {
	    spray = 0.0;
	}
	else {
	    int a;

	    a = rand() % (2*weapon[p->weapon].spray+1);
	    a -= weapon[p->weapon].spray+1;
	    spray = deg2rad(a/2000.0);

	    if (p->powerup_duration[POWERUP_LASER] > server_time)
		spray /= 2.0;
	}

	weapon_barrel_displacement(p->weapon, p->mirror, p->angle, &rx, &ry);

	/* To avoid bullets from being spawned in a wall, we spawn the
	 * bullet near the player's body, which we can safely assume
	 * is not in a wall, and use the time to step it from there to
	 * the barrel in 1 frame.
	 */
	tick = 1000*weapon[p->weapon].gun_x/weapon[p->weapon].speed;

	p->fire_delay = 1.0 / (weapon[p->weapon].firing_rate);

	packet.time = server_time-tick;
	packet.id = id;
	packet.class = p->weapon;
	packet.parent = p->id;
	packet.x = p->x+rx;
	packet.y = p->y+ry+19.0;
	packet.angle = p->angle + spray;

	if (p->powerup_duration[POWERUP_BERSERKER] > server_time) {
	    packet.modifier = 2.0;
	}
	else {
	    packet.modifier = 1.0;
	}

	if (packet.class == WEAPON_SHOTGUN)
	    id += 10;
	else
	    id++;

	if (p->weapon != WEAPON_BLASTER)
	    p->weapon_ammo[p->weapon]--;

	p->packet_size |= PACKET_INCLUDE_FIRING;

	bullet_new(&packet, state);
    }
}


static bool input_can_fire(const player_t *p)
{
    assert(p->weapon < NUM_WEAPON_CLASSES);

    if (p->fire_delay > 0.0)
	return false;

    if ((p->weapon != WEAPON_BLASTER) &&
	(p->weapon_ammo[p->weapon] == 0))
	return false;

    return true;
}

/*--------------------------------------------------------------*/

void input_process(player_t *p, const int impy, game_state_t *state)
{
    assert(p);
    assert(p->alive);
    assert(state);

    if (impy & INPUT_MOVE_LEFT)
	p->vx -= 0.75;
    if (impy & INPUT_MOVE_RIGHT)
	p->vx += 0.75;

    if (impy & INPUT_MOVE_UP)
	input_process_up(p);
    else
	p->hold_keys &=~INPUT_MOVE_UP;

    if (impy & INPUT_MOVE_DOWN)
	input_process_down(p);
    else
	p->hold_keys &=~INPUT_MOVE_DOWN;

    if (impy & INPUT_FIRE)
	input_do_fire(p, state);
}


void input_process_dead(player_t *p, const int impy)
{
    assert(p);
    assert(!p->alive);

    if (impy & INPUT_RESPAWN) {
	struct packet_player_respawn packet;
	NLbyte buf[MAX_PACKET_SIZE];
	NLint len;

	packet.id = p->id;
	packet.loc = start_loc_rand();

	len = packet_player_respawn_encode(&packet, buf, sizeof(buf));
	server_broadcast(buf, len);

	player_new_life(p);
	start_loc_get(packet.loc, &p->x, &p->y);
    }
}

/*--------------------------------------------------------------*/

static void input_select_weapon(int mz, enum WEAPON_CLASS *select_weapon);
static void input_do_aim(const double dx, const double dy);
static void input_do_aim2(const double dx, const double dy);
static bool try_fire(enum WEAPON_CLASS class, const double angle,
		     const double tick);


/* input_get_impies:
 *
 * Returns the input key and mouse presses, AKA input Impies.
 */
int input_get_impies(const int mx, const int my, const int mz,
		     enum WEAPON_CLASS *select_weapon)
{
    int impy = 0;
    assert(player);

    impy = input_get_move_impies(mx, my);

    if (key[KEY_SPACE] && !(player->alive))
	impy |= INPUT_RESPAWN;

    *select_weapon = weapon_pick(player);

    if (mouse_b & 1) {
	if ((player->weapon != WEAPON_BLASTER) &&
	    (player->weapon_ammo[player->weapon] == 0)) {
	    *select_weapon = weapon_pick_no_ammo(player);
	}

	impy |= INPUT_FIRE;
    }

    input_select_weapon(mz, select_weapon);

    if (key[KEY_TAB])
	show_score = true;
    else
	show_score = false;

    return impy;
}


int input_get_move_impies(const int mx, const int my)
{
    int impy = 0;
    assert(player);

    player->mirror = (mx < player->x);
    input_do_aim(mx-player->x, my-(player->y+19.0));

    if (key[KEY_A])
	impy |= INPUT_MOVE_LEFT;
    if (key[KEY_D])
	impy |= INPUT_MOVE_RIGHT;
    if (key[KEY_W])
	impy |= INPUT_MOVE_UP;
    if (key[KEY_S])
	impy |= INPUT_MOVE_DOWN;

    return impy;
}


static void input_select_weapon(int mz, enum WEAPON_CLASS *select_weapon)
{
    if (mz > 0) {
	enum WEAPON_CLASS sel, sel2;

	sel = player->weapon+1;
	sel2 = player->weapon;
	for (; (sel < NUM_WEAPON_CLASSES) && (mz > 0); sel++) {
	    if (player->have_weapon[sel]) {
		if ((!skip_ammo) ||
		    (sel == WEAPON_BLASTER) ||
		    (player->weapon_ammo[sel] > 0)) {
		    sel2 = sel;
		    mz--;
		}
	    }
	}

	*select_weapon = sel2;
    }
    else if (mz < 0) {
	enum WEAPON_CLASS sel, sel2;

	sel = sel2 = player->weapon;
	do {
	    if (sel == WEAPON_FIRST)
		break;

	    sel--;
	    if (player->have_weapon[sel]) {
		if ((!skip_ammo) ||
		    (sel == WEAPON_BLASTER) ||
		    (player->weapon_ammo[sel] > 0)) {
		    sel2 = sel;
		    mz++;
		}
	    }
	} while (mz < 0);

	*select_weapon = sel2;
    }
    else if (mz == 0) {
	int ks[NUM_WEAPON_CLASSES] = {
	    KEY_1, KEY_2, KEY_3, KEY_4, KEY_5, KEY_6, KEY_7
	};
	enum WEAPON_CLASS sel;

	for (sel = NUM_WEAPON_CLASSES; sel > WEAPON_FIRST; sel--) {
	    if (key[ks[sel-1]] && player->have_weapon[sel-1]) {
		*select_weapon = sel-1;
		break;
	    }
	}
    }

    weapon_pick_reset();
}


static void input_do_aim(const double dx, const double dy)
{
    if (player->weapon != WEAPON_GRENADE_LAUNCHER) {
	double rx, ry;
	double theta = atan2(dy, dx);

	weapon_barrel_displacement(player->weapon,
				   player->mirror, theta, &rx, &ry);

	player->angle = atan2(dy-ry, dx-rx);
    }
    else {
	input_do_aim2(dx, dy);
    }
}


static void input_do_aim2(const double dx, const double dy)
{
    loft_parameters_t loft;
    double t_direct, t_indirect;
    double a_direct, a_indirect;

    loft.dx = dx;
    loft.dy = dy;
    loft.v = weapon[player->weapon].speed;
    loft.ay = -GRAVITY2;

    if (!loft_find_times(&loft, &t_direct, &t_indirect)) {
	double angle = atan2(dy, dx);

	if (angle < -M_PI_2) {
	    /* Special case: 3rd quadrant. */
	    angle = 3*M_PI_2 - (2*M_PI+angle);
	}
	else {
	    angle = M_PI_2 - angle;
	}

	player->angle = M_PI_2 - angle/2.0;

	return;
    }

    a_direct = loft_find_angle(&loft, t_direct);
    if (x_in_map(player->x+dx) && y_in_map(player->y+19.0+dy)) {
	if (map_occupies(player->x+dx, player->y+19.0+dy)) {
	    player->angle = a_direct;
	    return;
	}
    }

    if (try_fire(player->weapon, a_direct, t_direct)) {
	player->angle = a_direct;
	return;
    }

    a_indirect = loft_find_angle(&loft, t_indirect);
    if (try_fire(player->weapon, a_indirect, t_indirect)) {
	player->angle = a_indirect;
	return;
    }

    player->angle = a_direct;
}


static bool try_fire(enum WEAPON_CLASS class, const double angle,
		     const double tick)
{
    bullet_t b;

    b.class = class;
    b._spawn_time = 0;
    b._spawn_x = player->x;
    b._spawn_y = player->y + 19.0;
    b._spawn_angle = angle;
    b._spawn_vx = weapon[class].speed * cos(angle);
    b._spawn_vy = weapon[class].speed * sin(angle);

    b.time = b._spawn_time;
    b.x = b._spawn_vx;
    b.y = b._spawn_vy;
    b.grenade_life = 100;

    return (bullet_update_unit(&b, tick * 1000.0, NULL));
}
