/* chat.c,
 *
 * Chat history.
 */

#include <allegro.h>
#include <assert.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "chat.h"
#include "network.h"


static int input_pos;
bool input_enabled;
char input_line[MAX_CHAT_LEN];

static int chat_colour[NUM_CHAT_CLASSES];
static chat_t message[NUM_CHAT_HISTORY];
static unsigned int num_messages;

bool chat_new_messages;

/*--------------------------------------------------------------*/

int chat_colour_from_class(const enum CHAT_MESSAGE_CLASS c)
{
    if (c >= NUM_CHAT_CLASSES) {
	return makecol(0xff, 0x00, 0xff);
    }
    else {
	return chat_colour[c];
    }
}

/*--------------------------------------------------------------*/

static void chat_input_unit(const int c, const int sc);


const char *chat_input(const int c, const int sc)
{
    if (sc == KEY_ENTER) {
	if (input_enabled) {
	    input_enabled = false;
	    return input_line;
	}
	else {
	    input_line[0] = '\0';
	    input_pos = 0;
	    input_enabled = true;
	}
    }

    if (!input_enabled)
	return NULL;

    chat_input_unit(c, sc);
    return NULL;
}


static void chat_input_unit(const int c, const int sc)
{
    switch (sc) {
      case KEY_BACKSPACE:
	  if (input_pos > 0)
	      input_line[--input_pos] = '\0';
	  break;

      case KEY_INSERT:
      case KEY_DEL:
      case KEY_HOME:
      case KEY_END:
      case KEY_PGUP:
      case KEY_PGDN:
      case KEY_LEFT:
      case KEY_RIGHT:
      case KEY_UP:
      case KEY_DOWN:
	  break;

      default:
	  if ((c >= ' ') && (input_pos < MAX_CHAT_LEN-1)) {
	      input_line[input_pos++] = c;
	      input_line[input_pos] = '\0';
	  }
	  break;
    }
}

/*--------------------------------------------------------------*/

static void chat_push(void);


void chat_scroll(const int tick)
{
    while ((message[0].str) && (message[0].time < tick)) {
	chat_push();
    }
}


static void chat_push(void)
{
    unsigned int i;

    free(message[0].str);

    for (i = 0; i < NUM_CHAT_HISTORY-1; i++)
	message[i] = message[i+1];

    message[NUM_CHAT_HISTORY-1].time = 0;
    message[NUM_CHAT_HISTORY-1].str = NULL;
    num_messages--;
}

/*--------------------------------------------------------------*/

void chat_add(const int tick, const enum CHAT_MESSAGE_CLASS class,
	      const char *format, ...)
{
    char cat[1024];
    va_list ap;

    va_start(ap, format);
    vsnprintf(cat, sizeof(cat), format, ap);
    va_end(ap);

    if (num_messages >= NUM_CHAT_HISTORY)
	chat_push();

    message[num_messages].time = tick+10000;
    if (class < NUM_CHAT_CLASSES)
	message[num_messages].class = class;
    else
	message[num_messages].class = CHAT_CLIENT_ERROR;
    message[num_messages].str = strdup(cat);
    num_messages++;

    chat_new_messages = true;
}


const chat_t *chat_get(const unsigned int i)
{
    assert(i < NUM_CHAT_HISTORY);

    if (message[i].str == NULL)
	return NULL;
    else
	return &message[i];
}


unsigned int chat_num_lines(void)
{
    return num_messages;
}

/*--------------------------------------------------------------*/

void chat_init(void)
{
    unsigned int i;

    for (i = 0; i < NUM_CHAT_HISTORY; i++) {
	message[i].time = 0;
	message[i].str = NULL;
    }

    num_messages = 0;
}


void chat_init_colours(void)
{
    chat_colour[CHAT_SERVER_ERROR]   = makecol(0xff, 0x00, 0x00);
    chat_colour[CHAT_FRONTEND_ERROR] = makecol(0xff, 0x40, 0x00);
    chat_colour[CHAT_CLIENT_ERROR]   = makecol(0xff, 0x80, 0x00);
    chat_colour[CHAT_SERVER_LOG]     = makecol(0x80, 0x80, 0x80);
    chat_colour[CHAT_FRONTEND_LOG]   = makecol(0xa0, 0xa0, 0xa0);
    chat_colour[CHAT_GAME_MESSAGE]   = makecol(0xff, 0xff, 0xff);
    chat_colour[CHAT_BABBLE]	     = makecol(0x40, 0x40, 0xff);
}


void chat_shutdown(void)
{
    unsigned int i;

    for (i = 0; i < NUM_CHAT_HISTORY; i++) {
	if (message[i].str) {
	    free(message[i].str);
	    message[i].str = NULL;
	}
    }
}
