/* texdraw.c,
 *
 * Routines to help with drawing out sprites.
 */

#include <assert.h>
#include "texdraw.h"


/*--------------------------------------------------------------*/

static void texdraw_sprite_2d(const texcoord2d_t *texcoord,
			      const GLdouble X0, const GLdouble Y0,
			      const GLdouble X1, const GLdouble Y1)
{
#define DIVS   4
 
    int i, j;
    GLdouble texx[DIVS+1], texy[DIVS+1];
    GLdouble verx[DIVS+1], very[DIVS+1];
 
    for (i = 0; i <= DIVS; i++) {
       texx[i] = texcoord->left + (texcoord->right-texcoord->left)   * i/DIVS;
       texy[i] = texcoord->bottom + (texcoord->top-texcoord->bottom) * i/DIVS;
       verx[i] = X0 + (X1-X0) * i/DIVS;
       very[i] = Y0 + (Y1-Y0) * i/DIVS;
    }
 
    for (j = 0; j < DIVS; j++) {
       for (i = 0; i < DIVS; i++) {
           glTexCoord2d(texx[i+0], texy[j+0]);
           glVertex2d  (verx[i+0], very[j+0]);
 
           glTexCoord2d(texx[i+1], texy[j+0]);
           glVertex2d  (verx[i+1], very[j+0]);
       
           glTexCoord2d(texx[i+1], texy[j+1]);
           glVertex2d  (verx[i+1], very[j+1]);

           glTexCoord2d(texx[i+0], texy[j+1]);
           glVertex2d  (verx[i+0], very[j+1]);
       }
    }
 
#undef DIVS

#if 0
    glTexCoord2d(texcoord->left, texcoord->bottom);
    glVertex2d(X0, Y0);

    glTexCoord2d(texcoord->right, texcoord->bottom);
    glVertex2d(X1, Y0);

    glTexCoord2d(texcoord->right, texcoord->top);
    glVertex2d(X1, Y1);

    glTexCoord2d(texcoord->left, texcoord->top);
    glVertex2d(X0, Y1);
#endif
}


static void texdraw_sprite_hflip_2d(const texcoord2d_t *texcoord,
				    const GLdouble X0, const GLdouble Y0,
				    const GLdouble X1, const GLdouble Y1)
{
    glTexCoord2d(texcoord->right, texcoord->bottom);
    glVertex2d(X0, Y0);

    glTexCoord2d(texcoord->left, texcoord->bottom);
    glVertex2d(X1, Y0);

    glTexCoord2d(texcoord->left, texcoord->top);
    glVertex2d(X1, Y1);

    glTexCoord2d(texcoord->right, texcoord->top);
    glVertex2d(X0, Y1);
}

/*--------------------------------------------------------------*/

/* gl_draw_sprite_2d:
 *
 * Draw the pre-binded texture to the screen, using the given texture
 * coordinates.  Positive y is upwards.
 *
 * This should be called between a glBegin(GL_QUADS) and glEnd block.
 */
void gl_draw_sprite_2d(const texcoord2d_t *texcoord,
		       const GLdouble X0, const GLdouble Y0)
{
    assert(texcoord);

    texdraw_sprite_2d(texcoord, X0, Y0, (X0+texcoord->w), (Y0+texcoord->h));
}


/* gl_draw_sprite_hflip_2d:
 *
 * Just like 'gl_draw_sprite_2d', but the texture is flipped
 * horizontally.
 *
 * This should be called between a glBegin(GL_QUADS) and glEnd block.
 */
void gl_draw_sprite_hflip_2d(const texcoord2d_t *texcoord,
			     const GLdouble X0, const GLdouble Y0)
{
    assert(texcoord);

    texdraw_sprite_hflip_2d(texcoord, X0, Y0,
			    (X0+texcoord->w), (Y0+texcoord->h));
}

/*--------------------------------------------------------------*/

/* gl_draw_sprite_2dx:
 *
 * Similar to 'gl_draw_sprite_2d', but the texture is double the size.
 *
 * This should be called between a glBegin(GL_QUADS) and glEnd block.
 */
void gl_draw_sprite_2dx(const texcoord2d_t *texcoord,
			const GLdouble X0, const GLdouble Y0)
{
    assert(texcoord);

    texdraw_sprite_2d(texcoord, (X0/2.0), (Y0/2.0),
		      (X0+texcoord->w)/2.0, (Y0+texcoord->h)/2.0);
}


/* gl_draw_sprite_hflip_2dx:
 *
 * Similar to 'gl_draw_sprite_2dx', but flip the texture.
 *
 * This should be called between a glBegin(GL_QUADS) and glEnd block.
 */
void gl_draw_sprite_hflip_2dx(const texcoord2d_t *texcoord,
			      const GLdouble X0, const GLdouble Y0)
{
    assert(texcoord);

    texdraw_sprite_hflip_2d(texcoord, (X0/2.0), (Y0/2.0),
			    (X0+texcoord->w)/2.0, (Y0+texcoord->h)/2.0);
}
