/* sv-internal.c,
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "network.h"
#include "strlcpy.h"
#include "sv-internal.h"


client_data_t client_data[MAX_CLIENTS];	/* Shared */
NLint group_all_clients;

/*--------------------------------------------------------------*/

#define server_print_error()	network_print_error("Server", __LINE__)

/*--------------------------------------------------------------*/

const char *client_type_str(const enum CLIENT_TYPE type)
{
    const char *str[NUM_CLIENT_TYPES] = {
	"unused", "connecting", "robot", "gamer", "admin"
    };

    if (type >= NUM_CLIENT_TYPES) {
	return "Client type unknown";
    }
    else {
	return str[type];
    }
}

/*--------------------------------------------------------------*/

static int cmp_client_score(const void *a, const void *b)
{
    const client_data_t *aa = a;
    const client_data_t *bb = b;
    assert(aa);
    assert(bb);

    if ((aa->clid != 0) && (bb->clid == 0))
	return -1;

    if ((aa->clid == 0) && (bb->clid != 0))
	return 1;

    return (bb->sc.total_score - aa->sc.total_score);
}


void sv_intern_sort_by_score(void)
{
    qsort(client_data, MAX_CLIENTS, sizeof(client_data_t), cmp_client_score);
}

/*--------------------------------------------------------------*/

int sv_intern_client_from_sock(const NLsocket sock)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].sock == sock)
	    return i;
    }

    return -1;
}


int sv_intern_client_from_name(const char *name)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (strcmp(client_data[i].name, name) == 0)
	    return i;
    }

    return -1;
}


int sv_intern_client_from_client_id(const client_id id)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].clid == id)
	    return i;
    }

    return -1;
}


int sv_intern_client_from_id(const player_id id)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].id == id)
	    return i;
    }

    return -1;
}


int sv_intern_unused_client(void)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (client_data[i].type == CLIENT_UNUSED)
	    return i;
    }

    return -1;
}

/*--------------------------------------------------------------*/

bool sv_intern_set_admin(client_data_t *data)
{
    assert(data);

    if (data->type != CLIENT_GAMER)
	return false;

    data->type = CLIENT_ADMIN;
    return true;
}


bool sv_intern_set_gamer(client_data_t *data)
{
    assert(data);

    if (data->type != CLIENT_ADMIN)
	return false;

    data->type = CLIENT_GAMER;
    return true;
}

/*--------------------------------------------------------------*/

int sv_intern_open(const NLsocket sock)
{
    int idx = sv_intern_unused_client();

    if (idx < 0) {
	return -1;
    }

    if (nlGroupAddSocket(group_all_clients, sock) != NL_TRUE) {
	server_print_error();
	return -1;
    }

    client_data[idx].sock = sock;
    strlcpy(client_data[idx].name, "Anonymous", sizeof(client_data[idx].name));
    client_data[idx].type = CLIENT_CONNECTING;

    return idx;
}


void sv_intern_close(const NLsocket sock)
{
    int i = sv_intern_client_from_sock(sock);

    if (i < 0) {
	assert(0);
	return;
    }

    nlGroupDeleteSocket(group_all_clients, sock);
    nlClose(sock);
    client_data[i].sock = NL_INVALID;
    client_data[i].type = CLIENT_UNUSED;
}

/*--------------------------------------------------------------*/

void sv_intern_init(void)
{
    unsigned int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	client_data[i].clid = 0;
	client_data[i].sock = NL_INVALID;
	client_data[i].name[0] = '?';
	client_data[i].name[1] = '\0';
	client_data[i].type = CLIENT_UNUSED;
    }
}
