/* step.c,
 *
 * Functions to walk long distances, while performing collision
 * detection after each step.
 */

#include <assert.h>
#include "maxmin.h"
#include "step.h"


#define ABS(x)		((x) >= 0 ? (x) : -(x))

/* velocity_to_steps:
 *
 * Convert a velocity, (vx, vy), into small stepping displacements.
 * Returns the number of steps of (dx, dy) to make.
 *
 * Note: steps*(dx, dy) != (vx, vy) since steps is an integer.
 */
int velocity_to_steps(const double vx, const double vy,
		      double *dx, double *dy)
{
    double avx = ABS(vx);
    double avy = ABS(vy);
    int steps;

    if (avx > avy) {
	steps = avx;
	if (steps <= 0)
	    return 0;

	*dx = vx / avx;
	*dy = vy / avx;
    }
    else {
	steps = avy;
	if (steps <= 0)
	    return 0;

	*dx = vx / avy;
	*dy = vy / avy;
    }

    return steps;
}


/* walk:
 *
 * Try to walk (vx, vy) pixels, using func->step to walk and detect
 * collisions.  Returns true if no collision took place.
 */
bool walk(void *p, void *work, double vx, const double vy,
	  const walk_callbacks_t *func)
{
    bool virgin = true;
    bool collided = false;
    double dx, dy;
    int steps;

    func->copy(p, work);
    steps = velocity_to_steps(vx, vy, &dx, &dy);

    while (steps > 0) {
	int result;

	result = func->step(work, dx, dy);
	if (result != STEP_NO_COLLIDE) {
	    collided = true;

	    if (result & STEP_COLLIDE_X) {
		dx = 0.0;

		/* Don't count the first step as a collision.  This
		   usually occurs when we are holding left/right while
		   next to a wall.. */
		if (virgin) {
		    collided = false;
		    vx = 0.0;
		}
	    }
	    if (result & STEP_COLLIDE_Y)
		dy = 0.0;

	    if ((dx == 0.0) && (dy == 0.0))
		break;

	}

	virgin = false;
	steps--;
    }

    if (!collided) {
	/* If we haven't hit a wall, try to go directly to our
	 * destination.  This fixes errors caused by needing to take a
	 * step shorter than 1.0*(dx, dy).
	 */
	if (func->step(p, vx, vy) != STEP_NO_COLLIDE)
	    collided = true;
    }

    if (collided)
	func->copy(work, p);

    return !collided;
}

/*--------------------------------------------------------------*/

/* velocity_to_time_increments:
 *
 * Find the time (in milliseconds) required for a projectile moving
 * with velocity (vx, vy) to move 1 pixel horizontally or vertially.
 */
static int velocity_to_time_increments(const double vx, const double vy)
{
    if ((vx*vx) + (vy*vy) <= (5.0*5.0))
	return 1;
    else
	return maxi(1, 1000.0/maxd(ABS(vx), ABS(vy)) - 1);
}


/* walk_time:
 *
 * Try to walk a particle from time to end_time using func->step to
 * move and detect collisions.  Returns whether or not we were able to
 * step up to tick = end_time.
 *
 * Data is some any extra data to be passed to the stepping function.
 */
bool walk_time(void *p, const walk_time_callbacks_t *func,
	       int tick, const int end_time, const void *data)
{
    double vx, vy;
    int collisions;
    int delta_time;
    assert(p);
    assert(func);

    func->velocity(p, &vx, &vy);
    delta_time = velocity_to_time_increments(vx, vy);

    while (tick <= end_time) {
	collisions = func->step(p, tick, data);

	if (collisions != STEP_NO_COLLIDE) {
	    if (func->collided(p, tick-delta_time, collisions))
		return false;
	}

	if (tick == end_time) {
	    break;
	}

	tick = mini(tick+delta_time, end_time);
    }

    return true;
}


/* walk_time_crude:
 *
 * Similar to walk_time, but take larger steps at a time, then
 * backtrack if we hit a wall.
 */
bool walk_time_crude(void *p, const walk_time_callbacks_t *func,
		     int tick, const int end_time, const void *data)
{
    bool collided = false;
    double vx, vy;
    int collisions, col;
    int delta_time;
    int step_size;
    assert(p);
    assert(func);

    func->velocity(p, &vx, &vy);
    delta_time = velocity_to_time_increments(vx, vy);
    step_size = 5;

    while (tick <= end_time) {
	collisions = func->step(p, tick, data);

	if (collisions != STEP_NO_COLLIDE) {
	    collided = true;

	    do {
		step_size--;
		tick -= delta_time;
		col = func->step(p, tick, data);
		if (col == STEP_NO_COLLIDE)
		    break;
		collisions = col;
	    } while (step_size > 0);

	    func->collided(p, tick, collisions);
	    break;
	}
	else {
	    if (tick == end_time)
		break;

	    tick = mini(tick+step_size*delta_time, end_time);
	}
    }

    return !collided;
}
