/* sound.c,
 *
 * Sound stuff.
 */

#include <allegro.h>
#include <assert.h>
#include <stdio.h>

#include "camera.h"
#include "server.h"
#include "sound.h"


#define SOURCE_FIRE_LOCAL_FIRST		0
#define SOURCE_FIRE_LOCAL_LAST		(SOURCE_FIRE_LOCAL_FIRST+1)
#define SOURCE_FIRE_REMOTE_FIRST	2
#define SOURCE_FIRE_REMOTE_LAST		(SOURCE_FIRE_REMOTE_FIRST+4-1)
#define SOURCE_MISSILE_FIRST	6
#define SOURCE_MISSILE_LAST	(SOURCE_MISSILE_FIRST+4-1)
#define SOURCE_EXPLOSION_FIRST	10
#define SOURCE_EXPLOSION_LAST	(SOURCE_EXPLOSION_FIRST+4-1)
#define MAX_SOUND_SOURCES	14


typedef struct {
    /* Spawn state */
    float _spawn_x, _spawn_y;
    float _spawn_vx, _spawn_vy;
    int _spawn_time, _next_spawn_time;
    int _base_vol;

    player_id parent;		/* 0 if not spawned by player */

    /* Current state */
    const SAMPLE *sample;
    int voice;
} sound_source_t;


typedef struct {
    const char *fn;
    const int vol;
    SAMPLE *sample;
} sound_t;


static sound_source_t sound_source[MAX_SOUND_SOURCES];
static bool sound_enabled = true;
static unsigned int sound_volume = 128;
bool reverse_stereo = false;

/*--------------------------------------------------------------*/

enum TRY_SOUND_SOURCE_RET {
    SOUND_SOURCE_ERROR_ALLOCATING_VOICE,
    SOUND_SOURCE_IN_USE_BY_SAME_SAMPLE,
    SOUND_SOURCE_IN_USE_BY_OTHER_SAMPLE,
    SOUND_SOURCE_RESTART,
    SOUND_SOURCE_FREE
};


/* sound_volume_from_dist:
 *
 * Given a base volume and distance squared, returns the volume to
 * play with AFTER taking 'sound_volume' into account.
 */
static int sound_volume_from_dist(const int base, const double dist_sq)
{
    int vol;

    vol = base * (1.0 - dist_sq/(750.0*750.0));
    vol = vol * (double)(sound_volume/255.0);
    return MID(0, vol, 255);
}


/* sound_pan_from_dist:
 *
 * Given a horizontal distance, return the pan amount, taking into
 * account 'reverse_stereo'.
 */
static int sound_pan_from_dist(const double dx)
{
    int pan;

    pan = MID(0, 127.5 + dx*127.5/480, 255);
    return (reverse_stereo ? (255 - pan) : pan);
}


/* try_sound_source:
 *
 * Check if we can play sample on the nth sound source.  If
 * successful, the sample is allocated a voice.
 */
static enum TRY_SOUND_SOURCE_RET
try_sound_source(const int n, const SAMPLE *sample)
{
    sound_source_t *src;
    SAMPLE *chk;
    assert(n < MAX_SOUND_SOURCES);

    src = &sound_source[n];
    if (src->voice >= 0) {
	chk = voice_check(src->voice);

	if (src->_next_spawn_time > server_time) {
	    if (chk == sample)
		return SOUND_SOURCE_IN_USE_BY_SAME_SAMPLE;
	    else
		return SOUND_SOURCE_IN_USE_BY_OTHER_SAMPLE;
	}

	/* Current source is playing the same sample. */
	if ((chk == src->sample) && (chk == sample)) {
	    src->sample = sample;
	    return SOUND_SOURCE_RESTART;
	}

	/* Current source has stopped playing. */
	if (voice_get_position(src->voice) < 0) {
	    /* XXX?  Does reallocate_voice have a bug? */
	    release_voice(src->voice);
	    src->voice = allocate_voice(sample);

	    if (src->voice < 0) {
		return SOUND_SOURCE_ERROR_ALLOCATING_VOICE;
	    }
	    else {
		src->sample = sample;
		return SOUND_SOURCE_FREE;
	    }
	}

	return SOUND_SOURCE_IN_USE_BY_OTHER_SAMPLE;

    }
    else {
	/* Current source is not used. */
	src->voice = allocate_voice(sample);
	if (src->voice < 0) {
	    return SOUND_SOURCE_ERROR_ALLOCATING_VOICE;
	}
	else {
	    src->sample = sample;
	    return SOUND_SOURCE_FREE;
	}
    }
}


static void log_load_sound(const char *fn, const bool loaded)
{
    if (loaded)
	fprintf(stdout, "[Sound] Loaded '%s'\n", fn);
    else
	fprintf(stderr, "[Sound] Error loading '%s'\n", fn);
}


static void load_sound(sound_t *sfx)
{
    if (sfx->fn) {
	sfx->sample = load_sample(sfx->fn);
	log_load_sound(sfx->fn, sfx->sample);
    }
    else {
	sfx->sample = NULL;
    }
}

/*--------------------------------------------------------------*/

#define NUM_EXPLOSION_SOUNDS		3

static sound_t explosion_sound[NUM_EXPLOSION_SOUNDS] = {
    { SFX_DIR "explosion_short_A1.wav", 128, NULL },
    { SFX_DIR "explosion_short_B1.wav", 128, NULL },
    { SFX_DIR "explosion_short_C1.wav", 128, NULL },
};


void play_explosion(const float x, const float y)
{
    int i = rand() % NUM_EXPLOSION_SOUNDS;
    sound_t *sfx = &explosion_sound[i];
    sound_source_t *src;
    float dx = x - (camera.x+640/2);
    float dy = y - (camera.y+480/2);
    int n, pan, vol;

    if (!sfx->sample)
	return;

    vol = sound_volume_from_dist(sfx->vol, (dx*dx) + (dy*dy));
    if (vol <= 0)
	return;

    pan = sound_pan_from_dist(dx);

    for (n = SOURCE_EXPLOSION_FIRST; n <= SOURCE_EXPLOSION_LAST; n++) {
	enum TRY_SOUND_SOURCE_RET ret;

	ret = try_sound_source(n, sfx->sample);
	if (ret == SOUND_SOURCE_ERROR_ALLOCATING_VOICE) {
	    break;
	}

	if (ret == SOUND_SOURCE_RESTART ||
	    ret == SOUND_SOURCE_FREE) {
	    src = &sound_source[n];
	    voice_set_position(src->voice, 0);
	    voice_set_volume(src->voice, vol);
	    voice_set_pan(src->voice, pan);
	    voice_set_priority(src->voice, 255);
	    voice_start(src->voice);

	    src->_spawn_x = x;
	    src->_spawn_y = y;
	    src->_spawn_vx = 0;
	    src->_spawn_vy = 0;
	    src->_spawn_time = server_time;
	    src->_next_spawn_time = src->_spawn_time + 1000;
	    src->_base_vol = sfx->vol;
	    src->parent = 0;
	    break;
	}
    }
}


void explosion_sound_load(void)
{
    int i;

    if (!sound_enabled)
	return;

    for (i = 0; i < NUM_EXPLOSION_SOUNDS; i++) {
	load_sound(&explosion_sound[i]);
    }
}


void explosion_sound_free(void)
{
    int i;

    for (i = 0; i < NUM_EXPLOSION_SOUNDS; i++) {
	if (explosion_sound[i].sample) {
	    destroy_sample(explosion_sound[i].sample);
	    explosion_sound[i].sample = NULL;
	}
    }
}

/*--------------------------------------------------------------*/

#define NUM_PLAYER_HURT_SOUNDS		4

static sound_t player_hurt_sound[NUM_PLAYER_HURT_SOUNDS] = {
    { SFX_DIR "bullethit_character_A.wav", 64, NULL },
    { SFX_DIR "bullethit_character_B.wav", 64, NULL },
    { SFX_DIR "bullethit_character_C.wav", 64, NULL },
    { SFX_DIR "bullethit_character_D.wav", 64, NULL }
};


void play_player_hurt(const double dist_sq, const double dx)
{
    int i = rand() % NUM_PLAYER_HURT_SOUNDS;
    sound_t *sfx = &player_hurt_sound[i];
    int vol, pan;

    vol = sound_volume_from_dist(sfx->vol, dist_sq);
    if (vol <= 0)
	return;

    pan = sound_pan_from_dist(dx);

    if (sfx->sample) {
	play_sample(sfx->sample, vol, pan, 1000, 0);
    }
}


void player_sound_load(void)
{
    int i;

    if (!sound_enabled)
	return;

    for (i = 0; i < NUM_PLAYER_HURT_SOUNDS; i++) {
	load_sound(&player_hurt_sound[i]);
    }
}


void player_sound_free(void)
{
    int i;

    for (i = 0; i < NUM_PLAYER_HURT_SOUNDS; i++) {
	if (player_hurt_sound[i].sample) {
	    destroy_sample(player_hurt_sound[i].sample);
	    player_hurt_sound[i].sample = NULL;
	    continue;
	}
    }
}

/*--------------------------------------------------------------*/

typedef struct {
    const char *shoot_fn;
    const int shoot_vol;

    /* msec to wait before we can restart the voice. */
    const int shoot_delay;

    SAMPLE *shoot_sample;
} weapon_sound_t;


static weapon_sound_t weapon_sound[NUM_WEAPON_CLASSES] = {
    /* Missile launcher */
    { SFX_DIR "rocket3.wav", 255, 0, NULL },

    /* Micro-missile launcher */
    { SFX_DIR "Shoot_Striker1.wav", 128, 110, NULL },

    /* Grenade launcher */
    { SFX_DIR "Shoot_Throwable.wav", 128, 200, NULL },

    /* Blaster */
    { SFX_DIR "blaster.wav", 96, 125, NULL },

    /* Shotgun */
    { SFX_DIR "Shoot_PumpShotgun.wav", 96, 350, NULL },

    /* Vulcan */
    { SFX_DIR "Shoot_MP5.wav", 64, 110, NULL },
    
    /* Minigun */
    { SFX_DIR "Shoot_IngramDual.wav", 128, 120, NULL }
};


static sound_t missile_sound = {
    SFX_DIR "smoke_grenade_loop1.wav", 192, NULL
};


void play_weapon_fire(const enum WEAPON_CLASS w, const player_id id,
		      const float x, const float y)
{
    weapon_sound_t *sfx;
    sound_source_t *src;
    int source_first, source_last;
    int n, vol, pan;
    float dx = x - (camera.x+640/2);
    float dy = y - (camera.y+480/2);
    assert(w < NUM_WEAPON_CLASSES);

    sfx = &weapon_sound[w];
    if (!sfx->shoot_sample)
	return;

    vol = sound_volume_from_dist(sfx->shoot_vol, (dx*dx) + (dy*dy));
    if (vol <= 0)
	return;

    if (player && (id == player->id)) {
	source_first = SOURCE_FIRE_LOCAL_FIRST;
	source_last = SOURCE_FIRE_LOCAL_LAST;
    }
    else {
	source_first = SOURCE_FIRE_REMOTE_FIRST;
	source_last = SOURCE_FIRE_REMOTE_LAST;
    }

    for (n = source_first; n <= source_last; n++) {
	enum TRY_SOUND_SOURCE_RET ret;

	ret = try_sound_source(n, sfx->shoot_sample);

	if ((ret == SOUND_SOURCE_ERROR_ALLOCATING_VOICE) ||
	    (ret == SOUND_SOURCE_IN_USE_BY_SAME_SAMPLE &&
	     id == sound_source[n].parent)) {
	    break;
	}

	if (ret == SOUND_SOURCE_RESTART ||
	    ret == SOUND_SOURCE_FREE) {
	    src = &sound_source[n];
	    pan = sound_pan_from_dist(dx);

	    voice_set_position(src->voice, 0);
	    voice_set_volume(src->voice, vol);
	    voice_set_pan(src->voice, pan);
	    voice_set_priority(src->voice, 255);
	    voice_start(src->voice);

	    src->_spawn_x = x;
	    src->_spawn_y = y;
	    src->_spawn_vx = 0;
	    src->_spawn_vy = 0;
	    src->_spawn_time = server_time;
	    src->_next_spawn_time = src->_spawn_time + sfx->shoot_delay;
	    src->_base_vol = sfx->shoot_vol;
	    src->parent = id;
	    break;
	}
    }
}


int play_missile_flight(const float x, const float y,
			const float vx, const float vy)
{
    enum TRY_SOUND_SOURCE_RET ret;
    sound_source_t *src;
    int vol, pan;
    float dx = x - (camera.x+640/2);
    float dy = y - (camera.y+480/2);
    int n;

    if (!missile_sound.sample)
	return -1;

    for (n = SOURCE_MISSILE_FIRST; n <= SOURCE_MISSILE_LAST; n++) {
	ret = try_sound_source(n, missile_sound.sample);

	if (ret == SOUND_SOURCE_ERROR_ALLOCATING_VOICE)
	    return -1;

	if ((ret == SOUND_SOURCE_RESTART) ||
	    (ret == SOUND_SOURCE_FREE)) {
	    src = &sound_source[n];
	    vol = sound_volume_from_dist(missile_sound.vol, (dx*dx) + (dy*dy));
	    pan = sound_pan_from_dist(dx);

	    voice_set_position(src->voice, 0);
	    voice_set_volume(src->voice, vol);
	    voice_set_pan(src->voice, pan);
	    voice_set_playmode(src->voice, PLAYMODE_LOOP);
	    voice_set_priority(src->voice, 255);
	    voice_start(src->voice);

	    src->_spawn_x = x;
	    src->_spawn_y = y;
	    src->_spawn_vx = vx;
	    src->_spawn_vy = vy;
	    src->_spawn_time = server_time;
	    src->_next_spawn_time = src->_spawn_time + 1000;
	    src->_base_vol = missile_sound.vol;
	    src->parent = 0;
	    return n;
	}
    }

    return -1;
}


void stop_missile_flight(const int n)
{
    sound_source_t *src;
    assert(n < MAX_SOUND_SOURCES);

    src = &sound_source[n];
    if ((src->voice >= 0) &&
	(voice_check(src->voice) == missile_sound.sample)) {
	voice_ramp_volume(src->voice, 500 * sound_volume/255, 0);
	voice_set_playmode(src->voice, PLAYMODE_FORWARD);
	release_voice(src->voice);
	src->sample = NULL;
	src->voice = -1;
    }
}


void weapon_sound_load(void)
{
    enum WEAPON_CLASS w;

    if (!sound_enabled)
	return;

    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	if (!weapon_sound[w].shoot_fn) {
	    weapon_sound[w].shoot_sample = NULL;
	    continue;
	}

	weapon_sound[w].shoot_sample = load_sample(weapon_sound[w].shoot_fn);
	log_load_sound(weapon_sound[w].shoot_fn, weapon_sound[w].shoot_sample);
    }

    missile_sound.sample = load_sample(missile_sound.fn);
    log_load_sound(missile_sound.fn, missile_sound.sample);
}


void weapon_sound_free(void)
{
    enum WEAPON_CLASS w;

    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	if (weapon_sound[w].shoot_sample) {
	    destroy_sample(weapon_sound[w].shoot_sample);
	    weapon_sound[w].shoot_sample = NULL;
	}
    }

    if (missile_sound.sample) {
	destroy_sample(missile_sound.sample);
	missile_sound.sample = NULL;
    }
}

/*--------------------------------------------------------------*/

void sound_poll(void)
{
    int i;
    int vol, pan;
    float t, x, y, dx, dy, doppler;

    if (!sound_enabled)
	return;

    for (i = 0; i < MAX_SOUND_SOURCES; i++) {
	sound_source_t *src = &sound_source[i];

	if (src->voice < 0)
	    continue;

	t = (server_time - src->_spawn_time) / 1000.0;
	if (t < 0)
	    continue;

	if (voice_check(src->voice) == NULL) {
	    src->voice = -1;
	    continue;
	}

	x = src->_spawn_x + src->_spawn_vx*t;
	y = src->_spawn_y + src->_spawn_vy*t;
	dx = x - (camera.x+640/2);
	dy = y - (camera.y+480/2);

	vol = sound_volume_from_dist(src->_base_vol, (dx*dx) + (dy*dy));
	pan = sound_pan_from_dist(dx);

	if (ABS(src->_spawn_vx) < 0.5) {
	    doppler = 1.0;
	}
	else {
	    /* Fake doppler effect that depends on the distance
	       between the source and the observer.
	     */
	    doppler = 1.0 - 0.05*ABS(dx)/(640/2);
	    doppler = MID(0.95, doppler, 1.15);
	}

	voice_set_volume(src->voice, vol);
	voice_set_pan(src->voice, pan);
	voice_set_frequency(src->voice, src->sample->freq*doppler);
    }
}

/*--------------------------------------------------------------*/

void sound_set_volume(const int volume)
{
    sound_volume = MID(0, volume, 255);
}


unsigned int sound_get_volume(void)
{
    return sound_volume;
}

/*--------------------------------------------------------------*/

void sound_start(void)
{
    int i;

    for (i = 0; i < MAX_SOUND_SOURCES; i++) {
	sound_source_t *src = &sound_source[i];

	src->sample = NULL;
	src->voice = -1;
    }
}


void sound_stop(void)
{
    int i;

    for (i = 0; i < MAX_SOUND_SOURCES; i++) {
	if (sound_source[i].voice >= 0) {
	    deallocate_voice(sound_source[i].voice);
	    sound_source[i].sample = NULL;
	    sound_source[i].voice = -1;
	}
    }
}


void sound_init(void)
{
    if (!sound_enabled)
	return;

    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) == 0) {
	sound_enabled = true;
	fprintf(stdout, "Sound installed\n");
    }
    else {
	sound_enabled = false;
	fprintf(stderr, "Error installing sound: %s\n", allegro_error);
	return;
    }
}


void sound_shutdown(void)
{
    sound_stop();
}
