/* smoke.c,
 *
 * Smoke (and explosion) animations.
 */


#include <alleggl.h>
#include <assert.h>
#include "camera.h"
#include "list.h"
#include "server.h"
#include "smoke.h"
#include "stats-smoke.h"
#include "texdraw.h"


static BITMAP *bmp_smoke;
static GLuint tex_smoke;

static list_head_t smoke_list;	/* Client only */
static int smoke_count;

/*--------------------------------------------------------------*/

static void smoke_free(smoke_t *s);

/*--------------------------------------------------------------*/

void smoke_update(void)
{
    const anim_sprite_t *sprite;
    smoke_t *s, *nx;
    int alive_time;

    list_for_each_next(s, nx, &smoke_list) {
	assert(s->class < NUM_SMOKE_CLASSES);

	sprite = &smoke_sprite[s->class];
	alive_time = server_time - s->_spawn_time;
	s->frame = alive_time/sprite->frame_time;

	if (s->frame >= sprite->nframes) {
	    list_remove(s);
	    smoke_free(s);
	}
	else {
	    s->x = s->_spawn_x + s->_spawn_vx*alive_time/1000.0;
	    s->y = s->_spawn_y + s->_spawn_vy*alive_time/1000.0;
	}
    }
}

/*--------------------------------------------------------------*/

static void smoke_draw_unit(const smoke_t *s);


void smoke_draw(void)
{
    smoke_t *s;

    glBindTexture(GL_TEXTURE_2D, tex_smoke);
    glBegin(GL_QUADS);

    list_for_each(s, &smoke_list) {
	smoke_draw_unit(s);
    }

    glEnd();			/* glBegin(GL_QUADS) */
}


static void smoke_draw_unit(const smoke_t *s)
{
#define BITMAP_W	1024.0
#define BITMAP_H	128.0

    const anim_sprite_t *sprite;
    texcoord2d_t coord;
    assert(s);
    assert(s->class < NUM_SMOKE_CLASSES);

    sprite = &smoke_sprite[s->class];

    if (!camera_can_see(&camera, s->x, s->y, sprite->w, sprite->h))
	return;

    coord.w = sprite->w;
    coord.h = sprite->h;
    coord.top = (sprite->y)/BITMAP_H;
    coord.bottom = (sprite->y+sprite->h)/BITMAP_H;
    coord.left = (sprite->x + (s->frame*sprite->w))/BITMAP_W;
    coord.right = (sprite->x + ((s->frame+1) * sprite->w))/BITMAP_W;

    gl_draw_sprite_2d(&coord, s->x-sprite->w/2, s->y-sprite->h/2);

#undef BITMAP_H
#undef BITMAP_W
}

/*--------------------------------------------------------------*/

void smoke_new(const enum SMOKE_CLASS class,
	       const float x, const float y,
	       const float vx, const float vy, const int tick)
{
    smoke_t *s = malloc(sizeof(*s));
    assert(class < NUM_SMOKE_CLASSES);
    assert(s);

    s->class = class;

    /* Spawn state. */
    s->_spawn_time = tick;
    s->_spawn_x = x;
    s->_spawn_y = y;
    s->_spawn_vx = vx;
    s->_spawn_vy = vy;

    /* Current state. */
    s->frame = 0;

    list_add(smoke_list, s);
    smoke_count++;
}


static void smoke_free(smoke_t *s)
{
    free(s);
    smoke_count--;
}

/*--------------------------------------------------------------*/

void smoke_start(void)
{
    list_init(smoke_list);
    smoke_count = 0;
}


void smoke_stop(void)
{
    list_free(smoke_list, smoke_free);
    assert(smoke_count == 0);
}


void smoke_init(void)
{
    bmp_smoke = load_bitmap("data/smoke.tga", NULL);
    assert(bmp_smoke);

    tex_smoke = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_smoke, -1);
}


void smoke_shutdown(void)
{
    if (bmp_smoke) {
	destroy_bitmap(bmp_smoke);
	bmp_smoke = NULL;
    }
}
