/* score.c,
 *
 * Simple scoring system, common to both client and server.
 */

#include <assert.h>

#include "score.h"
#include "sv-internal.h"


bool show_score;
static bool sort_score;

/*--------------------------------------------------------------*/

void sort_scores(void)
{
    if (sort_score) {
	sv_intern_sort_by_score();
	sort_score = false;
    }
}

/*--------------------------------------------------------------*/

/* score_get_rank:
 *
 * Return the rank of the player with index idx, assuming that
 * client_data is sorted.
 * 
 * c is set to '=' if many players have the same score, otherwise it
 * is ' '.
 */
int score_get_rank(const int idx, char *c)
{
    int i;

    for (i = idx-1; i >= 0; i--) {
	if (client_data[idx].sc.total_score != client_data[i].sc.total_score)
	    break;
    }
    i++;

    if (i+1 == MAX_CLIENTS) {
	/* You're the last one! */
	*c = ' ';
	return MAX_CLIENTS;
    }

    if ((client_data[i+1].clid != 0) &&
	(client_data[i].sc.total_score == client_data[i+1].sc.total_score)) {
	*c = '=';
    }
    else {
	*c = ' ';
    }

    return i+1;
}

/*--------------------------------------------------------------*/

static void score_kill(const player_id id);
static void score_death(const player_id id);
static void score_suicide(const player_id id);


void score(const player_id killer, const player_id killee)
{
    if (killer == killee) {
	score_suicide(killer);
    }
    else {
	score_kill(killer);
	score_death(killee);
    }

    sort_score = true;
}


static void score_kill(const player_id id)
{
    score_t *sc;
    int idx;

    idx = sv_intern_client_from_id(id);
    if (idx < 0)
	return;

    sc = &client_data[idx].sc;
    sc->total_score++;
    sc->total_kills++;
    sc->session_score++;
    sc->session_kills++;
}


static void score_death(const player_id id)
{
    score_t *sc;
    int idx;

    idx = sv_intern_client_from_id(id);
    if (idx < 0)
	return;

    sc = &client_data[idx].sc;
    sc->total_deaths++;
    sc->session_deaths++;
}


static void score_suicide(const player_id id)
{
    score_t *sc;
    int idx;

    idx = sv_intern_client_from_id(id);
    if (idx < 0)
	return;

    sc = &client_data[idx].sc;
    sc->total_score--;
    sc->total_deaths++;
    sc->session_score--;
    sc->session_deaths++;
}

/*--------------------------------------------------------------*/

void score_reset(score_t *sc)
{
    sc->total_score = 0;
    sc->session_score = 0;
    sc->total_kills = 0;
    sc->session_kills = 0;
    sc->total_deaths = 0;
    sc->session_deaths = 0;
}

/*--------------------------------------------------------------*/

void score_begin_session(void)
{
    unsigned int c;

    for (c = 0; c < MAX_CLIENTS; c++) {
	client_data[c].sc.session_kills  = 0;
	client_data[c].sc.session_deaths = 0;
	client_data[c].sc.session_score  = 0;
    }
}
