/* 02miser.c,
 *
 * A robot which is more miserly about firing.
 */

#include <assert.h>
#include <math.h>
#include <stdbool.h>

#include "bullet.h"
#include "container.h"
#include "input.h"
#include "robots/01dwarf.h"
#include "robots/02miser.h"
#include "stats-container.h"
#include "weapon-pick.h"
#include "weapon.h"


#define SQ(x)	((x)*(x))

/*--------------------------------------------------------------*/

static bool miser_pick_weapon(robot_t *rob, const player_t *enemy);
static bool miser_try_weapon(robot_t *rob, bullet_t *dummy, const double dist);
static bool miser_try_explosive(robot_t *rob, const enum WEAPON_CLASS w,
				const double rr);


static int miser_fire(robot_t *rob, const player_t *enemy)
{
    enum WEAPON_CLASS old_weapon;
    assert(rob);
    assert(rob->player);
    assert(enemy);

    if (rob->player->fire_delay > 0.0)
	return 0;

    old_weapon = rob->player->weapon;

    if (miser_pick_weapon(rob, enemy)) {
	if (rob->player->weapon != old_weapon)
	    rob->player->packet_size |= PACKET_INCLUDE_WEAPON;

	return INPUT_FIRE;
    }
    else {
	return 0;
    }
}


static bool miser_pick_weapon(robot_t *rob, const player_t *enemy)
{
    bullet_t b;
    double dx, dy, rr;
    bool alive;
    assert(rob);
    assert(rob->player);
    assert(enemy);

    dx = enemy->x - rob->player->x;
    dy = enemy->y - rob->player->y;
    rr = SQ(dx) + SQ(dy);
    rob->player->angle = atan2(dy, dx);
    rob->player->mirror =
	((M_PI_2 < rob->player->angle) || (rob->player->angle < -M_PI_2));

    alive = miser_try_weapon(rob, &b, sqrt(rr));

    dx = b.x - rob->player->x;
    dy = b.y - rob->player->y;
    rr = SQ(dx) + SQ(dy);

    /* Try explosives.  Note that we don't care if it hits a wall or
       not, only whether we are outside of the blast radius. */
    if (miser_try_explosive(rob, WEAPON_MISSILE_LAUNCHER, rr)) {
	rob->player->weapon = WEAPON_MISSILE_LAUNCHER;
	return true;
    }

    if (miser_try_explosive(rob, WEAPON_MICRO_MISSILE, rr)) {
	rob->player->weapon = WEAPON_MICRO_MISSILE;
	return true;
    }

    /* If we died because we hit a container, maybe fire anyway. */
    if (b.hurtee_container) {
	const container_t *c;

	c = container_from_id(b.hurtee_container);
	if (!container_stats[c->class].explosion.explosive) {
	    alive = true;
	}
	else if (rr > SQ(container_stats[c->class].explosion.size)) {
	    alive = true;
	}
    }

    /* After explosives are considered, pick one of minigun, vulcan
       and blaster. */
    if (alive) {
	rob->player->weapon = weapon_pick_no_ammo(rob->player);
	return true;
    }
    else {
	return false;
    }
}


/* miser_try_weapon:
 *
 * Try firing a gravity-less projectile (ie. everything but grenades),
 * velocity 100.0 pixel/sec, with parameters given by robot->player,
 * and return true if it is still alive after travelling dist.
 */
static bool miser_try_weapon(robot_t *rob, bullet_t *b, const double dist)
{
    double time;
    assert(b);

    b->class = WEAPON_BLASTER;
    b->parent = rob->player->id;
    b->_spawn_time = 0;
    b->_spawn_x = rob->player->x;
    b->_spawn_y = rob->player->y + 19.0;
    b->_spawn_vx = 100.0*cos(rob->player->angle);
    b->_spawn_vy = 100.0*sin(rob->player->angle);
    b->hurtee_container = 0;

    b->time = b->_spawn_time;
    b->x = b->_spawn_x;
    b->y = b->_spawn_y;

    time = dist*1000.0/100.0;

    /* Be satisfied if it travels 3/4 dist without colliding. */
    time *= 0.75;

    return bullet_update_unit(b, time, NULL);
}


static bool miser_try_explosive(robot_t *rob, const enum WEAPON_CLASS w,
				const double r_sq)
{
    assert(rob);
    assert(w < NUM_WEAPON_CLASSES);

    if (!rob->player->have_weapon[w])
	return false;

    if (rob->player->weapon_ammo[w] <= 0)
	return false;

    if (r_sq < SQ(weapon[w].explosion.size))
	return false;
    else
	return true;
}

/*--------------------------------------------------------------*/

int miser_get_impies(const unsigned int idx, robot_t *rob,
		     const game_state_t *state)
{
    player_t *enemy;
    double min_dist_sq;
    int impy = 0;
    assert(rob);
    assert(rob->player);
    assert(state);

    if (!(rob->player->alive))
	return INPUT_RESPAWN;

    enemy = player_closest(rob->player, &min_dist_sq, state);
    if (!enemy)
	return 0;

    impy  = dwarf_move(idx, rob, enemy->x, enemy->y);
    impy |= miser_fire(rob, enemy);

    return impy;
}

