/* 01dwarf.c,
 *
 * A robot with some path finding ability.
 */

#include <GL/gl.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>

#include "container.h"
#include "gizmo.h"
#include "gizmo.inc"
#include "input.h"
#include "map.h"
#include "robots/00grunt.h"
#include "robots/01dwarf.h"
#include "universal.h"


#define SQ(x)	((x)*(x))
#define SEARCH_SPACE_X_2	15
#define SEARCH_SPACE_Y_2	15
#define SEARCH_SPACE_X		(SEARCH_SPACE_X_2*2 + 1)
#define SEARCH_SPACE_Y		(SEARCH_SPACE_Y_2*2 + 1)
#define MAX_TRACE_PATH_LEN	256

enum DWARF_MOVE {
    DWARF_MOVE_LEFT,
    DWARF_MOVE_RIGHT,
    DWARF_MOVE_UP,
    DWARF_MOVE_DOWN
};

typedef struct {
    int x, y;
} coord_t;

typedef struct {
    /* Note: the path is stored backwards. */
    coord_t trace_path[MAX_TRACE_PATH_LEN];
    int trace_path_len;
} dwarf_data_t;

static dwarf_data_t dwarf_data[MAX_ROBOTS];

/*--------------------------------------------------------------*/

static double vy_after_displacement2(const double s, const double u);
static double vy_after_tile_w(const double u);


/* dy_after_displacing_tile_w:
 *
 * The expected y-displacement (in pixels) after travelling TILE_W
 * with velocity (3.0, u).  3.0 is the horzontal terminal velocity.
 */
static double dy_after_displacing_tile_w(const double u)
{
    double t = TILE_W/3.0;

    return u*t - 0.5*GRAVITY*t*t;
}


/* vy_after_displacement:
 *
 * The y-velocity (in pixel per tick) after travelling TILE_W or
 * TILE_H (downwards) with initial y-velocity, u.
 */

static double vy_after_displacement(const enum DWARF_MOVE move, const double u)
{
    switch (move) {
      case DWARF_MOVE_LEFT:
      case DWARF_MOVE_RIGHT:
	  return vy_after_tile_w(u);
      case DWARF_MOVE_DOWN:
	  return -vy_after_displacement2(-TILE_H, u);
      case DWARF_MOVE_UP:
      default:
	  assert(false);
	  return 0.0;
    }
}

/* vy_after_displacement2:
 *
 * The y-velocity (in pixel per tick) after travelling s pixels, with
 * initial y-velocity, u, and constant acceleration, -GRAVITY.
 */

static double vy_after_displacement2(const double s, const double u)
{
    double vv = u*u - 2.0*GRAVITY*s;

    if (vv < 0.0) {
	assert(false);
	return 0.0;
    }
    else {
	return sqrt(vv);
    }
}


static double vy_after_tile_w(const double u)
{
    double t = TILE_W/3.0;

    return u - GRAVITY*t;
}

/*--------------------------------------------------------------*/

typedef struct {
    int parent;			/* array index of spawning node */
    int x, y;			/* tile location (tile units) */
    double vy;			/* y velocity visited tile with */
    double yy;			/* sub-tile pixels */
} finder_node_t;


static bool valid_move_normal(const bool on_ground, const enum DWARF_MOVE m, 
			      const finder_node_t *n, int *dy, double *yy);
static bool valid_move_ladder(const enum DWARF_MOVE m, const finder_node_t *n,
			      double *yy);


/* indexes_from_coord:
 *
 * Convert a (x, y) in tile units to a pair of indexes for the visited
 * nodes array.  Return true if it is on the map.
 */
static bool indexes_from_coord(const int x, const int y,
			       const finder_node_t *start, int *rx, int *ry)
{
    assert(start);
    assert(rx != NULL);
    assert(ry != NULL);

    *rx = x - start->x + SEARCH_SPACE_X_2;
    *ry = y - start->y + SEARCH_SPACE_Y_2;

    return ((0 <= *rx) && (*rx < SEARCH_SPACE_X) &&
	    (0 <= *ry) && (*ry < SEARCH_SPACE_Y));
}


static bool node_on_ground(const finder_node_t *n)
{
    if (map.tile[n->y][n->x].shape)
	return true;

    if ((tile_in_map(n->x, n->y-1)) &&
	(map.tile[n->y-1][n->x].shape)) {
	return true;
    }

    if (container_bounding(n->x*TILE_W+TILE_W_2, n->y*TILE_H-TILE_H_2, true))
	return true;

    return false;
}


static bool valid_move(const bool on_ground, const enum DWARF_MOVE m, 
		       const finder_node_t *n, int *dy, double *yy,
		       bool *in_ladder)
{
    gizmo_id gid;
    assert(n);
    assert(dy);
    assert(yy);

    gid = gizmo_bounding(n->x*TILE_W + TILE_W_2, n->y*TILE_H + TILE_H_2);
    if (gid) {
	gizmo_t *giz = gizmo_from_id(gid);

	if (giz->class <= GIZMO_LADDERL5) {
	    *in_ladder = true;
	    return valid_move_ladder(m, n, yy);
	}
    }

    *in_ladder = false;
    return valid_move_normal(on_ground, m, n, dy, yy);
}


static bool valid_move_normal(const bool on_ground, const enum DWARF_MOVE m, 
			      const finder_node_t *n, int *dy, double *yy)
{
    if (on_ground) {
	*yy = 0.0;
	return true;
    }

    if (m == DWARF_MOVE_DOWN) {
	*yy = TILE_H;
	return true;
    }

    if (n->vy >= 0.0) {		/* Jumping. */
	if (m == DWARF_MOVE_UP) {
	    /* Cannot reach above tile with our velocity. */
	    if (n->yy + SQ(n->vy)/(2*GRAVITY) < TILE_H)
		return false;

	    *yy = 0.0;
	}
	else {
	    int ddy = *dy;
	    double y = n->yy + dy_after_displacing_tile_w(n->vy);
	    
	    while (y > TILE_H) {
		y -= TILE_H;
		ddy++;
		
		if ((0 <= n->y+ddy) && (n->y+ddy < map.h) &&
		    map.tile[n->y+ddy][n->x].shape)
		    return false;
	    }

	    *dy = ddy;
	    *yy = y;
	}
    }
    else {			/* Falling */
	if (m == DWARF_MOVE_UP) {
	    return false;
	}
	else {
	    int ddy = *dy;
	    double y = n->yy + dy_after_displacing_tile_w(n->vy);

	    while (y < 0.0) {
		yy += TILE_H;
		ddy--;
		
		if ((n->y+ddy < 0) ||
		    (map.tile[n->y+ddy][n->x].shape))
		    return false;
	    }

	    *dy = ddy;
	    *yy = y;
	}
    }

    return true;
}


static bool valid_move_ladder(const enum DWARF_MOVE m, const finder_node_t *n,
			      double *yy)
{
    if (m == DWARF_MOVE_UP) {
	*yy = 0;
	return true;
    }
    else if (m == DWARF_MOVE_DOWN) {
	*yy = TILE_H;
	return true;
    }
    else {
	*yy = n->yy;
	return true;
    }    
}


/* queue_node:
 *
 * Return true if the point (x, y) should be added to the queue.
 */
static bool queue_node(const int x, const int y, const region_id last_region,
		       const finder_node_t *start,
		       finder_node_t visited[SEARCH_SPACE_Y][SEARCH_SPACE_X])
{
    int rx, ry;
    assert(start);

    if (!tile_in_map(x, y))
	return false;

    /* Map tile occupied. */
    if (map.tile[y][x].shape) {
	/* If same region, continue. */
	if (map.tile[y][x].region &&
	    map.tile[y][x].region == last_region)
	    ;
	else
	    return false;
    }

    if (!indexes_from_coord(x, y, start, &rx, &ry))
	return false;

    if (visited[ry][rx].parent != -1)
	return false;

    visited[ry][rx].parent = 0;
    return true;
}


static bool dwarf_path_find(const finder_node_t *start,
			    const unsigned int dest_region,
			    dwarf_data_t *dwarf)
{
    const int dx[4] = { -1, +1,  0,  0 };
    const int dy[4] = {  0,  0, +1, -1 };

    finder_node_t queue[1024];
    finder_node_t visited[SEARCH_SPACE_Y][SEARCH_SPACE_X];
    int j, x, y, rx, ry;
    int i;			/* Queue node visiting */
    int q_last;			/* Next queue node to insert into. */
    assert(start);
    assert(dwarf);

    if (!tile_in_map(start->x, start->y))
	return false;

    for (i = 0; i < 1024; i++)
	queue[i].parent = -1;

    for (y = 0; y < SEARCH_SPACE_Y; y++) {
	for (x = 0; x < SEARCH_SPACE_X; x++)
	    visited[y][x].parent = -1;
    }

    i = 0;
    q_last = 1;
    indexes_from_coord(start->x, start->y, start, &rx, &ry);
    visited[ry][rx].parent = 0;
    queue[i].parent = 0;
    queue[i].x = start->x;
    queue[i].y = start->y;
    queue[i].vy = start->vy;
    queue[i].yy = start->yy;

    while (i < q_last) {
	const finder_node_t *curr;
	enum DWARF_MOVE m;
	bool on_ground;
	assert(i < 1024);

	curr = &queue[i];

	if (map.tile[curr->y][curr->x].region == dest_region)
	    goto found_way;

	on_ground = node_on_ground(curr);

	for (m = DWARF_MOVE_LEFT; m <= DWARF_MOVE_DOWN; m++) {
	    finder_node_t *q;
	    int ddx = dx[m];
	    int ddy = dy[m];
	    double yy;
	    bool in_ladder;

	    if (on_ground && m == DWARF_MOVE_DOWN)
		continue;

	    if (!valid_move(on_ground, m, curr, &ddy, &yy, &in_ladder))
		continue;

	    if (!queue_node(curr->x + ddx, curr->y + ddy,
			    map.tile[curr->y][curr->x].region,
			    start, visited)) {
		continue;
	    }

	    q = &queue[q_last];
	    q->parent = i;
	    q->x = curr->x + ddx;
	    q->y = curr->y + ddy;
	    q->yy = yy;

	    if (in_ladder) {
		q->vy = 0.0;
	    }
	    else if (on_ground) {
		if (m == DWARF_MOVE_UP) {
		    q->vy = 5.6;
		}
		else {
		    q->vy = 0.0;
		}
	    }
	    else {
		if (m == DWARF_MOVE_UP) {
		    q->vy = vy_after_displacement2(yy + TILE_H - curr->yy,
						   curr->vy);
		}
		else {
		    q->vy = vy_after_displacement(m, curr->vy);
		}
	    }

	    q_last++;
	}

	/* Next. */
	i++;
    }

    dwarf->trace_path_len = 0;
    return false;

 found_way:

    for (j = 0; i != 0; i = queue[i].parent, j++) {
	assert(j < MAX_TRACE_PATH_LEN);

	dwarf->trace_path[j].x = queue[i].x * TILE_W + TILE_W_2;
	dwarf->trace_path[j].y = queue[i].y * TILE_H + TILE_H_2;
    }
    dwarf->trace_path_len = j;

    return true;
}

/*--------------------------------------------------------------*/

static int dwarf_trace_path(const robot_t *rob, const dwarf_data_t *dwarf)
{
    int impy = 0;
    int i;
    assert(rob);
    assert(rob->player);
    assert(dwarf);

    i = dwarf->trace_path_len-1;
    if (i < 0)
	return 0;

    if (dwarf->trace_path[i].x < rob->player->x)
	impy |= INPUT_MOVE_LEFT;
    if (dwarf->trace_path[i].x > rob->player->x)
	impy |= INPUT_MOVE_RIGHT;
    if (dwarf->trace_path[i].y > rob->player->y)
	impy |= INPUT_MOVE_UP;
    if (dwarf->trace_path[i].y < rob->player->y)
	impy |= INPUT_MOVE_DOWN;

    return impy;
}

/*--------------------------------------------------------------*/

static bool dwarf_find_new_path(const finder_node_t *start);


int dwarf_move(const unsigned int idx, robot_t *rob,
	       const int dx, const int dy)
{
    unsigned int curr_region;
    unsigned int dest_region;
    dwarf_data_t *dwarf;
    int px, py;
    assert(rob);
    assert(rob->player);
    assert(idx < MAX_ROBOTS);

    px = (int)rob->player->x;
    py = (int)rob->player->y;
    dwarf = &dwarf_data[idx];

    if (x_in_map(px) && y_in_map(py)) {
	curr_region = map.tile[py/TILE_H][px/TILE_W].region;
    }
    else {
	curr_region = 0;
    }

    if (x_in_map(dx) && y_in_map(dy)) {
	dest_region = map.tile[dy/TILE_H][dx/TILE_W].region;
    }
    else {
	dest_region = 0;
    }

    if ((dest_region == 0) ||
	(dest_region == curr_region)) {
	int impy = 0;

	if (dx < rob->player->x - 128)
	    impy |= INPUT_MOVE_LEFT;
	if (dx > rob->player->x + 128)
	    impy |= INPUT_MOVE_RIGHT;
	if (dy < rob->player->y)
	    impy |= INPUT_MOVE_DOWN;
	if (dy > rob->player->y)
	    impy |= INPUT_MOVE_UP;

	return impy;
    }
    else if (dest_region != 0) {
	finder_node_t start;
	int i;

	i = dwarf->trace_path_len-1;
	if (i > 0) {
	    double delta_x = dwarf->trace_path[i].x - rob->player->x;
	    double delta_y = dwarf->trace_path[i].y - rob->player->y;

	    if (SQ(delta_x) + SQ(delta_y) < SQ(8.0))
		dwarf->trace_path_len--;
	}

	start.parent = -1;
	start.x = rob->player->x/TILE_W;
	start.y = rob->player->y/TILE_H;
	start.vy = rob->player->vy;
	start.yy = rob->player->y - start.x * TILE_W;

	if (dwarf_find_new_path(&start))
	    dwarf_path_find(&start, dest_region, dwarf);

	return dwarf_trace_path(rob, dwarf);
    }

    return 0;
}


static bool dwarf_find_new_path(const finder_node_t *start)
{
    assert(start);

    if (SQ(start->vy) < 1.0)
	return true;

    if (node_on_ground(start))
	return true;

    return false;
}

/*--------------------------------------------------------------*/

int dwarf_get_impies(const unsigned int idx, robot_t *rob,
		     const game_state_t *state)
{
    player_t *enemy;
    double min_dist_sq;
    int impy = 0;
    assert(rob);
    assert(rob->player);
    assert(state);

    if (!(rob->player->alive))
	return INPUT_RESPAWN;

    enemy = player_closest(rob->player, &min_dist_sq, state);
    if (!enemy)
	return 0;

    impy  = dwarf_move(idx, rob, enemy->x, enemy->y);
    impy |= grunt_fire(rob, enemy);

    return impy;
}

/*--------------------------------------------------------------*/

#ifdef DEBUG_ROBOT
void dwarf_draw(const unsigned int idx)
{
    dwarf_data_t *dwarf = &dwarf_data[idx];
    int i;

    glDisable(GL_TEXTURE_2D);
    glBegin(GL_LINE_STRIP);

    glVertex2i(robot[idx].player->x, robot[idx].player->y);
    for (i = dwarf->trace_path_len-1; i >= 0; i--)
	glVertex2i(dwarf->trace_path[i].x, dwarf->trace_path[i].y);

    glEnd();		/* glBegin(GL_LINE_STRIP) */

    for (i = dwarf->trace_path_len-1; i >= 0; i--) {
	glRectf(dwarf->trace_path[i].x-2.0, dwarf->trace_path[i].y-2.0,
		dwarf->trace_path[i].x+2.0, dwarf->trace_path[i].y+2.0);
    }

    glEnable(GL_TEXTURE_2D);
}
#endif
