/* robot.c,
 *
 * Robots and stuff.
 */

#include <assert.h>
#include <math.h>
#include <stdlib.h>

#include "bullet.h"
#include "input.h"
#include "network.h"
#include "robot.h"
#include "robots/00grunt.h"
#include "robots/01dwarf.h"
#include "robots/02miser.h"
#include "weapon-pick.h"
#include "weapon.h"


const char *robot_name[NUM_ROBOT_NAMES] = {
    "Robot", "Bot", "BiffBot", "Gobot"
};

robot_t robot[MAX_ROBOTS];	/* Server only */

unsigned int num_robots;

/*--------------------------------------------------------------*/

static void robot_logic_unit(const unsigned int idx, const game_state_t *state);
static int robot_get_impies(const unsigned int idx, robot_t *rob,
			    const game_state_t *state);


void robot_logic(const game_state_t *state)
{
    unsigned int idx;

    for (idx = 0; idx < MAX_ROBOTS; idx++)
	robot_logic_unit(idx, state);
}


static void robot_logic_unit(const unsigned int idx, const game_state_t *state)
{
    int impy;
    assert(state);

    if (!(robot[idx].player))
	return;

    if (state->editor_enabled) {
	impy = 0;
    }
    else {
	impy = robot_get_impies(idx, &robot[idx], state);
    }

    robot[idx].player->last_impy = impy;
}


static int robot_get_impies(const unsigned int idx, robot_t *rob,
			    const game_state_t *state)
{
    assert(rob);
    assert(state);

    switch (rob->class) {
      case ROBOT_GRUNT:
	  return grunt_get_impies(rob, state);
      case ROBOT_DWARF:
	  return dwarf_get_impies(idx, rob, state);
      case ROBOT_MISER:
	  return miser_get_impies(idx, rob, state);
      default:
	  assert(false);
	  return 0;
    }
}

/*--------------------------------------------------------------*/

#ifdef DEBUG_ROBOT
static void robot_draw_unit(const unsigned int idx);


void robot_draw(void)
{
    unsigned int idx;

    if (client_server_mode != I_AM_CLIENT_SERVER)
	return;

    for (idx = 0; idx < MAX_ROBOTS; idx++)
	robot_draw_unit(idx);
}


static void robot_draw_unit(const unsigned int idx)
{
    const robot_t *r = &robot[idx];

    if (r->player == NULL)
	return;

    switch (r->class) {
      case ROBOT_GRUNT:
	  break;
      case ROBOT_DWARF:
      case ROBOT_MISER:
	  dwarf_draw(idx);
	  break;
      default:
	  assert(false);
	  break;
    }
}
#endif

/*--------------------------------------------------------------*/

void robot_attach(robot_t *rob, const player_id id, const game_state_t *state)
{
    player_t *p;
    assert(rob);
    assert(state);

    p = player_from_id(id, state);
    assert(p);

    rob->player = p;
    rob->class = ROBOT_MISER;
}
