/* region.c,
 *
 * Regions mark areas of the map which are connected "simply".
 *
 * Points A and B are connected simply if there we can walk from A to
 * B and vice-versa without jumping, ladder climbing, etc.
 */

#include <assert.h>
#include <stdio.h>

#include "region.h"


/* Region numbers range from 1-255.  Mask is stored as:
 *
 *   char[0] ->  7,  6,  5,  4,  3,  2,  1,  0
 *   char[1] -> 15, 14, 13, 12, 11, 10,  9,  8
 *
 * Region 0 is marked as used.
 */
static unsigned char region_usage_mask[8];

/*--------------------------------------------------------------*/

static void mask_bit_from_region_id(const region_id id, int *byte, int *bit)
{
    *byte = (id / 8);
    *bit  = 0x01 << (id - (*byte)*8);
}

/*--------------------------------------------------------------*/

void region_mark_used(const region_id id)
{
    int byte, bit;

    mask_bit_from_region_id(id, &byte, &bit);
    region_usage_mask[byte] |= bit;
}

/*--------------------------------------------------------------*/

region_id unused_region(void)
{
    region_id id;
    int i;
    char c;

    for (i = 0, id = 0; i < 8; i++) {
	if (region_usage_mask[i] == 0xff) {
	    id += 8;
	    continue;
	}

	c = region_usage_mask[i];
	while (c & 0x01) {
	    c >>= 1;
	    id++;
	}

	return id;
    }

    return 0;
}
