/* player.c,
 *
 * Player related stuff, such as data loading, drawing, etc.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdbool.h>
#include "angle.h"
#include "backpack.h"
#include "chat.h"
#include "common.h"
#include "container.h"
#include "gizmo.h"
#include "input.h"
#include "list.h"
#include "map.h"
#include "network.h"
#include "player.h"
#include "score.h"
#include "server.h"
#include "sound.h"
#include "step.h"
#include "sv-internal.h"
#include "texdraw.h"
#include "universal.h"
#include "weapon.h"


enum FACE_DIR {
    FACING_FORWARD,
    FACING_UP,
    FACING_DOWN
};


static BITMAP *bmp_body;
static BITMAP *bmp_arms;
static GLuint tex_body;
static GLuint tex_arms;

player_t *player;		/* Local player. */

/*--------------------------------------------------------------*/

static bool player_headfoot_in_tile(const int x, const int y, const int dy)
{
    int xx;

    if (!x_in_map(x) || !y_in_map(y))
	return false;

    for (xx = -PLAYER_W_2+2; xx <= PLAYER_W_2-2; xx++) {
	if (x_in_map(x+xx) && map_occupies(x+xx, y))
	    return true;
    }

    /* Rounded edges. */
    if (y_in_map(y+dy)) {
	if ((x_in_map(x-PLAYER_W_2+1) && map_occupies(x-PLAYER_W_2+1, y+dy)) ||
	    (x_in_map(x+PLAYER_W_2-1) && map_occupies(x+PLAYER_W_2-1, y+dy)))
	    return true;
    }

    if (y_in_map(y+2*dy)) {
	if ((x_in_map(x-PLAYER_W_2) && map_occupies(x-PLAYER_W_2, y+2*dy)) ||
	    (x_in_map(x+PLAYER_W_2) && map_occupies(x+PLAYER_W_2, y+2*dy)))
	    return true;
    }
	    
    return false;
}


static bool player_headfoot_in_container(const int x, const int y)
{
    int xx;

    for (xx = -PLAYER_W_2+2; xx <= PLAYER_W_2-2; xx++) {
	if (container_bounding(x+xx, y, true))
	    return true;
    }

    return false;
}


static bool player_head_in_tile(const int x, const int y)
{
    if (player_headfoot_in_tile(x, y+PLAYER_H-1, -1))
	return true;

    return player_headfoot_in_container(x, y+PLAYER_H-1);
}


static bool player_foot_in_tile(const int x, const int y)
{
    if (player_headfoot_in_tile(x, y, 1))
	return true;

    return player_headfoot_in_container(x, y);
}


static bool player_side_in_tile(const int x, const int dx, const int y,
				int *ramp)
{
    int yy;

    if (!x_in_map(x+dx) || !y_in_map(y))
	return false;

    for (yy = 6; yy < PLAYER_H; yy++) {
	if (y_in_map(y+yy) && map_occupies(x+dx, y+yy))
	    return true;

	if (container_bounding(x+dx, y+yy, true))
	    return true;
    }

    /* Ramping. */
    for (yy = 5; yy >= 0; yy--) {
	if (player_foot_in_tile(x, y+yy)) {
	    if (!player_head_in_tile(x, y+yy+1)) {
		*ramp = yy+1;
		return false;
	    }
	    else
		return true;
	}
    }

    return false;
}

/*--------------------------------------------------------------*/

player_t *player_touching_point(const double X1, const double Y1,
				const game_state_t *state)
{
    player_t *p;
    assert(state);

    list_for_each(p, &state->player_list) {
	if (!p->alive)
	    continue;

	if ((p->x-PLAYER_W_2 <= X1) && (X1 <= p->x+PLAYER_W_2) &&
	    (p->y <= Y1) && (Y1 <= p->y+PLAYER_H-1)) {
	    return p;
	}
    }

    return NULL;
}


player_t *player_touching_box(const int X1, const int Y1,
			      const int W1, const int H1,
			      const game_state_t *state)
{
    player_t *p;
    assert(state);

    list_for_each(p, &state->player_list) {
	if (!p->alive)
	    continue;

	if ((X1-W1/2 <= p->x+PLAYER_W_2) &&
	    (X1+W1/2 >= p->x-PLAYER_W_2) &&
	    (Y1 <= p->y+PLAYER_H-1) && (Y1+H1 >= p->y)) {
	    return p;
	}
    }

    return NULL;
}

/*--------------------------------------------------------------*/

static void
player_die(player_t *p, const enum DEATH_REASON why,
	   const double vx, const double vy, const player_id murderer,
	   const bool backpack, game_state_t *state);


void player_hurt(player_t *p, const int damage,
		 const double vx, const double vy,
		 const player_id hurter, game_state_t *state)
{
    assert(p);
    assert(state);

    p->health -= damage;
    p->packet_size |= PACKET_INCLUDE_HEALTH;

    if (p->health <= 0) {
	enum DEATH_REASON why;
	
	if (hurter == p->id)
	    why = DEATH_BY_SUICIDE;
	else
	    why = DEATH_BY_OPPONENT;

	player_die(p, why, vx, vy, hurter, true, state);
	score(hurter, p->id);
    }
}


static void
player_die(player_t *p, const enum DEATH_REASON why,
	   const double vx, const double vy, const player_id murderer,
	   const bool backpack, game_state_t *state)
{
    struct packet_player_die packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    enum WEAPON_CLASS w;
    assert(state);

    p->alive = false;

    packet.time = server_time;
    packet.id = p->id;
    packet.why = why;
    packet.x = p->x;
    packet.y = p->y;
    packet.vx = p->vx*GAME_SPEED + vx;
    packet.vy = p->vy*GAME_SPEED + vy;
    packet.bp = 0;

    if (backpack) {
	for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	    if ((w != WEAPON_BLASTER) && (p->weapon_ammo[w] > 0)) {
		packet.bp = backpack_generate_unique_id();
		break;
	    }
	}
    }

    packet.murderer = murderer;

    len = packet_player_die_encode(&packet, buf, sizeof(buf));
    server_broadcast(buf, len);

    if (packet.bp) {
	backpack_new2(&packet, p, state);
    }
}

/*--------------------------------------------------------------*/

static void player_update_unit(player_t *p, game_state_t *state);
static void player_copy_dynamics(const void *s, void *d);
static int player_step(void *q, const double dx, const double dy);


void player_update(game_state_t *state)
{
    player_t *p;
    double dx, dy, angle;
    assert(state);
    assert(state->as_server);

    list_for_each(p, &state->player_list) {
	if (p->alive) {
	    input_process(p, p->last_impy, state);
	    player_update_unit(p, state);

	    /* Tracker. */
	    if (p->powerup_duration[POWERUP_TRACKER] < server_time)
		continue;

	    if (!player_tracker_angle(p, &dx, &dy, &angle, state))
		continue;

	    if ((dx*dx <= 640.0*640.0) && (dy*dy <= 480.0*480.0))
		continue;

	    p->packet_size |= PACKET_INCLUDE_TRACKER;
	    p->tracker_angle = angle;
	}
	else {
	    input_process_dead(p, p->last_impy);
	}
    }
}


static void player_update_unit(player_t *p, game_state_t *state)
{
    const walk_callbacks_t callbacks = {
	player_copy_dynamics,
	player_step
    };
    assert(state);

    player_t dummy;

    p->movement_mode = MOVEMENT_NORMAL;

    p->vx = MID(-3.0, p->vx, 3.0);
    walk(p, &dummy, p->vx, p->vy, &callbacks);

    if (gizmo_update_with_player(p)) {
	return;
    }
    else {
	/* Very far down */
	if (p->y < -480.0) {
	    player_die(p, DEATH_BY_FALL, 0.0, 0.0, p->id, false, state);
	    score(p->id, p->id);
	}

	/* Landed? */
	else if ((p->vy <= 0.0) &&
		 (player_foot_in_tile(p->x, p->y-1.0))) {
	    p->vx *= 0.80;
	    p->vy = 0.0;
	    p->can_jump = true;
	    p->jump_frame = MAX_JUMP_FRAMES;
	}
	else {
	    p->vx *= 0.90;
	    p->vy -= GRAVITY;
	    p->can_jump = false;
	}
    }
}


static void player_copy_dynamics(const void *s, void *d)
{
    const player_t *src = s;
    player_t *dest = d;

    dest->x = src->x;
    dest->y = src->y;
    dest->vx = src->vx;
    dest->vy = src->vy;
}


static int player_step(void *q, const double dx, const double dy)
{
    player_t *p = q;
    bool support = false;
    int ret = STEP_NO_COLLIDE;
    int px = p->x + dx;
    int ramp = 0;

    if (((dx < 0.0) && (player_side_in_tile(px,-PLAYER_W_2, p->y, &ramp))) ||
	((dx > 0.0) && (player_side_in_tile(px,+PLAYER_W_2, p->y, &ramp)))) {
	p->vx = 0.0;
	ret |= STEP_COLLIDE_X;
    }
    else {
	p->x += dx;

	if (ramp) {
	    p->y += ramp;
	    p->vx /= ABS(ramp);
	}
    }

    if (((dy >  0.01) && (player_head_in_tile(p->x, p->y+dy))) ||
	((dy < -0.01) && (player_foot_in_tile(p->x, p->y+dy)))) {
	support = true;
    }
    else {
	gizmo_t *g;

	p->y += dy;

	if ((g = player_foot_in_gizmo(p->x, p->y))) {

	    if (((dy >  0.01) && !(p->hold_keys & INPUT_MOVE_UP)) ||
		((dy < -0.01) && !(p->hold_keys & INPUT_MOVE_DOWN))) {
		support = true;
		p->y = g->y + g->h;
	    }
	}
    }

    if (support) {
	p->y = floor(p->y);
	p->vy = 0.0;
	ret |= STEP_COLLIDE_BOTTOM;
    }

    return ret;
}

/*--------------------------------------------------------------*/

void player_save_history(player_t *p,
			 const struct packet_player_update *packet)
{
    int i, max;

    for (i = MAX_HISTORIES-1; i > 0; i--)
	p->history[i] = p->history[i-1];

    p->history[0].server_time = packet->time;
    p->history[0].x = packet->x;
    p->history[0].y = packet->y;
    p->history[0].tics = 0;

    for (i = 1; i < MAX_HISTORIES; i++) {
	p->history[i].tics = (p->history[i-1].server_time -
			      p->history[i].server_time);

	if (p->history[i].tics <= 0)
	    break;
    }

    max = i-1;
    if (max == 0) {
	p->vx = 0.0;
	p->vy = 0.0;
	return;
    }

    p->vx = (p->history[0].x - p->history[1].x) / p->history[1].tics;

    p->vy = 0.0;
    for (i = 0; i < max; i++)
	p->vy += (p->history[i].y - p->history[i+1].y) / p->history[i+1].tics;
    p->vy /= max;
}

/*--------------------------------------------------------------*/

static void player_predict_move_unit(player_t *p);


void player_predict_move(game_state_t *state)
{
    player_t *p;
    assert(state);
    assert(!state->as_server);

    list_for_each(p, &state->player_list) {
	if (p->alive) {
	    assert(p->weapon < NUM_WEAPON_CLASSES);

	    player_predict_move_unit(p);
	    p->history[0].tics++;

	    if (p->frame > 0) {
		p->frame_delay--;
		if (p->frame_delay <= 0) {
		    p->frame_delay = weapon_sprite[p->weapon].frame_time;
		    p->frame++;
		    if (p->frame >= weapon_sprite[p->weapon].nframes)
			p->frame = 0;
		}
	    }
	}
    }
}


static void player_predict_move_unit(player_t *p)
{
    const walk_callbacks_t callbacks = {
	player_copy_dynamics,
	player_step
    };
    player_t dummy;

    if (p->history[0].tics <= 0) {
	/* We just received a update packet.  No prediction. */
	return;
    }

    walk(p, &dummy, p->vx, p->vy, &callbacks);

    if ((p->vy <= 0.0) &&
	(player_foot_in_tile(p->x, p->y-1.0)));
    else
	p->vy -= GRAVITY;
}

/*--------------------------------------------------------------*/

static void player_draw_unit(const player_t *p);
static void player_draw_head(const bool mirror, const bool berserker,
			     const enum FACE_DIR facing,
			     const unsigned char face_num);
static void player_draw_body(const bool mirror);
static void player_draw_arms(const bool mirror, const enum WEAPON_CLASS w,
			     const int frame);


void player_draw(const game_state_t *state)
{
    player_t *p;
    assert(state);
    assert(!state->as_server);

    list_for_each(p, &state->player_list) {
	if (p->alive)
	    player_draw_unit(p);
    }
}


static void player_draw_unit(const player_t *p)
{
    const client_data_t *client;
    bool mirror;
    bool berserker;
    unsigned char face_num;
    int idx;
    int name_col = makecol(0xff, 0xff, 0xff);
    int len;
    enum FACE_DIR facing;
    assert(p);

    idx = sv_intern_client_from_id(p->id);
    if (idx < 0)
	return;

    client = &client_data[idx];
    mirror = p->mirror;
    berserker = (p->powerup_duration[POWERUP_BERSERKER] > server_time);

    if ((p->angle > deg2rad(30.0)) && (p->angle < deg2rad(150.0)))
	facing = FACING_UP;
    else if ((p->angle > deg2rad(-150.0)) && (p->angle < deg2rad(-30.0)))
	facing = FACING_DOWN;
    else
	facing = FACING_FORWARD;

    face_num = client->face_num;
    assert(face_num < NUM_FACES);

    glPushMatrix();

    glTranslated(rint(p->x), rint(p->y), 0.0);

    /* Player's name */
    len = text_length(al_font, client->name);
    allegro_gl_printf(al_font, -len/2.0, 0.0, 0.0, name_col, client->name);

    glBindTexture(GL_TEXTURE_2D, tex_body);
    glBegin(GL_QUADS);
    player_draw_head(mirror, berserker, facing, face_num);
    player_draw_body(mirror);
    glEnd();

    glTranslated(0.0, 19.0, 0.0);
    if (mirror)
	glRotated(rad2deg(p->angle-M_PI), 0.0, 0.0, 1.0);
    else
	glRotated(rad2deg(p->angle), 0.0, 0.0, 1.0);

    player_draw_arms(mirror, p->weapon, p->frame);

    glPopMatrix();
}


static void player_draw_head(const bool mirror, const bool berserker,
			     const enum FACE_DIR facing,
			     const unsigned char face_num)
{
#define BITMAP_W	512.0
#define BITMAP_H	256.0

    texcoord2d_t coord;

    coord.w = 29.0;
    coord.h = 21.0;
    coord.top = 1.0/BITMAP_H;
    coord.left = (face_num*30.0)/BITMAP_W;

    if (facing == FACING_UP)
	coord.top += 22.0/BITMAP_H;
    else if (facing == FACING_DOWN)
	coord.top += 44.0/BITMAP_H;

    if (berserker)
	coord.top += 66.0/BITMAP_H;

    coord.right = coord.left + 29.0/BITMAP_W;
    coord.bottom = coord.top + 21.0/BITMAP_H;

    if (mirror)
	gl_draw_sprite_hflip_2dx(&coord, -14.0, 43.0);
    else
	gl_draw_sprite_2dx(&coord, -14.0, 43.0);

#undef BITMAP_H
#undef BITMAP_W
}


static void player_draw_body(const bool mirror)
{
#define BITMAP_W	512.0
#define BITMAP_H	256.0

    const texcoord2d_t coord = {
	29.0, 43.0,
	213.0/BITMAP_H, (213.0+43.0)/BITMAP_H,
	0.0/BITMAP_W, (0.0+29.0)/BITMAP_W
    };

    if (mirror)
	gl_draw_sprite_hflip_2dx(&coord, -14.0, 0.0);
    else
	gl_draw_sprite_2dx(&coord, -14.0, 0.0);

#undef BITMAP_H
#undef BITMAP_W
}


static void player_draw_arms(const bool mirror, const enum WEAPON_CLASS w,
			     const int frame)
{
#define BITMAP_W	512.0
#define BITMAP_H	256.0

    const weapon_sprite_t *sprite;
    texcoord2d_t coord;
    assert(w < NUM_WEAPON_CLASSES);

    sprite = &weapon_sprite[w];
    coord.w = sprite->w;
    coord.h = sprite->h;
    coord.top = sprite->y/BITMAP_H;
    coord.bottom = (sprite->y+sprite->h)/BITMAP_H;
    coord.left = (sprite->x + frame*(sprite->w+3))/BITMAP_W;
    coord.right = coord.left + sprite->w/BITMAP_W;

    glBindTexture(GL_TEXTURE_2D, tex_arms);
    glBegin(GL_QUADS);

    if (mirror) {
	gl_draw_sprite_hflip_2dx(&coord,
				 -sprite->w+sprite->pivot_x,
				 -sprite->h+sprite->pivot_y);
    }
    else {
	gl_draw_sprite_2dx(&coord,
			   -sprite->pivot_x,
			   -sprite->h+sprite->pivot_y);
    }

    glEnd();			/* glBegin(GL_QUADS) */

#undef BITMAP_H
#undef BITMAP_W
}

/*--------------------------------------------------------------*/

void player_draw_avatar(const unsigned char face_num,
			const int x, const int y)
{
    int src_x, src_y;
    assert(face_num < NUM_FACES);

    src_x = 1+face_num*30.0;
    src_y = 1;

    masked_blit(bmp_body, screen, src_x, src_y, x, y, 29, 21);
}

/*--------------------------------------------------------------*/

player_t *player_from_id(const player_id id, const game_state_t *state)
{
    player_t *p;
    assert(state);

    list_for_each(p, &state->player_list) {
	if (p->id == id)
	    return p;
    }

    return NULL;
}


player_t *player_closest(const player_t *pl, double *min_dist_sq,
			 const game_state_t *state)
{
    player_t *p, *closest = NULL;
    double dx, dy, dist_sq;
    assert(min_dist_sq);
    assert(state);

    *min_dist_sq = 0.0;
    list_for_each(p, &state->player_list) {
	if (p == pl)
	    continue;

	if (p->alive == false)
	    continue;

	dx = p->x - pl->x;
	dy = p->y - pl->y;
	dist_sq = (dx*dx) + (dy*dy);

	if ((!closest) || (dist_sq < *min_dist_sq)) {
	    *min_dist_sq = dist_sq;
	    closest = p;
	}
    }

    return closest;
}


bool player_tracker_angle(const player_t *p,
			  double *dx, double *dy, double *angle,
			  const game_state_t *state)
{
    const player_t *closest;
    double dist_sq;
    assert(dx);
    assert(dy);
    assert(angle);
    assert(state);

    closest = player_closest(p, &dist_sq, state);
    if (!closest)
	return false;

    *dx = closest->x - p->x;
    *dy = closest->y - p->y;
    *angle = atan2(*dy, *dx);
    return true;
}

/*--------------------------------------------------------------*/

void player_new_life(player_t *p)
{
    enum WEAPON_CLASS w;
    enum POWERUP_CLASS u;
    int i;
    assert(p);

    p->alive = true;
    p->health = 100;
    p->next_bleed_time = 0;
    p->vx = 0.0;
    p->vy = 0.0;

    /* Weapons */
    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	p->have_weapon[w] = false;
	p->weapon_ammo[w] = 0;
    }
    p->have_weapon[WEAPON_BLASTER] = true;
    p->weapon = WEAPON_BLASTER;
    p->fire_delay = 0;
    p->frame = 0;
    p->frame_delay = 0;

    /* Powerups */
    for (u = POWERUP_BERSERKER; u < NUM_POWERUP_CLASSES; u++)
	p->powerup_duration[u] = 0;

    /* History */
    for (i = 0; i < MAX_HISTORIES; i++) {
	p->history[i].server_time = 0;
	p->history[i].tics = 0;
	p->history[i].x = p->x;
	p->history[i].y = p->y;
    }

    /* Movement */
    p->movement_mode = MOVEMENT_NORMAL;
    p->can_jump = false;
    p->hold_keys = 0;
    p->jump_frame = 0;

    p->packet_size = PACKET_INCLUDE_ANGLE;
    p->last_impy = 0;
    p->kills_this_life = 0;
}

/*--------------------------------------------------------------*/

player_id player_generate_unique_id(void)
{
    static player_id id;
    id++;
    return id;
}


void player_new(const struct packet_player_new *packet, game_state_t *state)
{
    player_t *p = malloc(sizeof(*p));
    int idx;
    assert(p);
    assert(packet);
    assert(state);

    player_new_life(p);
    p->id = packet->id;
    p->alive = packet->alive;
    p->x = packet->x;
    p->y = packet->y;
    p->mirror = false;
    p->angle = 0.0;
    p->sent_angle = 0.0;
    p->tracker_angle = 0.0;

    list_add(state->player_list, p);

    idx = sv_intern_client_from_id(p->id);
    assert(idx >= 0);
    chat_add(server_time, CHAT_SERVER_LOG, "%s joins the game as player %d!",
	     client_data[idx].name, p->id);
}


void player_shift(const double offx, const double offy, game_state_t *state)
{
    player_t *p;
    assert(state);

    list_for_each(p, &state->player_list) {
	p->x += offx;
	p->y += offy;
    }
}


void player_free(player_t *p)
{
    assert(p);

    list_remove(p);
    free(p);
}

/*--------------------------------------------------------------*/

void player_start(game_state_t *state)
{
    assert(state);

    list_init(state->player_list);
}


void player_stop(game_state_t *state)
{
    assert(state);

    list_free(state->player_list, player_free);
    player = NULL;
}


void player_init(void)
{
    bmp_body = load_bitmap("data/guy-body.tga", NULL);
    assert(bmp_body);

    bmp_arms = load_bitmap("data/guy-arms.tga", NULL);
    assert(bmp_arms);

    tex_body = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_body, -1);
    tex_arms = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_arms, -1);

    player_sound_load();
}


void player_shutdown(void)
{
    player_sound_free();

    if (bmp_arms) {
	destroy_bitmap(bmp_arms);
	bmp_arms = NULL;
    }

    if (bmp_body) {
	destroy_bitmap(bmp_body);
	bmp_body = NULL;
    }
}
