/* packet-pickup.c,
 *
 * Convert pickup command packets to/from a buffer, to be sent across
 * the network.
 */

#include <assert.h>
#include "network.h"
#include "packet-pickup.h"
#include "stats-pickup.h"


/*--------------------------------------------------------------*/

NLint packet_pickup_new_begin_encode(const int num,
				     NLbyte *buf, const NLint sz)
{
    NLint count = 0;
    assert(num > 0);
    assert(buf);

    writeByte(buf, count, COMMAND_PICKUP_NEW);
    writeByte(buf, count, num);

    assert(count <= sz);
    return count;
}


NLint packet_pickup_new_begin_decode(int *num,
				     const NLbyte *buf, const NLint sz)
{
    NLint count = 1;
    int n;
    assert(num);
    assert(buf);

    readByte(buf, count, n); *num = n;

    assert(count <= sz);
    return count;
}


NLint packet_pickup_new_encode(const struct packet_pickup_new *packet,
			       NLbyte *buf, const NLint sz, NLint count)
{
    int xx = packet->x + zero_x;
    int yy = packet->y + zero_y;
    char symbol = pickup_class_to_symbol(packet->class);
    assert(packet);
    assert(buf);

    writeLong (buf, count, packet->time);
    writePickupID(buf, count, packet->id);
    writePickupClass(buf, count, symbol);
    writeShort(buf, count, xx);
    writeShort(buf, count, yy);

    assert(count <= sz);
    return count;
}


NLint packet_pickup_new_decode(struct packet_pickup_new *packet,
			       const NLbyte *buf, const NLint sz, NLint count)
{
    int xx, yy;
    char symbol;
    assert(packet);
    assert(buf);

    readLong (buf, count, packet->time);
    readPickupID(buf, count, packet->id);
    readPickupClass(buf, count, symbol);
    packet->class = pickup_class_from_symbol(symbol);
    readShort(buf, count, xx); packet->x = xx - zero_x;
    readShort(buf, count, yy); packet->y = yy - zero_y;

    assert(count <= sz);
    return count;
}

/*--------------------------------------------------------------*/

NLint packet_pickup_quaff_encode(const struct packet_pickup_quaff *packet,
				 NLbyte *buf, NLint size)
{
    NLint count = 0;

    writeByte (buf, count, COMMAND_PICKUP_QUAFF);
    writeLong (buf, count, packet->time);
    writePickupID(buf, count, packet->id);
    writePlayerID(buf, count, packet->quaffer);

    assert(count <= size);
    return count;
}


void packet_pickup_quaff_decode(struct packet_pickup_quaff *packet,
				const NLbyte *buf, const NLint size)
{
    NLint count = 1;

    readLong (buf, count, packet->time);
    readPickupID(buf, count, packet->id);
    readPlayerID(buf, count, packet->quaffer);

    assert(count == size);
}

/*--------------------------------------------------------------*/
/* Map editor only.						*/
/*--------------------------------------------------------------*/

NLint packet_pickup_mov_encode(const struct packet_pickup_mov *packet,
			       NLbyte *buf, NLint buf_size)
{
    NLint count = 0;
    int xx = packet->x + zero_x;
    int yy = packet->y + zero_y;

    writePickupID(buf, count, packet->id);
    writeShort(buf, count, xx);
    writeShort(buf, count, yy);

    assert(count <= buf_size);
    return count;
}


void packet_pickup_mov_decode(struct packet_pickup_mov *packet,
			      const NLbyte *buf, const NLint buf_size)
{
    NLint count = 0;
    int xx, yy;

    readPickupID(buf, count, packet->id);
    readShort(buf, count, xx); packet->x = xx - zero_x;
    readShort(buf, count, yy); packet->y = yy - zero_y;

    assert(count == buf_size);
}


NLint packet_pickup_del_encode(const struct packet_pickup_del *packet,
			       NLbyte *buf, NLint buf_size)
{
    NLint count = 0;

    writePickupID(buf, count, packet->id);

    assert(count <= buf_size);
    return count;
}


void packet_pickup_del_decode(struct packet_pickup_del *packet,
			      const NLbyte *buf, const NLint buf_size)
{
    NLint count = 0;

    readPickupID(buf, count, packet->id);

    assert(count == buf_size);
}
