/* packet-map.c,
 *
 * Encode/decode a block of map to send over the network.
 */

#include <assert.h>

#include "network.h"
#include "packet-map.h"


/*--------------------------------------------------------------*/

NLint packet_map_encode(const map_t *src, const int X0, const int Y0,
			NLbyte *buf, NLint size)
{
    NLint count = 0;
    int x, y;

    writeByte(buf, count, COMMAND_MAP);
    writeShort(buf, count, X0);
    writeShort(buf, count, Y0);

    for (y = Y0; y < Y0+MAP_BLOCK_H; y++) {
	for (x = X0; x < X0+MAP_BLOCK_W; x++) {
	    if ((x >= src->w) || (y >= src->h)) {
		writeByte(buf, count, 0);
		writeByte(buf, count, 0);
	    }
	    else {
		writeByte(buf, count, src->tile[y][x].shape);
		writeByte(buf, count, src->tile[y][x].region);
	    }
	}
    }

    assert(count <= size);
    return count;
}


void packet_map_decode(map_t *dest, const NLbyte *buf, const NLint size,
		       game_state_t *state)
{
    NLint count = 1;
    int x, y, X0, Y0;

    (void)dest;
    readShort(buf, count, X0);
    readShort(buf, count, Y0);

    for (y = Y0; y < Y0+MAP_BLOCK_H; y++) {
	for (x = X0; x < X0+MAP_BLOCK_W; x++) {
	    unsigned int tile;
	    region_id region;

	    readByte(buf, count, tile);
	    readByte(buf, count, region);

	    map_set_tile(x, y, tile, state);
	    map_set_regions(x, y, region, 0);
	}
    }

    zero_tx = 0;
    zero_ty = 0;
    zero_x  = 0;
    zero_y  = 0;

    assert(count == size);
}

/*--------------------------------------------------------------*/

NLint packet_set_zero_encode(const int dx, const int dy,
			     NLbyte *buf, NLint size)
{
    NLint count = 0;
    signed char xx = dx;
    signed char yy = dy;

    writeByte(buf, count, COMMAND_SET_ZERO);
    writeByte(buf, count, xx);
    writeByte(buf, count, yy);

    assert(count <= size);
    return count;
}


void packet_set_zero_decode(const NLbyte *buf, const NLint size)
{
    NLint count = 1;
    signed char xx, yy;

    readByte(buf, count, xx);
    readByte(buf, count, yy);

    zero_tx += xx;
    zero_ty += yy;
    zero_x  += xx * TILE_W;
    zero_y  += yy * TILE_H;

    assert(count == size);
}
