/* packet-editor.c,
 *
 * Convert map editor command packets to/from a buffer, to be sent
 * across the network.
 */

#include <assert.h>
#include <stdio.h>
#include "camera.h"
#include "network.h"
#include "packet-editor.h"


/*--------------------------------------------------------------*/

enum EDITOR_MODE_ACTION
to_mode_action(const enum EDITOR_MODE mode, const enum EDITOR_ACTION action)
{
    switch (mode) {
      case EDIT_TILES:
	  if (action == EDIT_ADD)	return EDIT_TILE_SET;
	  break;

      case EDIT_CONTAINERS:
	  if (action == EDIT_ADD)	return EDIT_CONTAINER_ADD;
	  if (action == EDIT_MOVE)	return EDIT_CONTAINER_MOVE;
	  if (action == EDIT_DELETE)	return EDIT_CONTAINER_DELETE;
	  break;

      case EDIT_GIZMOS:
	  if (action == EDIT_ADD)	return EDIT_GIZMO_ADD;
	  if (action == EDIT_MOVE)	return EDIT_GIZMO_MOVE;
	  if (action == EDIT_DELETE)	return EDIT_GIZMO_DELETE;
	  break;

      case EDIT_PICKUPS:
	  if (action == EDIT_ADD)	return EDIT_PICKUP_ADD;
	  if (action == EDIT_MOVE)	return EDIT_PICKUP_MOVE;
	  if (action == EDIT_DELETE)	return EDIT_PICKUP_DELETE;
	  break;

      case EDIT_STARTS:
	  if (action == EDIT_ADD)	return EDIT_START_ADD;
	  if (action == EDIT_DELETE)	return EDIT_START_DELETE;
	  break;

      case EDIT_CANDELA:
	  if (action == EDIT_ADD)	return EDIT_CANDELA_ADD;
	  if (action == EDIT_DELETE)	return EDIT_CANDELA_DELETE;
	  break;

      case EDIT_REGIONS:
	  if (action == EDIT_ADD)	return EDIT_REGION_SET;
	  break;
    }

    return EDIT_UNKNOWN;
}

/*--------------------------------------------------------------*/

NLint packet_editor_encode(const struct packet_editor *packet,
			   NLbyte *buf, NLint size)
{
#define	BUFCOUNT	(buf), (size), (count)
#define	BUFSIZE		(buf+count), (size-count)

    NLint count = 0;

    writeByte(buf, count, COMMAND_INPUT_EDITOR);
    writeByte(buf, count, packet->mode);

    switch (packet->mode) {
      case EDIT_TILE_SET: {
	  short xx = packet->new.tile.x + zero_tx;
	  short yy = packet->new.tile.y + zero_ty;

	  writeShort(buf, count, packet->new.tile.tile);
	  writeByte (buf, count, packet->new.tile.brush_size);
	  writeShort(buf, count, xx);
	  writeShort(buf, count, yy);
	  break;
      }
      case EDIT_CONTAINER_ADD:
	  count = packet_container_new_encode(&packet->new.container_new,
					      BUFCOUNT);
	  break;
      case EDIT_CONTAINER_MOVE:
	  count += packet_container_mov_encode(&packet->new.container_mov,
					       BUFSIZE);
	  break;
      case EDIT_CONTAINER_DELETE:
	  count += packet_container_del_encode(&packet->new.container_del,
					       BUFSIZE);
	  break;
      case EDIT_GIZMO_ADD:
	  count = packet_gizmo_new_encode(&packet->new.gizmo_new, BUFCOUNT);
	  break;
      case EDIT_GIZMO_MOVE:
	  count += packet_gizmo_mov_encode(&packet->new.gizmo_mov, BUFSIZE);
	  break;
      case EDIT_GIZMO_DELETE:
	  count += packet_gizmo_del_encode(&packet->new.gizmo_del, BUFSIZE);
	  break;
      case EDIT_PICKUP_ADD:
	  count = packet_pickup_new_encode(&packet->new.pickup_new, BUFCOUNT);
	  break;
      case EDIT_PICKUP_MOVE:
	  count += packet_pickup_mov_encode(&packet->new.pickup_mov, BUFSIZE);
	  break;
      case EDIT_PICKUP_DELETE:
	  count += packet_pickup_del_encode(&packet->new.pickup_del, BUFSIZE);
	  break;
      case EDIT_START_ADD:
	  count += packet_start_loc_encode(&packet->new.start_new, BUFSIZE);
	  break;
      case EDIT_START_DELETE:
	  count += packet_start_del_encode(&packet->new.start_del, BUFSIZE);
	  break;
      case EDIT_CANDELA_ADD:
	  count = packet_candela_new_encode(&packet->new.candela_new,
					    BUFCOUNT);
	  break;
      case EDIT_CANDELA_DELETE:
	  count += packet_candela_delete_encode(&packet->new.candela_del,
						BUFSIZE);
	  break;
      case EDIT_REGION_SET: {
	  short xx = packet->new.tile.x + zero_tx;
	  short yy = packet->new.tile.y + zero_ty;

	  writeShort(buf, count, packet->new.region.region);
	  writeByte (buf, count, packet->new.region.brush_size);
	  writeShort(buf, count, xx);
	  writeShort(buf, count, yy);
	  break;
      }
      case EDIT_PLAYER_TELEPORT: {
	  short xx = packet->new.player_tp.x + zero_x;
	  short yy = packet->new.player_tp.y + zero_y;

	  writePlayerID(buf, count, packet->new.player_tp.id);
	  writeShort(buf, count, xx);
	  writeShort(buf, count, yy);
	  break;
      }
      case EDITOR_ENABLE:
      case EDITOR_DISABLE:
	  break;
      case EDIT_UNKNOWN:
      default:
	  fprintf(stderr, "[P.Editor] Unknown mode/action encoded: %d, %d\n",
		  packet->editor_mode, packet->action);
	  return 0;
    }

    assert(count <= size);
    return count;

#undef BUFSIZE
#undef BUFCOUNT
}


void packet_editor_decode(struct packet_editor *packet,
			  const NLbyte *buf, const NLint size)
{
#define BUFCOUNT	(buf), (size), (count)
#define	BUFSIZE		(buf+count), (size-count)

    NLint count = 1;

    readByte(buf, count, packet->mode);
    /* Note: packet->editor_mode and action are NOT reliable because
       I'm lazy. */

    switch (packet->mode) {
      case EDIT_TILE_SET: {
	  short xx, yy;

	  readShort(buf, count, packet->new.tile.tile);
	  readByte (buf, count, packet->new.tile.brush_size);
	  readShort(buf, count, xx);
	  readShort(buf, count, yy);

	  /* Be sure we keep the -ve sign. */
	  packet->new.tile.x = xx - zero_tx;
	  packet->new.tile.y = yy - zero_ty;

	  assert(count == size);
	  break;
      }
      case EDIT_CONTAINER_ADD:
	  packet_container_new_decode(&packet->new.container_new, BUFCOUNT);
	  break;
      case EDIT_CONTAINER_MOVE:
	  packet_container_mov_decode(&packet->new.container_mov, BUFSIZE);
	  break;
      case EDIT_CONTAINER_DELETE:
	  packet_container_del_decode(&packet->new.container_del, BUFSIZE);
	  break;
      case EDIT_GIZMO_ADD:
	  packet_gizmo_new_decode(&packet->new.gizmo_new, BUFCOUNT);
	  break;
      case EDIT_GIZMO_MOVE:
	  packet_gizmo_mov_decode(&packet->new.gizmo_mov, BUFSIZE);
	  break;
      case EDIT_GIZMO_DELETE:
	  packet_gizmo_del_decode(&packet->new.gizmo_del, BUFSIZE);
	  break;
      case EDIT_PICKUP_ADD:
	  packet_pickup_new_decode(&packet->new.pickup_new, BUFCOUNT);
	  break;
      case EDIT_PICKUP_MOVE:
	  packet_pickup_mov_decode(&packet->new.pickup_mov, BUFSIZE);
	  break;
      case EDIT_PICKUP_DELETE:
	  packet_pickup_del_decode(&packet->new.pickup_del, BUFSIZE);
	  break;
      case EDIT_START_ADD:
	  packet_start_loc_decode(&packet->new.start_new, BUFSIZE);
	  break;
      case EDIT_START_DELETE:
	  packet_start_del_decode(&packet->new.start_del, BUFSIZE);
	  break;
      case EDIT_CANDELA_ADD:
	  packet_candela_new_decode(&packet->new.candela_new, BUFCOUNT);
	  break;
      case EDIT_CANDELA_DELETE:
	  packet_candela_delete_decode(&packet->new.candela_del, BUFSIZE);
	  break;
      case EDIT_REGION_SET: {
	  short xx, yy;

	  readShort(buf, count, packet->new.region.region);
	  readByte (buf, count, packet->new.region.brush_size);
	  readShort(buf, count, xx);
	  readShort(buf, count, yy);

	  /* Be sure we keep the -ve sign. */
	  packet->new.region.x = xx - zero_tx;
	  packet->new.region.y = yy - zero_ty;

	  assert(count == size);
	  break;
      }
      case EDIT_PLAYER_TELEPORT: {
	  short xx, yy;

	  readPlayerID(buf, count, packet->new.player_tp.id);
	  readShort(buf, count, xx); packet->new.player_tp.x = xx - zero_x;
	  readShort(buf, count, yy); packet->new.player_tp.y = yy - zero_y;

	  assert(count == size);
	  break;
      }
      case EDITOR_ENABLE:
      case EDITOR_DISABLE:
	  camera.desired_scale = 1.0;
	  break;
      case EDIT_UNKNOWN:
      default:
	  fprintf(stderr, "[P.Editor] Unknown mode/action received: %d\n",
		  packet->mode);
    }

#undef BUFSIZE
#undef BUFCOUNT
}
