/* meat.c,
 *
 * Chunks of meat for when a player dies.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdlib.h>

#include "angle.h"
#include "camera.h"
#include "list.h"
#include "map.h"
#include "meat.h"
#include "particle.h"
#include "server.h"
#include "stats-meat.h"
#include "step.h"
#include "texdraw.h"
#include "universal.h"


static BITMAP *bmp_meat;
static GLuint tex_meat;

list_head_t meat_list;
static int meat_count;

/*--------------------------------------------------------------*/

static void meat_free(meat_t *m);

/*--------------------------------------------------------------*/

static void meat_update_unit(meat_t *m);
static void meat_velocity(const void *q, double *vx, double *vy);
static int meat_step(void *q, const int tick, const void *_);
static bool meat_collided(void *q, const int tick, const int collisions);


void meat_update(void)
{
    const meat_stats_t *stat;
    meat_t *m, *nx;

    list_for_each_next(m, nx, &meat_list) {
	meat_update_unit(m);

	if (m->life <= 0) {
	    list_remove(m);
	    meat_free(m);
	    continue;
	}

	assert(m->class < NUM_MEAT_CLASSES);
	stat = &meat_stats[m->class];

	if ((stat->particles.spew) && (m->time > m->next_blood_time)) {
	    enum PARTICLE_CLASS c;
	    double vx, vy;
	    meat_velocity(m, &vx, &vy);

	    /* Extra momentum from the spin of the meat. */
	    vx = vx/2.0 + 15.0*cos(deg2rad(m->spin-90));
	    vy = vy/2.0 + 15.0*sin(deg2rad(m->spin-90));

	    c = particle_class_from_category(stat->particles.cat);

	    particle_new_unit(c, m->x, m->y, vx, vy, m->next_blood_time);
	    m->next_blood_time =
		m->next_blood_time+25 * 100/particle_get_multiplier();
	}
    }
}


static void meat_update_unit(meat_t *m)
{
    const walk_time_callbacks_t callbacks = {
	meat_velocity,
	meat_step,
	meat_collided
    };

    m->life -= 1000.0/GAME_SPEED;
    m->spin += m->delta_spin;
    walk_time(m, &callbacks, m->time, server_time, NULL);
}


static void meat_velocity(const void *q, double *vx, double *vy)
{
    const meat_t *m = q;
    assert(m);

    *vx = m->_spawn_vx;
    *vy = m->_spawn_vy - GRAVITY2*(m->time-m->_spawn_time)/1000.0;
}


static void meat_move(meat_t *m, const int tick)
{
    double t = (tick - m->_spawn_time)/1000.0;

    m->time = tick;
    m->x = m->_spawn_x + m->_spawn_vx*t;
    m->y = m->_spawn_y + m->_spawn_vy*t - 0.5*GRAVITY2*t*t;
}


static int meat_step(void *q, const int tick, const void *_)
{
    double vx, vy;
    meat_t *m = q;
    int collisions = 0;
    (void)_;
    assert(m);

    meat_move(m, tick);

    /* Check collisions. */
    if (!x_in_map(m->x))
	return STEP_COLLIDE_X;

    if (!y_in_map(m->y))
	return STEP_COLLIDE_Y;

    meat_velocity(m, &vx, &vy);

    if ((vx < 0.0 && x_in_map(m->x-1) && map_occupies(m->x-1, m->y)) ||
	(vx > 0.0 && x_in_map(m->x+1) && map_occupies(m->x+1, m->y)))
	collisions |= STEP_COLLIDE_X;

    if ((vy < 0.0 && y_in_map(m->y-1) && map_occupies(m->x, m->y-1)) ||
	(vy > 0.0 && y_in_map(m->y+1) && map_occupies(m->x, m->y+1)))
	collisions |= STEP_COLLIDE_Y;

    return collisions;
}


static bool meat_collided(void *q, const int tick, const int collisions)
{
    double vx, vy;
    meat_t *m = q;
    assert(m);
    assert(m->class < NUM_MEAT_CLASSES);

    meat_move(m, tick);
    m->_spawn_time = tick;
    m->_spawn_x = m->x;
    m->_spawn_y = m->y;
    m->delta_spin /= 2.0;

    meat_velocity(m, &vx, &vy);

    if (collisions & STEP_COLLIDE_X)
	m->_spawn_vx = -vx * meat_stats[m->class].bounciness;
    if (collisions & STEP_COLLIDE_Y)
	m->_spawn_vy = -vy * meat_stats[m->class].bounciness;

    return true;
}

/*--------------------------------------------------------------*/

static void meat_draw_unit(const meat_t *m);


void meat_draw(void)
{
    meat_t *m;

    glBindTexture(GL_TEXTURE_2D, tex_meat);

    list_for_each(m, &meat_list)
	meat_draw_unit(m);
}


static void meat_draw_unit(const meat_t *m)
{
    const sprite_t *sprite;
    texcoord2d_t coord;
    assert(m->class < NUM_MEAT_CLASSES);

    sprite = &meat_sprite[m->class];

    if (!camera_can_see(&camera, m->x, m->y, sprite->w/2.0, sprite->h/2.0))
	return;

    texcoord_from_sprite(&coord, sprite, bmp_meat->w, bmp_meat->h);

    glPushMatrix();
    glTranslated(m->x, m->y, 0.0);
    glRotated(m->spin, 0.0, 0.0, 1.0);

    glBegin(GL_QUADS);
    gl_draw_sprite_2d(&coord, -sprite->w/4.0, -sprite->h/4.0);
    glEnd();			/* glBegin(GL_QUADS) */

    glPopMatrix();
}

/*--------------------------------------------------------------*/

static void meat_new_unit(const enum MEAT_CLASS class,
			  const float x, const float y,
			  const float vx, const float vy,
			  const double angle, const int tick);


void meat_new(const enum MEAT_CATEGORY cat,
	      const float x, const float y,
	      const float vx, const float vy, const int tick)
{
    enum MEAT_CLASS class;
    unsigned int num;
    int i;
    assert(cat < NUM_MEAT_CATEGORIES);

    num = meat_spawn_num[cat];

    for (i = 0; i < 180; i += 180/num) {
	class = meat_class_from_category(cat);
	meat_new_unit(class, x, y, vx, vy, deg2rad(i), tick);
    }
}


static void meat_new_unit(const enum MEAT_CLASS class,
			  const float x, const float y,
			  const float vx, const float vy,
			  const double angle, const int tick)
{
    const meat_stats_t *stat;
    meat_t *m = malloc(sizeof(*m));
    double v;
    assert(class < NUM_MEAT_CLASSES);
    assert(m);

    stat = &meat_stats[class];
    v = stat->v_min + (rand() % stat->v_rand);
    m->class = class;

    /* Spawn state. */
    m->_spawn_time = tick - (rand() % 100);
    m->_spawn_x = x;
    m->_spawn_y = y;
    m->_spawn_vx = vx + v*cos(angle);
    m->_spawn_vy = vy + v*sin(angle);

    /* Current state. */
    m->life = stat->life_min + (rand() % stat->life_rand);
    m->time = m->_spawn_time;
    m->x = m->_spawn_vx;
    m->y = m->_spawn_vy;
    m->spin = 0.0;
    m->delta_spin = 5.0;

    m->next_blood_time = m->time;

    list_add(meat_list, m);
    meat_count++;
}


static void meat_free(meat_t *m)
{
    free(m);
    meat_count--;
}

/*--------------------------------------------------------------*/

void meat_start(void)
{
    list_init(meat_list);
    meat_count = 0;
}


void meat_stop(void)
{
    list_free(meat_list, meat_free);
    assert(meat_count == 0);
}


void meat_init(void)
{
    bmp_meat = load_bitmap("data/meat.tga", NULL);
    assert(bmp_meat);

    tex_meat = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_meat, -1);
}


void meat_shutdown(void)
{
    if (bmp_meat) {
	destroy_bitmap(bmp_meat);
	bmp_meat = NULL;
    }
}
