/* map.c,
 *
 * Map and tile stuff.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <stdio.h>

#include "angle.h"
#include "background.h"
#include "brush.h"
#include "camera.h"
#include "candela.h"
#include "container.h"
#include "gizmo.h"
#include "map.h"
#include "maxmin.h"
#include "network.h"
#include "packet-map.h"
#include "pickup.h"
#include "region.h"
#include "player.h"
#include "server.h"
#include "start-loc.h"
#include "texdraw.h"


typedef struct {
    float right;
    float top;
    float left;
    float bottom;
} tile_normal_t;

static BITMAP *bmp_tiles;
static GLuint tex_tiles;
static tile_normal_t *tile_normal;

map_t map;

/*--------------------------------------------------------------*/

static void get_tile_sprite_coord(const int tile, int *row, int *col)
{
    const int tiles_per_row = 4;

    /* The first tile starts at 1. */
    assert((1 <= tile) && (tile <= NUM_TILES));

    *row = (tile-1) / tiles_per_row;
    *col = (tile-1) - (*row)*tiles_per_row;
}

/*--------------------------------------------------------------*/

static void map_tile_reset(map_tile_t *t)
{
    assert(t);

    t->shape = 0;
    t->region = 0;
}

/*--------------------------------------------------------------*/

bool map_occupies(const int x, const int y)
{
    int row, col, xx, yy;
    int tile;
    assert(x_in_map(x));
    assert(y_in_map(y));

    tile = map.tile[y/TILE_H][x/TILE_W].shape;

    if (!tile)
	return false;

    /* Do pixel perfect collision detection. */
    get_tile_sprite_coord(tile, &row, &col);
    xx = col*TILE_W + x%TILE_W;
    yy = row*TILE_H + (TILE_H-1) - y%TILE_H;
    
    if (getpixel(bmp_tiles, xx, yy) == bitmap_mask_color(bmp_tiles))
	return false;
    else
	return true;
}


bool tile_in_map(const int x, const int y)
{
    return ((0 <= x) && (x < map.w) &&
	    (0 <= y) && (y < map.h));
}


bool x_in_map(const int x)
{
    return ((0 <= x) && (x <= map.w*TILE_W-1));
}


bool y_in_map(const int y)
{
    return ((0 <= y) && (y <= map.h*TILE_H-1));
}


float normal_angle(const int x, const int y, const enum TILE_FACE e)
{
    int i = map.tile[y/TILE_H][x/TILE_W].shape-1;

    switch (e) {
      case TILE_FACE_RIGHT:
	  return tile_normal[i].right;
      case TILE_FACE_TOP:
	  return tile_normal[i].top;
      case TILE_FACE_LEFT:
	  return tile_normal[i].left;
      case TILE_FACE_BOTTOM:
	  return tile_normal[i].bottom;
      default:
	  assert(false);
    }
}

/*--------------------------------------------------------------*/

void map_set_tile(int x, int y, const unsigned int tile, game_state_t *state)
{
#define EXTEND_AMOUNT	8

    assert(state);

    if (tile > NUM_TILES)
	return;

    if (y >= map.alloc_h) {
	map_resize_h(y+EXTEND_AMOUNT);
    }
    else if (y < 0) {
	int dy = EXTEND_AMOUNT-y;
	map_resize_h(map.h+dy);
	map.h += dy;
	map_shift(0, dy, state);
	y = EXTEND_AMOUNT;
    }

    if (x >= map.alloc_w) {
	map_resize_w(x+EXTEND_AMOUNT);
    }
    else if (x < 0) {
	int dx = EXTEND_AMOUNT-x;
	map_resize_w(map.w+dx);
	map.w += dx;
	map_shift(dx, 0, state);
	x = EXTEND_AMOUNT;
    }

    map.tile[y][x].shape = tile;

    map.w = maxi(map.w, x+1);
    map.h = maxi(map.h, y+1);

#undef EXTEND_AMOUNT
}


void map_set_tiles(const int x, const int y, const unsigned int tile,
		   const unsigned char brush_size, game_state_t *state)
{
    int xx, yy;

    for (yy = 0; yy < MAX_BRUSH_H; yy++) {
	for (xx = 0; xx < MAX_BRUSH_W; xx++) {
	    if (brushes[yy][xx] > brush_size)
		continue;

	    map_set_tile(x+xx-BRUSH_W_2, y+yy-BRUSH_H_2, tile, state);
	}
    }
}


static bool map_is_diff(const int x, const int y, const unsigned int tile)
{
    if (x_in_map(x) && y_in_map(y))
	return (map.tile[y/TILE_H][x/TILE_W].shape != tile);
    else
	return false;
}


bool map_is_different(const int x, const int y, const unsigned int tile,
		      const unsigned char brush_size)
{
    int xx, yy;

    for (yy = 0; yy < MAX_BRUSH_H; yy++) {
	for (xx = 0; xx < MAX_BRUSH_W; xx++) {
	    if (brushes[yy][xx] > brush_size)
		continue;

	    if (map_is_diff(x+(xx-BRUSH_W_2)*TILE_W,
			    y+(yy-BRUSH_H_2)*TILE_H, tile))
		return true;
	}
    }

    return false;
}


unsigned int map_select_tile(const int tile)
{
    return MID(0, tile, NUM_TILES);
}

/*--------------------------------------------------------------*/

void map_set_regions(const int X0, const int Y0, const region_id region,
		     const unsigned char brush_size)
{
    int x, y, xx, yy;

    for (yy = 0; yy < MAX_BRUSH_H; yy++) {
	for (xx = 0; xx < MAX_BRUSH_W; xx++) {
	    if (brushes[yy][xx] > brush_size)
		continue;

	    x = X0+xx-BRUSH_W_2;
	    y = Y0+yy-BRUSH_H_2;
	    if ((0 > X0) || (X0 >= map.alloc_w) ||
		(0 > Y0) || (Y0 >= map.alloc_h))
		continue;

	    map.tile[y][x].region = region;
	}
    }

    region_mark_used(region);
}

/*--------------------------------------------------------------*/

void map_bind_tiles(void)
{
    glBindTexture(GL_TEXTURE_2D, tex_tiles);
}


void map_draw(void)
{
    int x, y;
    int X0 = MID(0, camera.x/TILE_W, map.w-1);
    int Y0 = MID(0, camera.y/TILE_H, map.h-1);
    int xend = mini((camera.x + camera.view_width)/TILE_W,  map.w-1);
    int yend = mini((camera.y + camera.view_height)/TILE_H, map.h-1);

    map_bind_tiles();
    glBegin(GL_QUADS);

    for (y = Y0; y <= yend; y++) {
	for (x = X0; x <= xend; x++) {
	    if (map.tile[y][x].shape)
		map_draw_tile(map.tile[y][x].shape, x, y);
	}
    }

    glEnd();			/* glBegin(GL_QUADS) */
}


void map_draw_tile(const int nth, const int x, const int y)
{
#define BITMAP_W	128.0
#define BITMAP_H	512.0

    texcoord2d_t coord;
    int row, col;
    assert((1 <= nth) & (nth <= NUM_TILES));

    get_tile_sprite_coord(nth, &row, &col);

    /* +/- 0.5 to texture coordinates so they join properly. */
    coord.w = TILE_W;
    coord.h = TILE_H;
    coord.top    = (row*(TILE_H) + 0.5)/BITMAP_H;
    coord.bottom = (row*(TILE_H) + TILE_H - 0.5)/BITMAP_H;
    coord.left   = (col*(TILE_W) + 0.5)/BITMAP_W;
    coord.right  = (col*(TILE_W) + TILE_W - 0.5)/BITMAP_W;

    gl_draw_sprite_2d(&coord, x*TILE_W, y*TILE_H);

#undef BITMAP_H
#undef BITMAP_W
}

/*--------------------------------------------------------------*/

void map_resize_h(const int rows)
{
    int y;
    void *p;

    p = realloc(map.tile, rows * sizeof(map.tile[0]));
    assert(p);

    map.tile = p;

    for (y = map.alloc_h; y < rows; y++) {
	map.tile[y] = calloc(map.alloc_w, sizeof(map.tile[0][0]));
	assert(map.tile[y]);
    }

    map.alloc_h = rows;
    map.h = mini(map.h, rows);
}


void map_resize_w(const int cols)
{
    int x, y;

    for (y = 0; y < map.alloc_h; y++) {
	void *p;
	assert(map.tile[y]);

	p = realloc(map.tile[y], cols * sizeof(map.tile[0][0]));
	assert(p);

	map.tile[y] = p;

	for (x = map.alloc_w; x < cols; x++)
	    map_tile_reset(&map.tile[y][x]);
    }

    map.alloc_w = cols;
    map.w = mini(map.w, cols);
}


static void map_new(void)
{
    map.w = 0;
    map.h = 0;
    map.alloc_w = 0;
    map.alloc_h = 0;

    map_resize_h(10);
    map_resize_w(10);
}


static void get_shift_bounds(const int offx, const int w,
			     int *X0, int *X1, int *dx)
{
    if (offx >= 0) {
	/* Shift right => start from right, work left. */
	*X0 = w-1;
	*X1 = offx-1;
	*dx = -1;
    }
    else {
	/* Shift left => start from left, work right. */
	*X0 = 0;
	*X1 = w+offx;
	*dx = 1;
    }
}


static void map_tile_shift(const int offx, const int offy)
{
    int x, y, X0, Y0, X1, Y1, dx, dy;

    get_shift_bounds(offx, map.w, &X0, &X1, &dx);
    get_shift_bounds(offy, map.h, &Y0, &Y1, &dy);

    for (y = Y0; y != Y1; y += dy) {
	for (x = X0; x != X1; x += dx)
	    map.tile[y][x] = map.tile[y-offy][x-offx];

	for (; (0 <= x) && (x <= map.w-1); x += dx)
	    map_tile_reset(&map.tile[y][x]);
    }

    for (; (0 <= y) && (y <= map.h-1); y += dy) {
	for (x = 0; x < map.w; x++)
	    map_tile_reset(&map.tile[y][x]);
    }
}


/* map_shift:
 *
 * Shift all tiles and objects in a map by (dx, dy) tiles.
 */
void map_shift(const int dx, const int dy, game_state_t *state)
{
    double offx = TILE_W * dx;
    double offy = TILE_H * dy;
    assert(state);

    if (client_server_mode != I_AM_CLIENT_SERVER || state->as_server) {
	map_tile_shift(dx, dy);
	container_shift(offx, offy);
	gizmo_shift(offx, offy);
	pickup_shift(offx, offy);
	start_loc_shift(offx, offy);
	candela_shift(offx, offy);
    }

    player_shift(offx, offy, state);

    if (state->as_server) {
	NLbyte buf[MAX_PACKET_SIZE];
	NLint len;

	len = packet_set_zero_encode(dx, dy, buf, sizeof(buf));
	server_broadcast(buf, len);
    }
    else if (client_server_mode != I_AM_CLIENT_SERVER) {
	zero_tx -= dx;
	zero_ty -= dy;
	zero_x  -= offx;
	zero_y  -= offy;
    }

    background_offx += offx;
    background_offy += offy;
    camera.x += offx;
    camera.y += offy;
}

/*--------------------------------------------------------------*/

static void read_normals(void)
{
    FILE *fp;
    int i = 0;

    fp = fopen("data/tiles.n", "r");
    if (!fp) {
	printf("Error opening 'data/tiles.n' for reading!\n");
	return;
    }

    while (!feof(fp)) {
	int r, t, l, b;
	fscanf(fp, "{ %d, %d, %d, %d }\n", &r, &t, &l, &b);

	tile_normal[i].right  = deg2rad(r);
	tile_normal[i].top    = deg2rad(t);
	tile_normal[i].left   = deg2rad(l);
	tile_normal[i].bottom = deg2rad(b);
	i++;
    }

    fclose(fp);
}


void map_init(void)
{
    bmp_tiles = load_bitmap("data/tiles.tga", NULL);
    assert(bmp_tiles);

    tex_tiles = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_tiles, -1);

    tile_normal = malloc(sizeof(tile_normal_t) *
			 (bmp_tiles->w/TILE_W) * (bmp_tiles->h/TILE_H));
    assert(tile_normal);
    read_normals();

    map_new();
}


void map_shutdown(void)
{
    int y;

    for (y = 0; y < map.alloc_h; y++) {
	free(map.tile[y]);
    }
    free(map.tile);
    map.tile = NULL;
    map.w = 0;
    map.h = 0;
    map.alloc_w = 0;
    map.alloc_h = 0;

    if (tile_normal) {
	free(tile_normal);
	tile_normal = NULL;
    }

    if (bmp_tiles) {
	destroy_bitmap(bmp_tiles);
	bmp_tiles = NULL;
    }
}
