/* map-save.c,
 *
 * Map loading and saving routines.
 */

#include <allegro.h>
#include <assert.h>
#include <stdio.h>
#include "candela.h"
#include "container.h"
#include "gizmo.h"
#include "map-save.h"
#include "map.h"
#include "pickup.h"
#include "start-loc.h"
#include "stats-container.h"
#include "stats-gizmo.h"
#include "stats-pickup.h"


/*--------------------------------------------------------------*/

static int map_crop_x(const int X0, const int X1, const int dx);
static int map_crop_y(const int Y0, const int Y1, const int dy);


void map_crop(game_state_t *state)
{
    int X0, Y0, X1, Y1;
    assert(state);

    Y0 = map_crop_y(0, map.h, 1);

    /* No tiles in map. */
    if (Y0 == map.h) {
        map.w = 0;
        map.h = 0;
	return;
    }

    X0 = map_crop_x(0, map.w, 1);

    if ((X0 != 0) || (Y0 != 0))
	map_shift(-X0, -Y0, state);

    X1 = map_crop_x(map.w-X0-1, -1, -1);
    Y1 = map_crop_y(map.h-Y0-1, -1, -1);
    map.w = X1+1;
    map.h = Y1+1;
}


static int map_crop_x(const int X0, const int X1, const int dx)
{
    int x, y;

    for (x = X0; x != X1; x += dx) {
	for (y = 0; y < map.h; y++) {
	    if (map.tile[y][x].shape)
		return x;
	}
    }

    assert(false);
    return X1;
}


static int map_crop_y(const int Y0, const int Y1, const int dy)
{
    int x, y;

    for (y = Y0; y != Y1; y += dy) {
	for (x = 0; x < map.w; x++) {
	    if (map.tile[y][x].shape)
		return y;
	}
    }

    return Y1;
}

/*--------------------------------------------------------------*/

static void map_load_tiles(PACKFILE *fp, game_state_t *state)
{
    int x, y, w, h;
    unsigned int tile;
    assert(state);

    w = pack_igetw(fp);
    h = pack_igetw(fp);

    map_resize_h(h);
    map_resize_w(w);

    for (y = 0; y < h; y++) {
	for (x = 0; x < w; x++) {
	    tile = pack_igetw(fp);
	    map_set_tile(x, y, tile, state);
	    map.tile[y][x].region = pack_getc(fp);
	}
    }
}


static void map_load_tiles_default(game_state_t *state)
{
#define DEFAULT_H	9
#define DEFAULT_W	9

    int x, y;
    unsigned int tile;

    map_resize_h(DEFAULT_H);
    map_resize_w(DEFAULT_W);

    for (y = 0; y < DEFAULT_H; y++) {
	for (x = 0; x < DEFAULT_W; x++) {
	    if ((x == 0) || (x == DEFAULT_W-1) ||
		(y == 0) || (y == DEFAULT_H-1))
		tile = 1;
	    else
		tile = 0;

	    map_set_tile(x, y, tile, state);
	}
    }

#undef DEFAULT_W
#undef DEFAULT_H
}


static void map_save_tiles(PACKFILE *fp, game_state_t *state)
{
    int x, y;
    assert(state);

    map_crop(state);

    pack_iputw(map.w, fp);
    pack_iputw(map.h, fp);

    for (y = 0; y < map.h; y++) {
	for (x = 0; x < map.w; x++) {
	    pack_iputw(map.tile[y][x].shape, fp);
	    pack_putc(map.tile[y][x].region, fp);
	}
    }
}

/*--------------------------------------------------------------*/

static void map_load_starts_default(void);


static void map_load_starts(PACKFILE *fp)
{
    struct packet_start_loc packet;
    unsigned int i;

    start_loc_stop();
    start_loc_start();

    i = pack_getc(fp);

    if (i == 0) {
	map_load_starts_default();
	return;
    }

    if (i >= MAX_START_LOCS) {
	fprintf(stderr,
		"[MapLoad] Too many start locations!  expect: %d  max: %d\n",
		i, MAX_START_LOCS);

	while (i > MAX_START_LOCS) {
	    i--;

	    pack_igetw(fp);	/* packet.x */
	    pack_igetw(fp);	/* packet.y */
	}

	return;
    }

    while (i > 0) {
	i--;

	packet.id = i;
	packet.x = pack_igetw(fp);
	packet.y = pack_igetw(fp);
	start_loc_set(&packet);
    }
}


static void map_load_starts_default(void)
{
    struct packet_start_loc packet;

    packet.id = 0;
    packet.x = (map.w * TILE_W) / 2;
    packet.y = ((map.h-1) * TILE_H) / 2;
    start_loc_set(&packet);
}


static void map_save_starts(PACKFILE *fp)
{
    unsigned int i;

    pack_putc(start_loc_total(), fp);

    for (i = 0; i < MAX_START_LOCS; i++) {
	if (!starts[i].used)
	    continue;

	pack_iputw(starts[i].x, fp);
	pack_iputw(starts[i].y, fp);
    }
}

/*--------------------------------------------------------------*/

static void map_load_containers(PACKFILE *fp)
{
    struct packet_container_new packet;
    unsigned int n;
    char symbol;

    container_stop();
    container_start();

    n = pack_igetw(fp);

    packet.hidden = false;
    while (n > 0) {
	packet.id = container_generate_unique_id();
	symbol = pack_getc(fp);
	packet.class = container_class_from_symbol(symbol);
	packet.x = pack_igetw(fp);
	packet.y = pack_igetw(fp);

	container_new(&packet);
	n--;
    }
}


static void map_save_containers(PACKFILE *fp)
{
    container_t *c;
    char symbol;

    pack_iputw(container_total(), fp);

    list_for_each(c, &container_list) {
	symbol = container_class_to_symbol(c->class);
	pack_putc(symbol, fp);
	pack_iputw(c->x, fp);
	pack_iputw(c->y, fp);
    }
}

/*--------------------------------------------------------------*/

static void map_load_gizmos(PACKFILE *fp)
{
    struct packet_gizmo_new packet;
    unsigned int n;
    char symbol;

    gizmo_stop();
    gizmo_start();

    n = pack_igetw(fp);

    while (n > 0) {
	packet.id = gizmo_generate_unique_id();
	symbol = pack_getc(fp);
	packet.class = gizmo_class_from_symbol(symbol);
	packet.x = pack_igetw(fp);
	packet.y = pack_igetw(fp);

	gizmo_new(&packet);
	n--;
    }
}


static void map_save_gizmos(PACKFILE *fp)
{
    gizmo_t *g;
    char symbol;

    pack_iputw(gizmo_total(), fp);

    list_for_each(g, &gizmo_list) {
	symbol = gizmo_class_to_symbol(g->class);
	pack_putc(symbol, fp);
	pack_iputw(g->x, fp);
	pack_iputw(g->y, fp);
    }
}

/*--------------------------------------------------------------*/

static void map_load_pickups(PACKFILE *fp)
{
    struct packet_pickup_new packet;
    unsigned int n;
    char symbol;

    pickup_stop();
    pickup_start();

    n = pack_igetw(fp);

    while (n > 0) {
	packet.time = 0;
	packet.id = pickup_generate_unique_id();
	symbol = pack_getc(fp);
	packet.class = pickup_class_from_symbol(symbol);
	packet.x = pack_igetw(fp);
	packet.y = pack_igetw(fp);

	pickup_new(&packet);
	n--;
    }
}


static void map_save_pickups(PACKFILE *fp)
{
    pickup_t *p;
    char symbol;

    pack_iputw(pickup_total(), fp);

    list_for_each(p, &pickup_list) {
	symbol = pickup_class_to_symbol(p->class);
	pack_putc(symbol, fp);
	pack_iputw(p->x, fp);
	pack_iputw(p->y, fp);
    }
}

/*--------------------------------------------------------------*/

static void map_load_candela(PACKFILE *fp)
{
    struct packet_candela_new packet;
    unsigned int n;
    unsigned int i;
    assert(fp);

    candela_stop();
    candela_start();

    n = pack_igetw(fp);

    while (n > 0) {
	packet.id = candela_generate_unique_id();
	packet.size = CANDELA_INCLUDE_ALL;

	for (i = 0; i < 2; i++)
	    packet.posi[i] = pack_igetw(fp);

	for (i = 0; i < 3; i++)
	    packet.ambi[i] = pack_getc(fp)/255.0;

	packet.cutoff = pack_getc(fp); 
	packet.exponent = pack_getc(fp);

	candela_new(&packet);
	n--;
   }
}


static void map_save_candela(PACKFILE *fp)
{
    candela_t *c;
    unsigned char x;
    unsigned int i;

    pack_iputw(candela_total(), fp);

    list_for_each(c, &candela_list) {
	for (i = 0; i < 2; i++)
	    pack_iputw(c->posi[i], fp);

	for (i = 0; i < 3; i++) {
	    x = MID(0, c->ambi[i]*255, 255);
	    pack_putc(x, fp);
	}

	pack_putc(c->cutoff, fp);
	pack_putc(c->exponent, fp);
    }
}

/*--------------------------------------------------------------*/

void map_load(const char *filename, game_state_t *state)
{
    PACKFILE *fp;
    assert(state);

    fp = pack_fopen(filename, F_READ_PACKED);
    if (!fp) {
	fprintf(stderr, "[MapLoad] Could not open %s for reading!\n",
		filename);

	map_load_tiles_default(state);
	map_load_starts_default();
	return;
    }

    map_load_tiles(fp, state);
    map_load_starts(fp);
    map_load_containers(fp);
    map_load_gizmos(fp);
    map_load_pickups(fp);
    map_load_candela(fp);

    pack_fclose(fp);
}


void map_save(const char *filename, game_state_t *state)
{
    PACKFILE *fp;
    assert(state);

    fp = pack_fopen(filename, F_WRITE_PACKED);
    if (!fp) {
	fprintf(stderr, "[MapSave] Could not open %s for writing!\n",
		filename);
	return;
    }

    map_save_tiles(fp, state);
    map_save_starts(fp);
    map_save_containers(fp);
    map_save_gizmos(fp);
    map_save_pickups(fp);
    map_save_candela(fp);

    pack_fclose(fp);
}
