/* hud.c,
 *
 * Draw the stats for the local player.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include "angle.h"
#include "bullet.h"
#include "camera.h"
#include "chat.h"
#include "common.h"
#include "hud.h"
#include "maxmin.h"
#include "pickup.h"
#include "player.h"
#include "score.h"
#include "server.h"
#include "stats-pickup.h"
#include "sv-internal.h"
#include "texdraw.h"
#include "weapon.h"


static BITMAP *bmp_hud;
static GLuint tex_hud;

/*--------------------------------------------------------------*/

/* weapon_to_pickup:
 *
 * Convert a weapon class to the pickup giving that weapon for drawing
 * purposes.
 */
static enum PICKUP_CLASS weapon_to_pickup(enum WEAPON_CLASS class)
{
    const enum PICKUP_CLASS pick[NUM_WEAPON_CLASSES] = {
	PICKUP_WEAPON_MISSILE_LAUNCHER,
	PICKUP_WEAPON_MICRO_MISSILE,
	PICKUP_WEAPON_GRENADE_LAUNCHER,
	PICKUP_WEAPON_BLASTER,
	PICKUP_WEAPON_SHOTGUN,
	PICKUP_WEAPON_VULCAN,
	PICKUP_WEAPON_MINIGUN
    };

    if (class >= NUM_WEAPON_CLASSES) {
	fprintf(stderr, "[Hud] Unknown weapon: %d\n", class);
	return PICKUP_WEAPON_BLASTER;
    }

    return pick[class];
}


static enum PICKUP_CLASS weapon_to_ammo_pickup(enum WEAPON_CLASS class)
{
    const enum PICKUP_CLASS pick[NUM_WEAPON_CLASSES] = {
	PICKUP_AMMO_MISSILE_LAUNCHER,
	PICKUP_AMMO_MICRO_MISSILE,
	PICKUP_AMMO_GRENADE_LAUNCHER,
	PICKUP_AMMO_BLASTER,
	PICKUP_AMMO_SHOTGUN,
	PICKUP_AMMO_VULCAN,
	PICKUP_AMMO_MINIGUN
    };

    if (class >= NUM_WEAPON_CLASSES) {
	fprintf(stderr, "[Hud] Unknown weapon: %d\n", class);
	return PICKUP_AMMO_BLASTER;
    }

    return pick[class];
}


static char weapon_to_shortcut(enum WEAPON_CLASS class)
{
    const char shortcut[NUM_WEAPON_CLASSES] = { 
	'1', '2', '3', '4', '5', '6', '7'
    };

    if (class >= NUM_WEAPON_CLASSES) {
	fprintf(stderr, "[Hud] Unknown weapon: %d\n", class);
	return '\0';
    }

    return shortcut[class];
}


static enum PICKUP_CLASS pickup_from_powerup(enum POWERUP_CLASS class)
{
    const enum PICKUP_CLASS pick[NUM_POWERUP_CLASSES] = {
	PICKUP_BERSERKER,
	PICKUP_LASER,
	PICKUP_LIGHT_AMP,
	PICKUP_STEALTH,
	PICKUP_TRACKER
    };

    if (class >= NUM_POWERUP_CLASSES) {
	fprintf(stderr, "[Hud] Unknown powerup: %d\n", class);
	return POWERUP_BERSERKER;
    }

    return pick[class];
}

/*--------------------------------------------------------------*/

static void hud_draw_dead(void);
static void hud_draw_health(void);
static void hud_draw_weapons(void);
static void hud_draw_powerups(void);
static void hud_draw_scores(void);


void hud_draw(const bool allow_respawn)
{
    if (!player)
	return;

    hud_draw_health();
    hud_draw_weapons();

    if (player->alive) {
	hud_draw_powerups();
    }
    else if (allow_respawn) {
	hud_draw_dead();
    }

    if (show_score) {
	sort_scores();
	hud_draw_scores();
    }

    glColor3fv(glWhite);
}


void hud_draw_chat(void)
{
    unsigned int c, i;
    double y = 480.0-5.0;

    for (c = 0; c < NUM_CHAT_HISTORY; c++) {
	const chat_t *message = chat_get(c);

	if (!message)
	    break;
    }

    i = maxi(0, c-5);
    c = mini(5, c);
    for (; c > 0; c--, i++) {
	const chat_t *message = chat_get(i);
	assert(message);
	assert(message->class < NUM_CHAT_CLASSES);
	allegro_gl_printf(al_font, 5.0, y, 0.0,
			  chat_colour_from_class(message->class),
			  message->str);
	y -= 15.0;
    }

    if (input_enabled) {
	allegro_gl_printf(al_font, 5.0, y-5.0, 0.0,
			  chat_colour_from_class(CHAT_GAME_MESSAGE),
			  "%s_", input_line);
    }

    glColor3fv(glWhite);
}


static void hud_draw_dead(void)
{
    const char *str = "Press SPACE to Respawn";
    int len;

    len = text_length(al_font, str);

    glColor3fv(glWhite);
    allegro_gl_printf_ex(al_font, 320.0 - len/2.0, 280.0, 0.0, str);
}


static void hud_draw_health(void)
{
    const texcoord2d_t coord = { 24.0, 24.0, 0.0, 24.0/32.0, 0.0, 24.0/32.0 };
    
    glBindTexture(GL_TEXTURE_2D, tex_hud);
    glBegin(GL_QUADS);
    gl_draw_sprite_2d(&coord, 5.0, 5.0);
    glEnd();                   /* glBegin(GL_QUADS) */

    glColor3fv(glWhite);
    allegro_gl_printf_ex(al_font, 30.0, 21.0, 0.0, "%d", player->health);
    allegro_gl_printf_ex(al_font, 150.0, 21.0, 0.0, "Kills: %d",
			 player->kills_this_life);
}


static void hud_draw_weapons(void)
{
    enum WEAPON_CLASS w;
    int c, x, y;

    pickup_bind_texture();
    glBegin(GL_QUADS);

    for (w = WEAPON_FIRST, y = 0; w < NUM_WEAPON_CLASSES; w++) {
	if (!player->have_weapon[w])
	    continue;

	if (w == player->weapon) {
	    glColor3fv(glWhite);
	    x = 640-60-10;

	    pickup_draw_unit(weapon_to_ammo_pickup(w), 80.0, 5.0, true);
	}
	else {
	    const GLfloat col[] = { 0.4, 0.4, 0.4 };
	    glColor3fv(col);
	    x = 640-60;
	}

	pickup_draw_unit(weapon_to_pickup(w), x, y, true);

	y += 32;
    }

    glEnd();			/* glBegin(GL_QUADS) */

    /* Don't put allegro_gl_printf in the glBegin/glEnd block. */
    c = makecol(0xff, 0x80, 0x80);
    for (w = WEAPON_FIRST, y = 0; w < NUM_WEAPON_CLASSES; w++) {
	if (!player->have_weapon[w])
	    continue;

	allegro_gl_printf(al_font, 630.0, y+10, 0.0, c, "%c",
			  weapon_to_shortcut(w));

	if (w != WEAPON_BLASTER) {
	    allegro_gl_printf(al_font, 535.0, y+10.0, 0.0, c, "%d",
			      player->weapon_ammo[w]);
	}

	y += 32;
    }

    /* Write the current weapon's ammo in the status bar. */
    if (player->weapon != WEAPON_BLASTER) {
 	allegro_gl_printf(al_font, 100.0, 21.0, 0.0, makecol(0xff, 0xff, 0xff),
 			  "%d", player->weapon_ammo[player->weapon]);
    }
    else {
	glColor3fv(glWhite);
    }
}


static void hud_draw_powerups(void)
{
    enum POWERUP_CLASS p;
    int w, x;

    w = 0;
    for (p = POWERUP_FIRST; p < NUM_POWERUP_CLASSES; p++) {
	if (player->powerup_duration[p] >= server_time)
	    w += pickup_sprite[pickup_from_powerup(p)].w;
    }

    if (w <= 0)
	return;

    pickup_bind_texture();
    glBegin(GL_QUADS);
    x = 320.0 - w/2.0;
    for (p = POWERUP_FIRST; p < NUM_POWERUP_CLASSES; p++) {
	if (player->powerup_duration[p] >= server_time) {
	    enum PICKUP_CLASS c = pickup_from_powerup(p);

	    pickup_draw_unit(c, x, 8, true);
	    x += pickup_sprite[c].w;
	}
    }
    glEnd();			/* glBegin(GL_QUADS); */
}

/*--------------------------------------------------------------*/

static void hud_draw_tracker(const game_state_t *state);
static void hud_draw_laser(void);
static void hud_draw_laser2(void);


void hud_draw_special(const game_state_t *state)
{
    assert(state);

    if (!player || !(player->alive))
	return;

    if ((player->powerup_duration[POWERUP_TRACKER] < server_time) &&
	(player->powerup_duration[POWERUP_LASER] < server_time))
	return;

    glDisable(GL_ALPHA_TEST);
    glDisable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);
    glBlendEquation(GL_FUNC_ADD);

    if (player->powerup_duration[POWERUP_TRACKER] >= server_time) {
	hud_draw_tracker(state);
    }

    if (player->powerup_duration[POWERUP_LASER] >= server_time) {
	if (player->weapon != WEAPON_GRENADE_LAUNCHER)
	    hud_draw_laser();
	else
	    hud_draw_laser2();
    }

    glDisable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_ALPHA_TEST);
}


static void hud_draw_tracker(const game_state_t *state)
{
    const GLfloat arrow[4][2] = {
	{ 25.0, 3.0 }, { 25.0, -3.0 }, { 150.0, -1.0 }, { 150.0, 1.0 }
    };
    GLfloat tracker_col[] = { 0.25, 1.0, 0.25, 0.25 };
    double dd, dx, dy, angle;
    unsigned int i;
    assert(state);

    if (player->packet_size & PACKET_INCLUDE_TRACKER) {
	angle = player->tracker_angle;
    }
    else {
	if (!player_tracker_angle(player, &dx, &dy, &angle, state))
	    return;

	dd = dx*dx + dy*dy;

	if (dd < 10.0*10.0)
	    return;

	if (dd < 100.0*100.0)
	    tracker_col[3] = 0.25 * dd/(100.0*100.0);
    }

    glPushMatrix();
    glTranslated(player->x, player->y+19.0, 0.0);
    glColor4fv(tracker_col);

    /* Draw a circle */
    glBegin(GL_QUAD_STRIP);
    for (i = 0; i <= 32; i++) {
	double cosx = cos(2*M_PI*i/32);
	double sinx = sin(2*M_PI*i/32);

	glVertex2d( 98.0*cosx,  98.0*sinx);
	glVertex2d(100.0*cosx, 100.0*sinx);
    }
    glEnd();			/* glBegin(GL_QUAD_STRIP) */

    /* Draw the arrow. */
    glRotated(rad2deg(angle), 0.0, 0.0, 1.0);

    glBegin(GL_QUADS);
    for (i = 0; i < 4; i++)
	glVertex2fv(arrow[i]);
    glEnd();		/* glBegin(GL_QUADS) */

    glPopMatrix();
}


static void hud_draw_laser(void)
{
    GLfloat laser_col[] = { 1.0, 0.25, 0.25, 0.75 };
    unsigned int i, j;
    double rx, ry;

    weapon_barrel_displacement(player->weapon, player->mirror, player->angle,
			       &rx, &ry);

    glTranslated(player->x+rx, player->y+ry+19.0, 0.0);
    glRotated(rad2deg(player->angle), 0.0, 0.0, 1.0);
    glBegin(GL_LINES);

    for (i = 30, j = 10; i < 300; i += 5, j++) {
	if (j >= 10) {
	    j = 0;
	    laser_col[3] = 0.75 * (300-i)/300;
	    glColor4fv(laser_col);
	}

	glVertex2i(i, 0);
    }

    glEnd();			/* glBegin(GL_LINES) */
}


static void hud_draw_laser2(void)
{
    GLfloat laser_col[] = { 1.0, 0.25, 0.25, 0.75 };
    bullet_t dummy;
    unsigned int j;
    int t;

    dummy.class = player->weapon;
    dummy._spawn_time = 0;
    dummy._spawn_x = player->x;
    dummy._spawn_y = player->y+19.0;
    dummy._spawn_vx = weapon[dummy.class].speed*cos(player->angle);
    dummy._spawn_vy = weapon[dummy.class].speed*sin(player->angle);

    glBegin(GL_LINES);

    for (t = 50, j = 10; t < 1500; t += 25, j++) {
	if (j >= 10) {
	    j = 0;
	    laser_col[3] = 0.75 * (1000-t)/1000;
	    glColor4fv(laser_col);
	}

	bullet_move(&dummy, t);
	glVertex2i(dummy.x, dummy.y);
    }

    glEnd();			/* glBegin(GL_LINES) */
}


static void hud_draw_scores(void)
{
    int i, y;
    int col = makecol(0xff, 0xff, 0xff);

    for (i = 0, y = 360; i < MAX_CLIENTS; i++) {
	if (client_data[i].type != CLIENT_GAMER &&
	    client_data[i].type != CLIENT_ROBOT)
	    continue;

	allegro_gl_printf(al_font, 240.0, y, 0.0, col, "%s",
			  client_data[i].name);
	allegro_gl_printf(al_font, 320.0, y, 0.0, col, ": %d (%d)",
			  client_data[i].sc.total_score,
			  client_data[i].sc.session_score);

	y -= 15;
    }
}

/*--------------------------------------------------------------*/

void hud_init(void)
{
    bmp_hud = load_bitmap("data/hud.tga", NULL);
    assert(bmp_hud);
    
    tex_hud = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED, bmp_hud, -1);
}


void hud_shutdown(void)
{
    if (bmp_hud) {
	destroy_bitmap(bmp_hud);
	bmp_hud = NULL;
    }
}
