/* gizmo.c,
 *
 * Gizmos are special tile-like things, such as ladders and springs.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <stdio.h>

#include "camera.h"
#include "common.h"
#include "gizmo.h"
#include "input.h"
#include "map.h"
#include "list.h"
#include "sprite.h"
#include "stats-gizmo.h"
#include "texdraw.h"


enum GIZMO_COLLISION {
    GIZMO_PLAYER_NONE,
    GIZMO_PLAYER_INTERSECT,
    GIZMO_PLAYER_SUPPORT
};


static BITMAP *bmp_gizmo;
static GLuint tex_gizmo;

list_head_t gizmo_list;		/* Shared */
static int gizmo_count;

/*--------------------------------------------------------------*/

bool gizmo_in_tile(const enum GIZMO_CLASS g, const int x, const int y)
{
    const sprite_t *spr;
    int xx, yy, X0, Y0;
    assert(g < NUM_GIZMO_CLASSES);

    spr = &gizmo_sprite[g];

    /* Use these values to take care of round off errors. */
    X0 = x - spr->w/4;
    Y0 = y;

    for (xx = 0; xx < spr->w/2; xx++) {
	if (!x_in_map(X0+xx))
	    continue;

	if ((y_in_map(Y0) && map_occupies(X0+xx, Y0)) ||
	    (y_in_map(Y0+spr->h/2-1) && map_occupies(X0+xx, Y0+spr->h/2-1)))
	    return true;
    }

    for (yy = 0; yy < spr->h/2; yy++) {
	if (!y_in_map(Y0+yy))
	    continue;

	if ((x_in_map(X0) && map_occupies(X0, Y0+yy)) ||
	    (x_in_map(X0+spr->w/2-1) && map_occupies(X0+spr->w/2-1, Y0+yy)))
	    return true;
    }

    return false;
}

/*--------------------------------------------------------------*/

static void ladder_intersecting_player(player_t *p)
{
    p->vx *= 0.80;
    p->vy *= 0.75;
    p->movement_mode = MOVEMENT_IN_LADDER;
    p->can_jump = false;
}


static void ladder_supporting_player(player_t *p)
{
    p->vx *= 0.80;
    p->vy = 0.0;
    p->movement_mode = MOVEMENT_ABOVE_LADDER;
    p->can_jump = true;
    p->jump_frame = MAX_JUMP_FRAMES;
}


static void spring_intersecting_player(player_t *p)
{
    if (p->vy > 1.5)
	return;

    p->vx *= 0.80;
    p->vy = 1.0;
    p->movement_mode = MOVEMENT_NORMAL;
    p->can_jump = false;
}


static void spring_supporting_player(player_t *p)
{
    p->vx *= 0.80;

    if (p->hold_keys & INPUT_MOVE_DOWN)
	p->vy = 4.5;
    else if (p->last_impy & INPUT_MOVE_UP)
	p->vy = 8.3;
    else
	p->vy = 7.0;

    p->movement_mode = MOVEMENT_NORMAL;
    p->can_jump = false;
    p->hold_keys &=~INPUT_MOVE_UP;
    p->jump_frame = 0;
}

/*--------------------------------------------------------------*/

typedef void (*gizmo_intersecting_func_t)(player_t *p);
typedef void (*gizmo_supporting_func_t)(player_t *p);

static gizmo_intersecting_func_t intersect_func(enum GIZMO_CLASS class)
{
    if (class <= GIZMO_LADDERL5)
	return ladder_intersecting_player;
    if (class == GIZMO_SPRING)
	return spring_intersecting_player;

    return NULL;
}


static gizmo_supporting_func_t support_func(enum GIZMO_CLASS class)
{
    if (class <= GIZMO_LADDERL5)
	return ladder_supporting_player;
    if (class == GIZMO_SPRING)
	return spring_supporting_player;

    return NULL;
}

/*--------------------------------------------------------------*/

static enum GIZMO_COLLISION
gizmo_player_interaction(const gizmo_t *g, const double px, const double py)
{
    double gizmo_top;

    gizmo_top = g->y + g->h - 1.0;

    if (!((g->x-g->w/2.0 <= px+PLAYER_W_2+2.0) &&
	  (g->x+g->w/2.0 >= px-PLAYER_W_2-2.0)))
	return GIZMO_PLAYER_NONE;

    /* Check player-gizmo intersection. */
    if ((g->y <= py+PLAYER_H-1.0) &&
	(py <= gizmo_top))
	return GIZMO_PLAYER_INTERSECT;

    /* Check under player's foot for support. */
    if (ABS(gizmo_top - (py-1.0)) < 1.0)
	return GIZMO_PLAYER_SUPPORT;

    return GIZMO_PLAYER_NONE;
}


bool gizmo_update_with_player(player_t *p)
{
    gizmo_t *g;
    gizmo_intersecting_func_t intersect = NULL;
    gizmo_supporting_func_t support = NULL;

    list_for_each(g, &gizmo_list) {
	switch (gizmo_player_interaction(g, p->x, p->y)) {
	  case GIZMO_PLAYER_NONE:
	      break;
	  case GIZMO_PLAYER_INTERSECT:
	      intersect = intersect_func(g->class);
	      break;
	  case GIZMO_PLAYER_SUPPORT:
	      support = support_func(g->class);
	      break;
	}
    }

    if (intersect) {
	intersect(p);
	return true;
    }
    else if (support) {
	support(p);
	return true;
    }
    else
	return false;
}


gizmo_t *player_foot_in_gizmo(const double px, const double py)
{
    gizmo_t *g, *ret = NULL;

    list_for_each(g, &gizmo_list) {
	switch (gizmo_player_interaction(g, px, py)) {
	  case GIZMO_PLAYER_NONE:
	      break;
	  case GIZMO_PLAYER_INTERSECT:
	      return NULL;
	  case GIZMO_PLAYER_SUPPORT:
	      ret = g;
	      break;
	}
    }

    return ret;
}

/*--------------------------------------------------------------*/

void gizmo_bind_texture(void)
{
    glBindTexture(GL_TEXTURE_2D, tex_gizmo);
}


void gizmo_draw(const bool editor_enabled)
{
    const GLfloat *last = glWhite;
    gizmo_t *g;

    if (list_empty(gizmo_list))
	return;

    gizmo_bind_texture();
    glBegin(GL_QUADS);

    list_for_each(g, &gizmo_list) {
	if (editor_enabled)
	    sprite_tint_col(g->highlight, false, &last);

	gizmo_draw_unit(g->class, g->x, g->y, true);
    }

    if (last != glWhite)
	glColor3fv(glWhite);

    glEnd();			/* glBegin(GL_QUADS) */
}


void gizmo_draw_unit(const enum GIZMO_CLASS class, const int x, const int y,
		     const bool for_game)
{
    const sprite_t *sprite;
    texcoord2d_t coord;
    assert(class < NUM_GIZMO_CLASSES);

    sprite = &gizmo_sprite[class];

    if (for_game && !camera_can_see2(&camera, x, y,
				     sprite->w/2.0, sprite->h/2.0))
	return;

    texcoord_from_sprite(&coord, sprite, bmp_gizmo->w, bmp_gizmo->h);

    gl_draw_sprite_2d(&coord, x-sprite->w/4.0, y);
}

/*--------------------------------------------------------------*/

gizmo_t *gizmo_from_id(const gizmo_id id)
{
    gizmo_t *g;

    list_for_each(g, &gizmo_list) {
	if (g->id == id)
	    return g;
    }

    return NULL;
}


gizmo_id gizmo_bounding(const int x, const int y)
{
    gizmo_t *g;
    
    list_for_each(g, &gizmo_list) {
	if ((g->x-g->w/2.0 <= x) && (x <= g->x+g->w/2.0) &&
	    (g->y <= y) && (y < g->y+g->h))
	    return g->id;
    }

    return 0;
}


enum GIZMO_CLASS gizmo_select(const int n)
{
    return MID(0, n, NUM_GIZMO_CLASSES-1);
}

/*--------------------------------------------------------------*/

gizmo_id gizmo_generate_unique_id(void)
{
    static gizmo_id id;
    id++;
    return id;
}


int gizmo_total(void)
{
    return gizmo_count;
}


void gizmo_new(const struct packet_gizmo_new *packet)
{
    gizmo_t *g = malloc(sizeof(*g));
    assert(g);
    assert(packet);
    assert(packet->class < NUM_GIZMO_CLASSES);

    g->id = packet->id;
    g->class = packet->class;
    g->x = packet->x;
    g->y = packet->y;
    g->w = gizmo_sprite[g->class].w/2;
    g->h = gizmo_sprite[g->class].h/2;
    g->highlight = false;

    list_add(gizmo_list, g);
    gizmo_count++;
}


void gizmo_move(const struct packet_gizmo_mov *packet)
{
    gizmo_t *g;
    assert(packet);

    g = gizmo_from_id(packet->id);
    if (!g) {
	fprintf(stderr, "[Gizmo] Gizmo %d not found!  Move failed\n",
		packet->id);
	return;
    }

    g->x = packet->x;
    g->y = packet->y;
}


void gizmo_shift(const double offx, const double offy)
{
    gizmo_t *g;

    list_for_each(g, &gizmo_list) {
	g->x += offx;
	g->y += offy;
    }
}


static void gizmo_free(gizmo_t *g)
{
    free(g);
    gizmo_count--;
}


void gizmo_delete(const struct packet_gizmo_del *packet)
{
    gizmo_t *g;
    assert(packet);

    g = gizmo_from_id(packet->id);
    if (!g) {
	fprintf(stderr, "[Gizmo] Gizmo %d not found!  Delete failed\n",
		packet->id);
	return;
    }

    list_remove(g);
    gizmo_free(g);
}

/*--------------------------------------------------------------*/

void gizmo_start(void)
{
    list_init(gizmo_list);
    gizmo_count = 0;
}


void gizmo_stop(void)
{
    list_free(gizmo_list, gizmo_free);
    assert(gizmo_count == 0);
}


void gizmo_init(void)
{
    bmp_gizmo = load_bitmap("data/gizmo.tga", NULL);
    assert(bmp_gizmo);

    tex_gizmo = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					    bmp_gizmo, -1);
}


void gizmo_shutdown(void)
{
    if (bmp_gizmo) {
	destroy_bitmap(bmp_gizmo);
	bmp_gizmo = NULL;
    }
}
