/* gettime.c,
 */

#include <assert.h>
#ifndef __unix__
# include <windows.h>
#endif

#include "gettime.h"


#ifdef __unix__

void tv_get_time_of_day(struct timeval *tv)
{
    gettimeofday(tv, 0);
}


/* Destructively increase time tv by a duration (usec). */
void tv_increase_time(struct timeval *tv, const int duration)
{
    tv->tv_usec += duration;
}


/* Time difference in usec. */
int tv_time_between(const struct timeval *a, const struct timeval *b)
{
    return 1000000*(b->tv_sec - a->tv_sec) + (b->tv_usec - a->tv_usec);
}

#else  /* !__unix__ */

static __int64 timer_frequency;		/* counts per second */


void win_gettime_start(void)
{
    QueryPerformanceFrequency((LARGE_INTEGER *)&timer_frequency);
}


void win_get_time_of_day(__int64 *tv)
{
    QueryPerformanceCounter((LARGE_INTEGER *)tv);
}


void win_increase_time(__int64 *tv, const int duration)
{
    *tv += timer_frequency * duration / 1000000;
}


int win_time_between(const __int64 *a, const __int64 *b)
{
    return ((*b)-(*a)) * 1000000 / timer_frequency;
}

#endif /* !__unix__ */
