/* explosion.c,
 *
 * This is for the circular explosions which can hurt you, and not the
 * animated sprites.
 */

#include <alleggl.h>
#include <assert.h>
#include <math.h>
#include "common.h"
#include "container.h"
#include "explosion.h"
#include "list.h"
#include "maxmin.h"
#include "network.h"
#include "player.h"
#include "server.h"
#include "sound.h"


#define SQ(x)	((x)*(x))


list_head_t explosion_delete_list;	/* Server only */

/*--------------------------------------------------------------*/

static void explosion_expand(explosion_t *e, game_state_t *state);
static int explosion_damage(const explosion_t *e, const double r_sq);
static void explosion_container_collision(explosion_t *e, game_state_t *state);
static void explosion_hurt_container(explosion_t *e, container_t *c,
				     const double r_sq, game_state_t *state);
static void explosion_player_collision(explosion_t *e, game_state_t *state);
static void explosion_hurt_player(explosion_t *e, player_t *p,
				  const double r_sq, const double angle,
				  game_state_t *state);


void explosion_update(game_state_t *state)
{
    explosion_t *e, *nx;
    assert(state);
    
    list_for_each_next(e, nx, &state->explosion_list) {
	if (e->r < e->_spawn_max_r) {
	    explosion_expand(e, state);

	    if (e->lux) {
		e->lux->r = e->r;
	    }

	    continue;
	}

	if (state->as_server) {
	    list_remove(e);
	    list_add(explosion_delete_list, e);
	}
    }
}


static void explosion_expand(explosion_t *e, game_state_t *state)
{
    assert(e);
    assert(state);

    e->time = server_time;
    e->r = e->_spawn_r * pow(1.008, server_time-e->_spawn_time);
    e->r = mind(e->r, e->_spawn_max_r);

    if (state->as_server) {
	explosion_container_collision(e, state);
	explosion_player_collision(e, state);
    }
}


static void explosion_container_collision(explosion_t *e, game_state_t *state)
{
    container_t *c;
    assert(state);
    assert(state->as_server);

    list_for_each(c, &container_list) {
	double dx, dy, r_sq;

	if (c->hidden)
	    continue;

	dx = c->x - e->_spawn_x;
	dy = c->y - e->_spawn_y;
	r_sq = SQ(dx) + SQ(dy);

	if (r_sq <= SQ(e->r))
	    explosion_hurt_container(e, c, r_sq, state);
    }
}


static void explosion_hurt_container(explosion_t *e, container_t *c,
				     const double r_sq, game_state_t *state)
{
    int damage;
    assert(state);
    assert(state->as_server);

    damage = explosion_damage(e, r_sq);
    container_hurt(c->id, damage, e->parent, e->time, state);
}


static void explosion_player_collision(explosion_t *e, game_state_t *state)
{
    player_t *p;
    assert(state);
    assert(state->as_server);

    list_for_each(p, &state->player_list) {
	double dx, dy, r_sq;

	if (!p->alive)
	    continue;

	dx = p->x - e->_spawn_x;
	dy = (p->y+19.0) - e->_spawn_y;
	r_sq = SQ(dx) + SQ(dy);

	if (r_sq <= SQ(e->r))
	    explosion_hurt_player(e, p, r_sq, atan2(dy, dx), state);
    }
}


static void explosion_hurt_player(explosion_t *e, player_t *p,
				  const double r_sq, const double angle,
				  game_state_t *state)
{
    double ax, ay;
    int i, damage;
    assert(state);
    assert(state->as_server);

    damage = explosion_damage(e, r_sq);

    if (server_time >= p->next_bleed_time) {
	p->packet_size |= PACKET_INCLUDE_BLEED;
	p->next_bleed_time = server_time + 50;
    }

    ax = mind(damage, 50.0) * 0.1 * cos(angle);
    ay = mind(damage, 50.0) * 0.1 * sin(angle);
    if (ABS(p->vx) < ABS(ax))
	p->vx = MID(-ABS(ax), p->vx+ax, ABS(ax));
    if (ABS(p->vy) < ABS(ay))
	p->vy = MID(-ABS(ay), p->vy+ay, ABS(ay));

    for (i = 0; i < MAX_EXPLOSION_HURTEES; i++) {
	/* We've already hurt this player. */
	if (e->hurtee[i] == p->id)
	    return;

	if (e->hurtee[i] == 0) {
	    e->hurtee[i] = p->id;
	    player_hurt(p, damage, 0.0, 0.0, e->parent, state);
	    p->packet_size |= PACKET_INCLUDE_BLEED;
	    p->next_bleed_time = server_time + 50;
	    return;
	}
    }
}


static int explosion_damage(const explosion_t *e, const double r_sq)
{
    /* Damage works as an inverse square law with the following
     * conditions:
     *
     * At r <= 1.0, receive 100% damage.
     * At r  = max, receive   5% damage.
     */

    double rr = MID(0.05, 1.0 - 0.95*r_sq/SQ(e->_spawn_max_r), 1.0);
    return (e->damage * e->modifier * rr);
}

/*--------------------------------------------------------------*/

explosion_t *explosion_from_id(const explosion_id id, game_state_t *state)
{
    explosion_t *e;
    assert(state);

    list_for_each(e, &state->explosion_list) {
	if (e->id == id)
	    return e;
    }

    return NULL;
}

/*--------------------------------------------------------------*/
     
explosion_id explosion_generate_unique_id(void)
{
    static explosion_id id;
    id++;
    return id;
}


void explosion_new(const struct packet_explosion_new *packet,
		   const player_id parent, game_state_t *state)
{
    const GLfloat col[] = { 1.0, 1.0, 1.0, 0.25 };
    explosion_t *e = malloc(sizeof(*e));
    int i;
    assert(e);
    assert(packet);
    assert(state);

    e->id = packet->id;
    e->parent = parent;
    e->damage = packet->damage;
    e->modifier = packet->modifier;

    /* Spawn state. */
    e->_spawn_time = packet->time;
    e->_spawn_x = packet->x;
    e->_spawn_y = packet->y;
    e->_spawn_r = 10.0;
    e->_spawn_max_r = packet->max_r;

    /* Current state. */
    e->time = e->_spawn_time;
    e->r = e->_spawn_r;

    if (state->as_server) {
	e->lux = NULL;
    }
    else {
	e->lux = lux_new(e->_spawn_x, e->_spawn_y, 200, col);
    }

    for (i = 0; i < MAX_EXPLOSION_HURTEES; i++)
	e->hurtee[i] = 0;

    list_add(state->explosion_list, e);
}


void explosion_free(explosion_t *e)
{
    assert(e);

    if (e->lux) {
	lux_automate(e->lux);
	e->lux = NULL;
    }

    list_remove(e);
    free(e);
}

/*--------------------------------------------------------------*/

void explosion_start(game_state_t *state)
{
    assert(state);

    list_init(state->explosion_list);

    if (state->as_server)
	list_init(explosion_delete_list);
}


void explosion_stop(game_state_t *state)
{
    assert(state);

    list_free(state->explosion_list, explosion_free);

    if (state->as_server)
	list_free(explosion_delete_list, explosion_free);
}


void explosion_init(void)
{
    explosion_sound_load();
}


void explosion_shutdown(void)
{
    explosion_sound_free();
}
