/* coro.c,
 *
 * GNU Pth or Windows Fibres. 
 */

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "coro.h"


#define STACK_SIZE	(256*1024)

/*--------------------------------------------------------------*/

#ifdef __unix__

/* GNU Pth. */

bool coro_pth_init(void)
{
    if (pth_init()) {
	return true;
    }
    else {
	fprintf(stderr,
		"[Coro] Error initialising PTH!\n"
		"         Try running server and client separately.\n");
	return false;
    }
}


pth_t coro_pth_self(void)
{
    return pth_self();
}


typedef struct {
    void (*entry)(void *);
    void *arg;
} spawn2_t;


static void *coro_pth_spawn2(void *arg2)
{
    spawn2_t *spawn = arg2;
    assert(spawn);

    spawn->entry(spawn->arg);
    free(spawn);
    return NULL;
}


pth_t coro_pth_spawn(void __stdcall (*entry)(void *), void *arg)
{
    pth_attr_t attr;
    spawn2_t *spawn = malloc(sizeof(spawn2_t));
    pth_t tid;

    spawn->entry = entry;
    spawn->arg = arg;

    attr = pth_attr_new();
    pth_attr_set(attr, PTH_ATTR_JOINABLE, TRUE);
    pth_attr_set(attr, PTH_ATTR_STACK_SIZE, STACK_SIZE);
    tid = pth_spawn(attr, coro_pth_spawn2, spawn);
    pth_attr_destroy(attr);

    return tid;
}


bool coro_pth_yield(pth_t tid)
{
    return pth_yield(tid);
}


int coro_pth_join(pth_t tid)
{
    return pth_join(tid, NULL);
}

#else  /* !__unix__ */

/* Windows Fibres. */

bool coro_fibre_init(void)
{
    ConvertThreadToFiber(NULL);
    return true;
}


LPVOID coro_fibre_self(void)
{
    return GetCurrentFiber();
}


LPVOID coro_fibre_spawn(void __stdcall (*entry)(LPVOID), LPVOID arg)
{
    return CreateFiber(STACK_SIZE, entry, arg);
}


bool coro_fibre_yield(LPVOID fiber)
{
    SwitchToFiber(fiber);
    return true;
}


int coro_fibre_join(LPVOID fiber)
{
    WaitForSingleObject(fiber, INFINITE);
    DeleteFiber(fiber);
    return true;
}

#endif /* !__unix__ */
