/* container.c,
 *
 * Containers are things like crates.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include <stdio.h>
#include "camera.h"
#include "common.h"
#include "container.h"
#include "explosion.h"
#include "list.h"
#include "map.h"
#include "network.h"
#include "player.h"
#include "server.h"
#include "stats-container.h"
#include "texdraw.h"


static BITMAP *bmp_container;
static GLuint tex_container;

list_head_t container_list;
static int container_count;

/*--------------------------------------------------------------*/

bool container_in_tile(const enum CONTAINER_CLASS c, const int x, const int y)
{
    const sprite_t *spr;
    int xx, yy, X0, Y0;
    assert(c < NUM_CONTAINER_CLASSES);

    spr = &container_sprite[c];

    /* Use these values to take care of round off errors. */
    X0 = x - spr->w/4;
    Y0 = y;

    for (xx = 0; xx < spr->w/2; xx++) {
	if (!x_in_map(X0+xx))
	    continue;

	if ((y_in_map(Y0) && map_occupies(X0+xx, Y0)) ||
	    (y_in_map(Y0+spr->h/2-1) && map_occupies(X0+xx, Y0+spr->h/2-1)))
	    return true;
    }

    for (yy = 0; yy < spr->h/2; yy++) {
	if (!y_in_map(Y0+yy))
	    continue;

	if ((x_in_map(X0) && map_occupies(X0, Y0+yy)) ||
	    (x_in_map(X0+spr->w/2-1) && map_occupies(X0+spr->w/2-1, Y0+yy)))
	    return true;
    }

    return false;
}

/*--------------------------------------------------------------*/

static void container_maybe_explode(const container_t *c, 
				    const player_id parent, const int tick,
				    game_state_t *state);


void container_hurt(const container_id id, const int damage,
		    const player_id hurter, const int tick,
		    game_state_t *state)
{
    container_t *c;
    assert(state);

    c = container_from_id(id);
    if (!c) {
	assert(false);
    }

    c->health -= damage;
    if (c->health < 0) {
	struct packet_container_hide packet;
	NLbyte buf[MAX_PACKET_SIZE];
	NLint len;
	assert(c->class < NUM_CONTAINER_CLASSES);

	packet.time = tick;
	packet.id = c->id;

	len = packet_container_hide_encode(&packet, buf, sizeof(buf));
	server_broadcast(buf, len);

	c->hidden = true;
	c->health = container_stats[c->class].health;
	c->respawn_time = tick + container_stats[c->class].respawn_time;
	container_maybe_explode(c, hurter, tick, state);
    }
}


static void container_maybe_explode(const container_t *c, 
				    const player_id parent, const int tick,
				    game_state_t *state)
{
    struct packet_explosion_new packet;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(state);

    if (!container_is_explosive(c->class))
	return;

    packet.time = tick;
    packet.id = container_generate_unique_id();
    packet.x = c->x;
    packet.y = c->y;
    packet.max_r = container_stats[c->class].explosion.size;
    packet.damage = container_stats[c->class].explosion.damage;
    packet.modifier = 1.0;

    len = packet_explosion_new_encode(&packet, buf, sizeof(buf));

    server_broadcast(buf, len);

    explosion_new(&packet, parent, state);
}

/*--------------------------------------------------------------*/

static void container_maybe_show(container_t *c, const game_state_t *state);


void container_update(game_state_t *state)
{
    container_t *c;
    assert(state);

    list_for_each(c, &container_list) {
	if ((c->hidden) &&
	    (c->respawn_time <= server_time))
	    container_maybe_show(c, state);
    }
}


static void container_maybe_show(container_t *c, const game_state_t *state)
{
    const struct packet_container_show packet = {
	c->id
    };
    const sprite_t *sprite;
    NLbyte buf[MAX_PACKET_SIZE];
    NLint len;
    assert(c->class < NUM_CONTAINER_CLASSES);
    assert(state);

    sprite = &container_sprite[c->class];

    if (player_touching_box(c->x, c->y, sprite->w/2, sprite->h/2, state)) {
	c->respawn_time += 2000;	/* 2 seconds. */
	return;
    }

    len = packet_container_show_encode(&packet, buf, sizeof(buf));
    server_broadcast(buf, len);

    c->hidden = false;
    c->respawn_time = 0;
}

/*--------------------------------------------------------------*/

void container_bind_texture(void)
{
    glBindTexture(GL_TEXTURE_2D, tex_container);
}


void container_draw(const bool editor_enabled)
{
    const GLfloat *last = glWhite;
    container_t *c;

    container_bind_texture();
    glBegin(GL_QUADS);

    list_for_each(c, &container_list) {
	if (editor_enabled)
	    sprite_tint_col(c->highlight, c->hidden, &last);
	else if (c->hidden)
	    continue;

	container_draw_unit(c->class, c->x, c->y, true);

	if (c->hidden)
	    glColor3fv(glWhite);
    }

    if (last != glWhite)
	glColor3fv(glWhite);

    glEnd();			/* glBegin(GL_QUADS) */
}


void container_draw_unit(enum CONTAINER_CLASS class, const int x, const int y,
			 const bool for_game)
{
    const sprite_t *sprite;
    texcoord2d_t coord;
    assert(class < NUM_CONTAINER_CLASSES);

    sprite = &container_sprite[class];

    if (for_game &&
	!camera_can_see2(&camera, x, y, sprite->w/2.0, sprite->h/2.0))
	return;

    texcoord_from_sprite(&coord, sprite, bmp_container->w, bmp_container->h);
    gl_draw_sprite_2d(&coord, x-sprite->w/4.0, y);
}

/*--------------------------------------------------------------*/

container_t *container_from_id(container_id id)
{
    container_t *c;

    list_for_each(c, &container_list) {
	if (c->id == id)
	    return c;
    }

    return NULL;
}


container_id container_bounding(const int x, const int y,
				const bool skip_hidden)
{
    const sprite_t *sprite;
    container_t *c;

    list_for_each(c, &container_list) {
	if (skip_hidden && c->hidden)
	    continue;

	sprite = &container_sprite[c->class];

	if ((c->x-sprite->w/4.0 <= x) && (x <= c->x+sprite->w/4.0) &&
	    (c->y <= y) && (y < c->y+sprite->h/2.0))
	    return c->id;
    }

    return 0;
}


enum CONTAINER_CLASS container_select(const int n)
{
    return MID(CONTAINER_BARREL, n, NUM_CONTAINER_CLASSES-1);
}

/*--------------------------------------------------------------*/

container_id container_generate_unique_id(void)
{
    static container_id id;
    id++;
    return id;
}


int container_total(void)
{
    return container_count;
}


void container_new(const struct packet_container_new *packet)
{
    container_t *c = malloc(sizeof(*c));
    assert(c);
    assert(packet);
    assert(packet->class < NUM_CONTAINER_CLASSES);

    c->id = packet->id;
    c->class = packet->class;
    c->health = container_stats[c->class].health;
    c->x = packet->x;
    c->y = packet->y;
    c->respawn_time = 0;
    c->hidden = packet->hidden;
    c->highlight = false;

    list_add(container_list, c);
    container_count++;
}


void container_move(const struct packet_container_mov *packet)
{
    container_t *c;
    assert(packet);

    c = container_from_id(packet->id);
    if (!c) {
	fprintf(stderr, "[Container] Container %d not found!  Move failed\n",
		packet->id);
	return;
    }

    c->x = packet->x;
    c->y = packet->y;
}


void container_shift(const double offx, const double offy)
{
    container_t *c;

    list_for_each(c, &container_list) {
	c->x += offx;
	c->y += offy;
    }
}


static void container_free(container_t *c)
{
    free(c);
    container_count--;
}


void container_delete(const struct packet_container_del *packet)
{
    container_t *c;
    assert(packet);

    c = container_from_id(packet->id);
    if (!c) {
	fprintf(stderr, "[Container] Container %d not found!  Delete failed\n",
		packet->id);
	return;
    }

    list_remove(c);
    container_free(c);
}

/*--------------------------------------------------------------*/

void container_start(void)
{
    list_init(container_list);
    container_count = 0;
}


void container_stop(void)
{
    list_free(container_list, container_free);
    assert(container_count == 0);
}


void container_init(void)
{
    bmp_container = load_bitmap("data/container.tga", NULL);
    assert(bmp_container);

    tex_container = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					       bmp_container, -1);
}


void container_shutdown(void)
{
    if (bmp_container) {
	destroy_bitmap(bmp_container);
	bmp_container = NULL;
    }
}
