/* backpack.c,
 *
 * Backpacks are dropped by dead players.  They contain some ammo.
 */

#include <alleggl.h>
#include <allegro.h>
#include <assert.h>
#include "backpack.h"
#include "camera.h"
#include "map.h"
#include "network.h"
#include "packet-backpack.h"
#include "player.h"
#include "server.h"
#include "sprite.h"
#include "step.h"
#include "texdraw.h"
#include "universal.h"


static const sprite_t backpack_sprite = { 2, 2, 46, 45 };

static BITMAP *bmp_backpack;
static GLuint tex_backpack;

/*--------------------------------------------------------------*/

static void backpack_update_unit(backpack_t *b, const game_state_t *state);
static void backpack_velocity(const void *q, double *vx, double *vy);
static int backpack_step(void *q, const int tick, const void *state);
static bool backpack_collided(void *q, const int tick, const int collisions);


static void backpack_send_taken(const backpack_t *b)
{
    struct packet_backpack_taken packet;
    NLbyte lbuf[MAX_PACKET_SIZE], rbuf[MAX_PACKET_SIZE];
    NLint llen, rlen;
    enum WEAPON_CLASS w;

    packet.time = server_time;
    packet.id = b->id;

    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++) {
	packet.ammo[w] = b->weapon_ammo[w];
	b->taker->weapon_ammo[w] += b->weapon_ammo[w];
    }

    llen = packet_backpack_taken_encode(&packet, true,  lbuf, sizeof(lbuf));
    rlen = packet_backpack_taken_encode(&packet, false, rbuf, sizeof(rbuf));
    
    server_broadcast2(b->taker->id, lbuf, llen, rbuf, rlen);
}


void backpack_update(game_state_t *state)
{
    backpack_t *b, *nx;
    assert(state);

    list_for_each_next(b, nx, &state->backpack_list) {
	backpack_update_unit(b, state);

	if (b->taker) {
	    backpack_send_taken(b);
	    backpack_free(b);
	}
    }
}


static void backpack_update_unit(backpack_t *b, const game_state_t *state)
{
    const walk_time_callbacks_t callbacks = {
	backpack_velocity,
	backpack_step,
	backpack_collided
    };
    assert(state);

    walk_time(b, &callbacks, b->time, server_time, state);
}


static void backpack_velocity(const void *q, double *vx, double *vy)
{
    const backpack_t *b = q;
    assert(b);

    *vx = b->_spawn_vx;
    *vy = b->_spawn_vy - GRAVITY2*(b->time-b->_spawn_time)/1000.0;
}


static void backpack_move(backpack_t *b, const int tick)
{
    double t = (tick - b->_spawn_time)/1000.0;

    b->time = tick;
    if (tick > b->_spawn_time) {
	b->x = b->_spawn_x + b->_spawn_vx*t;
	b->y = b->_spawn_y + b->_spawn_vy*t - 0.5*GRAVITY2*t*t;
    }
    else {
	b->x = b->_spawn_x;
	b->y = b->_spawn_y;
    }
}


static int backpack_step(void *q, const int tick, const void *state0)
{
    int collisions = STEP_NO_COLLIDE;
    int x, y;
    backpack_t *b = q;
    const game_state_t *state = state0;
    assert(b);
    assert(state);

    backpack_move(b, tick);

    if (state->as_server) {
	player_t *p;

	p = player_touching_box(b->x, b->y, BACKPACK_W, BACKPACK_H, state);
	if (p) {
	    b->taker = p;
	    return STEP_COLLIDE_XY;
	}
    }

    if (!x_in_map(b->x))
	return STEP_COLLIDE_X;

    if (!y_in_map(b->y))
	return STEP_COLLIDE_Y;

    for (y = b->y; y <= b->y+BACKPACK_H-1; y++) {
	if (!y_in_map(y))
	    break;

	if (map_occupies(b->x-BACKPACK_W/2.0, y) ||
	    map_occupies(b->x+BACKPACK_W/2.0, y)) {
	    collisions |= STEP_COLLIDE_X;
	    break;
	}
    }

    for (x = b->x-BACKPACK_W/2.0; x <= b->x+BACKPACK_W/2.0; x++) {
	if (map_occupies(x, b->y)) {
	    collisions |= STEP_COLLIDE_BOTTOM;
	    break;
	}
	
	if (map_occupies(x, b->y+BACKPACK_H)) {
	    collisions |= STEP_COLLIDE_TOP;
	    break;
	}
    }

    return collisions;
}


static bool backpack_collided(void *q, const int tick, const int collisions)
{
    double vx, vy;
    backpack_t *b = q;
    assert(b);

    backpack_move(b, tick);
    backpack_velocity(b, &vx, &vy);

    b->_spawn_time = tick;
    b->_spawn_x = b->x;
    b->_spawn_y = b->y;

    if (collisions & STEP_COLLIDE_X)
	b->_spawn_vx = -vx/1.0;
    else
	b->_spawn_vx = vx;

    if (collisions & STEP_COLLIDE_TOP) {
	b->_spawn_vy = -vy;
    }
    else if (collisions & STEP_COLLIDE_BOTTOM) {
	b->_spawn_vx *= 0.75;
	b->_spawn_vy = 0.0;
    }
    else
	b->_spawn_vy = vy;

    return true;
}


/*--------------------------------------------------------------*/

static void backpack_draw_unit(const backpack_t *b);


void backpack_draw(const game_state_t *state)
{
    backpack_t *b;
    assert(state);

    glBindTexture(GL_TEXTURE_2D, tex_backpack);
    glBegin(GL_QUADS);

    list_for_each(b, &state->backpack_list)
	backpack_draw_unit(b);

    glEnd();			/* glBegin(GL_QUADS) */
}


static void backpack_draw_unit(const backpack_t *b)
{
    texcoord2d_t coord;

    if (!camera_can_see2(&camera, b->x, b->y,
			 backpack_sprite.w/2.0, backpack_sprite.h/2.0))
	return;

    texcoord_from_sprite(&coord, &backpack_sprite,
			 bmp_backpack->w, bmp_backpack->h);
    gl_draw_sprite_2d(&coord, b->x-backpack_sprite.w/4.0, b->y);
}

/*--------------------------------------------------------------*/

backpack_t *backpack_from_id(const backpack_id id, const game_state_t *state)
{
    backpack_t *b;
    assert(state);

    list_for_each(b, &state->backpack_list) {
	if (b->id == id)
	    return b;
    }

    return NULL;
}

/*--------------------------------------------------------------*/

backpack_id backpack_generate_unique_id(void)
{
    static backpack_id id;
    id++;
    return id;
}


backpack_t *backpack_new(const struct packet_player_die *packet,
			 game_state_t *state)
{
    backpack_t *b = malloc(sizeof(*b));
    enum WEAPON_CLASS w;
    assert(b);
    assert(packet);
    assert(state);

    b->id = packet->bp;
    b->parent = packet->id;

    /* Spawn state. */
    b->_spawn_time = packet->time;
    b->_spawn_x = packet->x;
    b->_spawn_y = packet->y;
    b->_spawn_vx = packet->vx;
    b->_spawn_vy = packet->vy + 100.0;

    /* Current state. */
    b->time = b->_spawn_time;
    b->x = b->_spawn_x;
    b->y = b->_spawn_y;

    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++)
	b->weapon_ammo[w] = 0;

    b->taker = NULL;

    /* Clients don't need to know about ammo in a backpack unless they
       pick it up. */

    list_add(state->backpack_list, b);
    return b;
}


void backpack_new2(const struct packet_player_die *packet,
		   const player_t *p, game_state_t *state)
{
    backpack_t *b;
    enum WEAPON_CLASS w;
    assert(p);
    assert(state);

    b = backpack_new(packet, state);

    for (w = WEAPON_FIRST; w < NUM_WEAPON_CLASSES; w++)
	b->weapon_ammo[w] = p->weapon_ammo[w] / 2;
}


void backpack_free(backpack_t *b)
{
    assert(b);

    list_remove(b);
    free(b);
}

/*--------------------------------------------------------------*/

void backpack_start(game_state_t *state)
{
    assert(state);

    list_init(state->backpack_list);    
}


void backpack_stop(game_state_t *state)
{
    assert(state);

    list_free(state->backpack_list, backpack_free);
}


void backpack_init(void)
{
    bmp_backpack = load_bitmap("data/backpack.tga", NULL);
    assert(bmp_backpack);

    tex_backpack = allegro_gl_make_texture_ex(AGL_TEXTURE_MASKED,
					      bmp_backpack, -1);
}


void backpack_shutdown(void)
{
    if (bmp_backpack) {
	destroy_bitmap(bmp_backpack);
	bmp_backpack = NULL;
    }
}
