/* svgame.c
 * 
 * Peter Wang <tjaden@users.sourceforge.net>
 */


#include <math.h>
#include <stdarg.h>
#include <string.h>
#include <allegro.h>
#include "alloc.h"
#include "error.h"
#include "list.h"
#include "map.h"
#include "mapfile.h"
#include "mathcnst.h"
#include "netmsg.h"
#include "object.h"
#include "objtypes.h"
#include "packet.h"
#include "sync.h"
#include "server.h"
#include "strlcpy.h"
#include "svclient.h"
#include "svintern.h"
#include "svstats.h"
#include "svticker.h"
#include "ticks.h"


static map_t *map;
static svticker_t *ticker;



#define Lsrv server_lua_namespace



/*
 *----------------------------------------------------------------------
 *	Creating new player objects
 *----------------------------------------------------------------------
 */


static start_t *pick_random_start (void)
{
    list_head_t *list;
    start_t *start;
    int n = 0;
    
    list = map_start_list (map);
    list_for_each (start, list) n++;

    if (n > 0) {
	n = rand () % n;	/* XXX temp */
	list_for_each (start, list)
	    if (!n--) return start;
    }

    return NULL;
}


static object_t *spawn_player (objid_t id)
{
    start_t *start;
    object_t *obj;
    svclient_t *c;

    start = pick_random_start ();
    obj = object_create_ex ("player", id);
    object_set_xy (obj, map_start_x (start), map_start_y (start));
    object_set_collision_tag (obj, id);
    map_link_object (map, obj);
    object_run_init_func (obj);

    if ((c = svclients_find_by_id (id))) {
	/* Call the player spawned hook.  */
	lua_getglobal (Lsrv, "_internal_player_spawned_hook");
	if (!lua_isfunction (Lsrv, -1))
	    error ("Missing _internal_player_spawned_hook\n");
	lua_pushnumber (Lsrv, id);
	lua_call (Lsrv, 1, 0);

	/* Reset network stuff.  */
	c->last_sent_aim_angle = 0;
    } 
    
    return obj;
}



/*
 *----------------------------------------------------------------------
 *	Handle packets that the main server module gives to us
 *	(network input)
 *----------------------------------------------------------------------
 */


void svgame_process_cs_gameinfo_packet (svclient_t *c, const char *buf,
					 size_t size)
{
    const void *end = buf + size;

    while (buf != end) {
	switch (*buf++) {
	    case MSG_CS_GAMEINFO_CONTROLS:
		c->old_controls = c->controls;
		buf += packet_decode (buf, "cf", &c->controls, &c->aim_angle);
		break;

	    case MSG_CS_GAMEINFO_WEAPON_SWITCH: {
		short len;
		char name[NETWORK_MAX_PACKET_SIZE];
		
		buf += packet_decode (buf, "s", &len, name);
		if (c->client_object) {
		    lua_pushstring (Lsrv, name);
		    object_call (Lsrv, c->client_object, "switch_weapon", 1);
		}
		break;
	    }

	    default:
		error ("error: unknown code in gameinfo packet (server)\n");
	}
    }
}


void svgame_process_cs_ping (svclient_t *c)
{
    c->pong_time = svticker_ticks (ticker);
    svclient_send_rdm_byte (c, MSG_SC_PONG);
}


void svgame_process_cs_boing (svclient_t *c)
{
    c->lag = (svticker_ticks (ticker) - c->pong_time) / 2;
}



/*
 *----------------------------------------------------------------------
 *	Helpers to send large gameinfo packets
 *	This will split a gameinfo packet into multiple if necessary
 *	(network output)
 *----------------------------------------------------------------------
 */


static char gameinfo_packet_buf[NETWORK_MAX_PACKET_SIZE];
static int gameinfo_packet_size;
static svclient_t *gameinfo_packet_dest;


static void start_gameinfo_packet (svclient_t *c)
{
    gameinfo_packet_buf[0] = MSG_SC_GAMEINFO;
    gameinfo_packet_size = 1;
    gameinfo_packet_dest = c;
}


static void add_to_gameinfo_packet_raw (void *buf, size_t size)
{
    if (gameinfo_packet_size + size > sizeof gameinfo_packet_buf) {
	if (gameinfo_packet_dest)
	    svclient_send_rdm (gameinfo_packet_dest, gameinfo_packet_buf,
			     gameinfo_packet_size);
	else
	    svclients_broadcast_rdm (gameinfo_packet_buf,
				     gameinfo_packet_size);
	gameinfo_packet_size = 1;
    }
    
    memcpy (gameinfo_packet_buf + gameinfo_packet_size, buf, size);
    gameinfo_packet_size += size;
}


static void add_to_gameinfo_packet (const char *fmt, ...)
{
    va_list ap;
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;
    
    va_start (ap, fmt);
    size = packet_encode_v (buf, fmt, ap);
    va_end (ap);

    add_to_gameinfo_packet_raw (buf, size);
}


static void done_gameinfo_packet (void)
{
    if (gameinfo_packet_size > 1) {
	if (gameinfo_packet_dest)
	    svclient_send_rdm (gameinfo_packet_dest, gameinfo_packet_buf,
			     gameinfo_packet_size);
	else
	    svclients_broadcast_rdm (gameinfo_packet_buf,
				     gameinfo_packet_size);
    }
}



/*
 *----------------------------------------------------------------------
 *	Queue for gameinfo packets
 *	The packets that are put in here are generally generated
 *	before we are ready to send them (network output)
 *----------------------------------------------------------------------
 */


typedef struct node {
    struct node *next;
    struct node *prev;
    void *packet;
    size_t size;
} node_t;


static list_head_t packet_queue;


static void gameinfo_packet_queue_init (void)
{
    list_init (packet_queue);
}


static void add_to_gameinfo_packet_queue (void *packet, size_t size)
{
    node_t *node = alloc (sizeof *node);
    node->packet = alloc (size);
    memcpy (node->packet, packet, size);
    node->size = size;
    list_append (packet_queue, node);
}


static void node_freer (node_t *node)
{
    free (node->packet);
    free (node);
}

static void gameinfo_packet_queue_shutdown (void)
{
    list_free (packet_queue, node_freer);
}


static void gameinfo_packet_queue_flush (void)
{
    node_t *node;

    list_for_each (node, &packet_queue)
	add_to_gameinfo_packet_raw (node->packet, node->size);

    list_free (packet_queue, node_freer);
}



/*
 *----------------------------------------------------------------------
 *	Make a MSG_SC_GAMEINFO_OBJECT_CREATE packet (network output)
 *----------------------------------------------------------------------
 */


/* XXX lots of potention buffer overflows */
static size_t make_object_creation_packet (object_t *obj, char *buf)
{
    int top = lua_gettop (Lsrv);
    const char *type;
    list_head_t *list;
    creation_field_t *f;
    char *p;

    /* look up object type alias */
    type = objtype_name (object_type (obj));
    lua_getglobal (Lsrv, "object_alias");
    lua_pushstring (Lsrv, type);
    lua_rawget (Lsrv, -2);
    if (lua_isstring (Lsrv, -1))
	type = lua_tostring (Lsrv, -1);
    
    /* create the packet */
    p = buf + packet_encode (buf, "cslcffffl", MSG_SC_GAMEINFO_OBJECT_CREATE,
			     type,
			     object_id (obj), object_hidden (obj),
			     object_x (obj), object_y (obj),
			     object_xv (obj), object_yv (obj),
			     object_collision_tag (obj));

    /* creation fields */
    list = object_creation_fields (obj);
    list_for_each (f, list) {
	int type = object_get_var_type (obj, f->name);
	
	switch (type) {
	    case LUA_TNUMBER:
		p += packet_encode (p, "csf", 'f', f->name,
				    object_get_number (obj, f->name));
		break;
	    case LUA_TSTRING:
		p += packet_encode (p, "css", 's', f->name,
				    object_get_string (obj, f->name));
		break;
	    default:
		error ("error: unsupported type for creation field (server)\n");
	}
    }

    p += packet_encode (p, "c", 0); /* terminator */

    lua_settop (Lsrv, top);

    return p - buf;
}



/*
 *----------------------------------------------------------------------
 *	Feeding the game state to clients (network output)
 *----------------------------------------------------------------------
 */


static void feed_game_state_to (svclient_t *c)
{
    start_gameinfo_packet (c);

    /* map */
    add_to_gameinfo_packet ("cs", MSG_SC_GAMEINFO_MAPLOAD,
			    server_current_map_file);

    /* objects */
    {
	list_head_t *list;
	object_t *obj;
	char buf[NETWORK_MAX_PACKET_SIZE];
	size_t size;

	list = map_object_list (map);
	list_for_each (obj, list) {
	    size = make_object_creation_packet (obj, buf);
	    add_to_gameinfo_packet_raw (buf, size);
	}
    }

    done_gameinfo_packet ();

    svclient_send_rdm_byte (c, MSG_SC_GAMESTATEFEED_DONE);

    /* After the svclient receives the game state, it will automatically
     * switch to 'paused' mode. */
}


static void perform_mass_game_state_feed (void)
{
    svclient_t *c;
    object_t *obj;
    int done;
    char byte;

    svclients_broadcast_rdm_byte (MSG_SC_GAMESTATEFEED_REQ);

    for_each_svclient (c) {
	if (c->state == SVCLIENT_STATE_JOINED) {
	    svclient_clear_ready (c);
	    svclient_set_cantimeout (c);
	    svclient_set_timeout (c, 5);
	}
    }

    sync_server_unlock ();
    do {
	sync_server_lock ();

	done = 1;
	for_each_svclient (c) {
	    if (c->state != SVCLIENT_STATE_JOINED)
		continue;
			    
	    if (svclient_ready (c))
		continue;

	    if (svclient_timed_out (c)) {
		svclient_set_state (c, SVCLIENT_STATE_STALE);
		server_log (1, "Client %s timed out during game state feed",
			    c->name);
		continue;
	    }
	    
	    if (svclient_receive_rdm (c, &byte, 1) <= 0) {
		done = 0;
		continue;
	    }

	    if (byte == MSG_CS_GAMESTATEFEED_ACK) {
		feed_game_state_to (c);
		svclient_set_ready (c);
		svclient_clear_cantimeout (c);
	    }
	}

	sync_server_unlock ();
    } while (!done);
    sync_server_lock ();

    list_for_each (obj, map_object_list (map))
	object_clear_replication_flags (obj);
	
    svclients_broadcast_rdm_byte (MSG_SC_RESUME);
}


static void perform_single_game_state_feed (svclient_t *c)
{
    char byte;

    svclient_send_rdm_byte (c, MSG_SC_GAMESTATEFEED_REQ);

    svclient_set_cantimeout (c);
    svclient_set_timeout (c, 5);

    sync_server_unlock ();
    while (1) {
	sync_server_lock ();

	if (svclient_timed_out (c)) {
	    svclient_set_state (c, SVCLIENT_STATE_STALE);
	    server_log (1, "Client %s timed out during game state feed", c->name);
	    sync_server_unlock ();
	    break;
	}

	if (svclient_receive_rdm (c, &byte, 1) <= 0) {
	    sync_server_unlock ();
	    continue;
	}

	if (byte == MSG_CS_GAMESTATEFEED_ACK) {
	    feed_game_state_to (c);
	    svclient_set_ready (c);
	    svclient_clear_cantimeout (c);
	    sync_server_unlock ();
	    break;
	}

	sync_server_unlock ();
    }
    sync_server_lock ();
}



/*
 *----------------------------------------------------------------------
 *	Sending in-game state packets (network output)
 *----------------------------------------------------------------------
 */


static void send_svclient_aim_angles (void)
{
    svclient_t *c;

    for_each_svclient (c) {
	if (ABS (c->aim_angle - c->last_sent_aim_angle) > (M_PI/16)) {
	    add_to_gameinfo_packet ("clf", MSG_SC_GAMEINFO_CLIENT_AIM_ANGLE,
				    c->id, c->aim_angle);
	    c->last_sent_aim_angle = c->aim_angle;
	}
    }
}


static void send_object_updates (void)
{
    list_head_t *object_list;
    object_t *obj;
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;
    
    object_list = map_object_list (map);
    list_for_each (obj, object_list) {
	if (object_stale (obj)) {
	    add_to_gameinfo_packet ("cl", MSG_SC_GAMEINFO_OBJECT_DESTROY,
				    object_id (obj));
	    continue;
	}

	if (object_need_replication (obj, OBJECT_REPLICATE_CREATE)) {
	    size = make_object_creation_packet (obj, buf);
	    add_to_gameinfo_packet_raw (buf, size);
	}

	if (object_need_replication (obj, OBJECT_REPLICATE_UPDATE))
	    add_to_gameinfo_packet ("clffffff", MSG_SC_GAMEINFO_OBJECT_UPDATE,
				    object_id (obj), 
				    object_x (obj), object_y (obj),
				    object_xv (obj), object_yv (obj),
				    object_total_xa (obj), object_total_ya (obj));

	if (object_need_replication (obj, OBJECT_REPLICATE_HIDDEN))
	    add_to_gameinfo_packet ("clc", MSG_SC_GAMEINFO_OBJECT_HIDDEN,
				    object_id (obj), object_hidden (obj));

	object_clear_replication_flags (obj);
    }
}



/*
 *----------------------------------------------------------------------
 *	Stuff that makes the game go around
 *----------------------------------------------------------------------
 */


static void handle_svclient_controls (void)
{
    svclient_t *c;
    object_t *obj;

    for_each_svclient (c) {
	if (c->state != SVCLIENT_STATE_JOINED)
	    continue;

	if (!(obj = c->client_object)) {
	    if (c->controls & CONTROL_RESPAWN) {
		c->client_object = spawn_player (c->id);
		object_set_replication_flag (c->client_object,
					     OBJECT_REPLICATE_CREATE);
	    }
	    continue;
	}

	object_update_ladder_state (obj, map);

	/*
	 * Left and right.
	 */
	switch (c->controls & (CONTROL_LEFT | CONTROL_RIGHT)) {
	    case CONTROL_LEFT:
		object_set_intrinsic_xa (obj, -2);
		break;
	    case CONTROL_RIGHT:
		object_set_intrinsic_xa (obj, +2);
		break;
	    default:
		object_set_intrinsic_xa (obj, 0);
	}
	    
	/*
	 * Up and down.
	 */
	switch (c->controls & (CONTROL_UP | CONTROL_DOWN)) {

	    case CONTROL_UP:
		if (object_head_above_ladder (obj)) {
		    object_set_intrinsic_ya (obj, -8);
		    object_set_jump (obj, 4);
		}
		else if (object_standing_on_ladder (obj) ||
			 object_in_ladder (obj)) {
		    object_set_intrinsic_ya (obj, -4);
		    object_set_jump (obj, 2);
		}
		else {
		    int jump = object_jump (obj);
		    if ((jump > 0) && (!object_extrinsic_ya (obj))) {
			object_add_intrinsic_ya (obj, (- object_mass (obj)
						       - 4/object_jump(obj)));
			object_set_jump (obj, (jump < 10) ? (jump + 1) : 0);
		    }
		    else if ((jump == 0) && (object_supported (obj, map))) {
			float yv = object_yv (obj);
			if ((yv >= 0.0) && (yv < 0.005))
			    object_set_jump (obj, 1);
		    }
		}
		break;

	    case CONTROL_DOWN:
		if (object_standing_on_ladder (obj) ||
		    object_head_above_ladder (obj) ||
		    object_in_ladder (obj)) {
		    object_set_number (obj, "_internal_down_ladder", 1);
		    object_set_intrinsic_ya (obj, 4);
		}
		break;

	    default:
		object_set_jump (obj, 0);
		object_set_number (obj, "_internal_down_ladder", 0);
		break;
	}

	/*
	 * Fire / drop mine.
	 */
	if (c->controls & CONTROL_FIRE)
	    object_call (Lsrv, obj, "_internal_fire_hook", 0);
	if (c->controls & CONTROL_DROP_MINE) {
	    object_call (Lsrv, obj, "_internal_drop_mine_hook", 0);
	    c->controls &= ~CONTROL_DROP_MINE; /* hack */
	}
    }
}


static void perform_physics (void)
{
    list_head_t *object_list;
    object_t *obj;

    handle_svclient_controls ();

    object_list = map_object_list (map);
    list_for_each (obj, object_list)
	object_do_physics (obj, map); /* XXX find better name */

    map_blasts_update_with_collisions (map);
}


static void poll_update_hooks (int elapsed_msecs)
{
    list_head_t *list;
    object_t *obj;

    list = map_object_list (map);
    list_for_each (obj, list)
	object_poll_update_hook (obj, elapsed_msecs);
}



/*
 *----------------------------------------------------------------------
 *	Lua bindings (stuff that makes the game go around part II)
 *----------------------------------------------------------------------
 */


object_t *svgame_spawn_object (const char *typename, float x, float y)
{
    object_t *obj;

    /* For when an object's init function calls spawn_object() while
     * the map is still being loaded.  */
    if (!map)
	return NULL;

    if (!(obj = object_create (typename)))
	return NULL;

    object_set_xy (obj, x, y);
    object_set_replication_flag (obj, OBJECT_REPLICATE_CREATE);
    map_link_object (map, obj);
    object_run_init_func (obj);
    return obj;
}


object_t *svgame_spawn_projectile (const char *typename, object_t *owner,
				   float speed, float delta_angle)
{
    svclient_t *c;
    object_t *obj;
    float angle;
    float cos_angle;
    float sin_angle;

    if (!(c = svclients_find_by_id (object_id (owner))))
	return NULL;

    if (!(obj = object_create (typename)))
	return NULL;

    angle = c->aim_angle + delta_angle;
    cos_angle = cos (angle);
    sin_angle = sin (angle);

    object_set_xy (obj,
		   /* This is a bit cheap and may need to be
		    * adjustable on a per-weapon basis.  */
		   object_x (owner) + 5 * cos_angle, 
		   object_y (owner) + 5 * sin_angle);
    object_set_xvyv (obj, speed * cos_angle, speed * sin_angle);
    object_set_collision_tag (obj, object_collision_tag (owner));
    object_set_replication_flag (obj, OBJECT_REPLICATE_CREATE);
    object_set_number (obj, "angle", angle);
    object_add_creation_field (obj, "angle");
    object_set_number (obj, "owner", object_id (owner));
    map_link_object (map, obj);
    object_run_init_func (obj);
    return obj;
}


object_t *svgame_spawn_projectile_raw (const char *typename, int owner,
				       float x, float y, float angle,
				       float speed)
{
    svclient_t *c;
    object_t *obj;
    float cos_angle;
    float sin_angle;

    if (!(c = svclients_find_by_id (owner)))
	return NULL;

    if (!(obj = object_create (typename)))
	return NULL;

    cos_angle = cos (angle);
    sin_angle = sin (angle);

    object_set_xy (obj, x, y);
    object_set_xvyv (obj, speed * cos_angle, speed * sin_angle);
    object_set_replication_flag (obj, OBJECT_REPLICATE_CREATE);
    object_set_number (obj, "angle", angle);
    object_add_creation_field (obj, "angle");
    object_set_number (obj, "owner", owner);
    map_link_object (map, obj);
    object_run_init_func (obj);
    return obj;
}


static void queue_particle_packet (char type, float x, float y,
				   long nparticles, float spread)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!map)	/* may be called while in editor */
	return;

    size = packet_encode (buf, "ccfflf", MSG_SC_GAMEINFO_PARTICLES_CREATE,
			  type, x, y, nparticles, spread);
    add_to_gameinfo_packet_queue (buf, size);
}

void svgame_spawn_blood (float x, float y, long nparticles, float spread)
{
    queue_particle_packet ('b', x, y, nparticles, spread);
}

void svgame_spawn_sparks (float x, float y, long nparticles, float spread)
{
    queue_particle_packet ('s', x, y, nparticles, spread);
}

void svgame_spawn_respawn_particles (float x, float y, long nparticles,
				     float spread)
{
    queue_particle_packet ('r', x, y, nparticles, spread);
}


void svgame_spawn_blod (float x, float y, long nparticles)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;
    
    size = packet_encode (buf, "cffl", MSG_SC_GAMEINFO_BLOD_CREATE,
			  x, y, nparticles);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_spawn_explosion (const char *name, float x, float y)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    size = packet_encode (buf, "csff", MSG_SC_GAMEINFO_EXPLOSION_CREATE,
			  name, x, y);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_spawn_blast (float x, float y, float radius, int damage, int owner)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    size = packet_encode (buf, "cfffl", MSG_SC_GAMEINFO_BLAST_CREATE,
			  x, y, radius, damage);
    add_to_gameinfo_packet_queue (buf, size);

    map_blast_create (map, x, y, radius, damage, owner);
}


/* XXX we only allow a single string arg at the moment
   XXX add more as they are required, somehow */
void svgame_call_method_on_clients (object_t *obj, const char *method,
				    const char *arg)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!map)	/* may be called while in editor */
	return;

    size = packet_encode (buf, "clss", MSG_SC_GAMEINFO_OBJECT_CALL,
			  object_id (obj), method, arg);
    add_to_gameinfo_packet_queue (buf, size);
}


/* Check if an object would collide with a player object, if it were
 * to be unhidden.  */
int svgame_object_would_collide_with_player_if_unhidden (object_t *obj)
{
    return (!object_stale (obj) &&
	    object_would_collide_with_player_if_unhidden (
		obj, map, object_x (obj), object_y (obj)));
}


void svgame_tell_health (object_t *obj, int health)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!object_is_client (obj))
	return;

    size = packet_encode (buf, "clcl", MSG_SC_GAMEINFO_CLIENT_STATUS,
			  object_id (obj), 'h', health);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_tell_armour (object_t *obj, int armour)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!object_is_client (obj))
	return;

    size = packet_encode (buf, "clcl", MSG_SC_GAMEINFO_CLIENT_STATUS,
			  object_id (obj), 'A', armour);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_tell_ammo (object_t *obj, int ammo)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!object_is_client (obj))
	return;

    size = packet_encode (buf, "clcl", MSG_SC_GAMEINFO_CLIENT_STATUS,
			  object_id (obj), 'a', ammo);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_set_score (client_id_t client_id, const char *score)
{
    svclient_t *c;
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    if (!(c = svclients_find_by_id (client_id)))
	return;

    svclient_set_score (c, score);

    size = packet_encode (buf, "cls", MSG_SC_GAMEINFO_CLIENT_SCORE,
			  client_id, score);
    add_to_gameinfo_packet_queue (buf, size);
}


void svgame_play_sound_on_clients (object_t *obj, const char *sound)
{
    char buf[NETWORK_MAX_PACKET_SIZE];
    size_t size;

    size = packet_encode (buf, "cffs", MSG_SC_GAMEINFO_SOUND_PLAY,
			  object_x (obj), object_y (obj), sound);
    add_to_gameinfo_packet_queue (buf, size);
}


const char *svgame_get_client_name (client_id_t client_id)
{
    svclient_t *c = svclients_find_by_id (client_id);
    return c ? c->name : NULL;
}


void svgame_broadcast_text_message (const char *msg)
{
    char buf[64];
    strlcpy (buf, msg, sizeof buf);
    svclients_broadcast_rdm_encode ("cs", MSG_SC_TEXT, buf);
}


void svgame_send_text_message (client_id_t client_id, const char *msg)
{
    svclient_t *c = svclients_find_by_id (client_id);
    if (c) {
	char buf[64];
	strlcpy (buf, msg, sizeof buf);
	svclient_send_rdm_encode (c, "cs", MSG_SC_TEXT, buf);
    }
}



/*
 *----------------------------------------------------------------------
 *	Main game loop
 *----------------------------------------------------------------------
 */


static void handle_new_svclient_feeds (void)
{
    svclient_t *c;

    for_each_svclient (c)
	if (svclient_wantfeed (c))
	    goto need_feeding;

    return;

  need_feeding:

    svclients_broadcast_rdm_byte (MSG_SC_PAUSE);

    for_each_svclient (c) if (svclient_wantfeed (c)) {
	server_log (0, "Feeding new client %s", c->name);
	perform_single_game_state_feed (c);

	{
	    object_t *obj = spawn_player (c->id);
	    c->client_object = obj;

	    {
		char buf[NETWORK_MAX_PACKET_SIZE+1] = { MSG_SC_GAMEINFO };
		size_t size = make_object_creation_packet (obj, buf+1);
		svclients_broadcast_rdm (buf, size+1);
	    }
	}

    	svclient_clear_wantfeed (c);
    }

    svclients_broadcast_rdm_byte (MSG_SC_RESUME);
}


static void purge_stale_objects (void)
{
    svclient_t *c;
    
    for_each_svclient (c)
	if ((c->client_object) && (object_stale (c->client_object)))
	    c->client_object = NULL;
    
    map_destroy_stale_objects (map);
}


static void svgame_poll (void)
{
    unsigned long t = svticker_ticks (ticker);
    long dt, i;

    if (!svticker_poll (ticker))
	return;
    if ((dt = svticker_ticks (ticker) - t) <= 0)
	return;
    
    handle_new_svclient_feeds ();

    for (i = 0; i < dt; i++)
	perform_physics ();

    poll_update_hooks (MSECS_PER_TICK * dt);

    start_gameinfo_packet (NULL);
    send_svclient_aim_angles ();
    send_object_updates ();
    gameinfo_packet_queue_flush ();
    done_gameinfo_packet ();

    purge_stale_objects ();

    if (svstats_poll ()) {
	char buf[1024];
	uszprintf (buf, sizeof buf,
		   "Incoming %.1f, outgoing %.1f  (avg bytes per sec)",
		   svstats_average_incoming_bytes,
		   svstats_average_outgoing_bytes);
	server_interface_set_status (buf);
    }
}



/*
 *----------------------------------------------------------------------
 *	Init and shutdown
 *----------------------------------------------------------------------
 */


static int init_game_state (void)
{
    svclient_t *c;

    /* Load map.  */
    map = map_load (server_next_map_file, 0, 0);
    if (!map) {
	server_log (1, "Couldn't load map %s", server_next_map_file);
	return -1;
    }
    string_set (server_current_map_file, server_next_map_file);

    /* Call new map hook.  */
    lua_getglobal (Lsrv, "_internal_new_map_hook");
    if (!lua_isfunction (Lsrv, -1))
	error ("Missing _internal_new_map_hook\n");
    lua_pushstring (Lsrv, server_current_map_file);
    lua_call (Lsrv, 1, 0);

    /* Spawn a bunch of players.  */
    for_each_svclient (c)
	if (c->state == SVCLIENT_STATE_JOINED)
	    c->client_object = spawn_player (c->id);

    return 0;
}


static void free_game_state (void)
{
    if (map) {
	map_destroy (map);
	map = NULL;
    }
}


static int svgame_init (void)
{
    server_log (1, "Entering game");

    gameinfo_packet_queue_init ();

    if (init_game_state () < 0) {
	server_log (1, "Error initialising game state");
	server_set_next_state (SERVER_STATE_LOBBY);
	return -1;
    }

    perform_mass_game_state_feed ();

    ticker = svticker_create (TICKS_PER_SECOND);
    svstats_init ();

    return 0;
}


static void svgame_shutdown (void)
{
    server_interface_set_status (NULL);

    svstats_shutdown ();
    svticker_destroy (ticker);

    free_game_state ();

    gameinfo_packet_queue_shutdown ();
}


server_state_procs_t the_procs =
{
    svgame_init,
    svgame_poll,
    svgame_shutdown
};

server_state_procs_t *svgame_procs = &the_procs;
