#ifndef		__GAME_INTERFACE_H__
#define		__GAME_INTERFACE_H__

#include <allegro.h>
#include "debug.h"

#ifdef ALLEGRO_WINDOWS
#define DLLEXPORT __declspec(dllexport)
#else
#define DLLEXPORT
#endif	//ALLEGRO_WINDOWS

struct GameSettings {
	char *name;		// short name of the variable (without spaces)
	char *label;	// label for the GUI
	char *options;	// pipe separated array of possible options, e.g. "first|second|third"
	int value;		// currently selected value (0 based)
};

class IGame {
	protected:
		// This string will contain the full path to the directory
		// where the game is installed including the trailing
		// slash/backslash, e.g. "c:\amgc2\games\mygame\" or
		// "/home/myname/amgc2/games/mygame/".
		char *game_path;

	public:
		// Constructor. Derived classes should not load data or
		// initialize game structure in the constructor. Use the
		// Init() method to do that!
		IGame() : game_path(0) {
		}

		// Destructor. Derived classs should use the Deinit() method
		// to clean up instead of the destructor.
		virtual ~IGame() {
			if (game_path) {
				delete [] game_path;
				game_path = 0;
			}
		}

		// should not contain any weird characters; should be fairly short;
		virtual char *GetName() = 0;

		// just first and last name will do;
		virtual char *GetAuthor() = 0;

		// Just a few short lines, probably no more than 250-300 characters or so.
		// \n can be used to separate paragraphs
		virtual char *GetDescription() = 0;

		// the version string; should be in format x.y (for example 1.0 or 2.43)
		virtual char *GetVersion() = 0;

		virtual char *GetIconPath() {
			return "game_icon.bmp";
		}

		virtual void SetGamePath(const char *path) {
			if (game_path) {
				delete [] game_path;
			}
			game_path = ustrdup(path);
			char *filename = get_filename(game_path);
			if (filename) {
				filename[0] = 0;
			}
			put_backslash(game_path);
		}

		// All initialization code should go here, including loading
		// data and initializing game structures. Should return true
		// if successful, false otherwise.
		virtual bool Init() {
			return true;
		}

		// All cleanup code should go here.
		virtual void Deinit() {
		}

		// All drawing is done here. The game should draw only to the
		// canvas bitmap, i.e. it should not touch the screen.
		virtual void Draw(BITMAP *canvas) = 0;

		// keys: state of the 8 keys the game may use; 1 is key is down, 0 otherwise
		//   keys[0] ... left
		//   keys[1] ... right
		//   keys[2] ... up
		//   keys[3] ... down
		//   keys[4] ... action A
		//   keys[5] ... action B
		//   keys[6] ... action C
		//   keys[7] ... action D
		//
		// mouse: state of the mouse
		//   mouse[0] ... x coordinate
		//   mouse[1] ... y coordinate
		//   mouse[2] ... button flags (left, right, middle)
		//   mouse[3] ... scroller position
		//
		// dT: time delta since the previous frame
		//
		// return value:
		//   0 ... game should continue
		//   1 ... game over
		virtual int Logic(int keys[8], int mouse[4], float dT) = 0;

		// Should return the current score or -1 if not applicable.
		virtual int Score() = 0;

		// Should return false if you intend to draw your own cursor or
		// if you don't want to use the mouse at all.
		virtual bool UseSystemMouse() {
			return true;
		}

		// These two functions notify the game that it has been paused
		// or resumed. The game should make sure not to process input,
		// physics, AI and other game logic while it is paused, but may
		// and should continue to process animation logic, play music,
		// etc.
		virtual void Pause() {}
		virtual void Resume() {}

		// Should return true if keys can be redefined (otherwise
		// global keys are used)
		virtual bool RedefinesKeys() {
			return false;
		}

		// Should return names of redefinable keys or NULL if
		// key can't be redefined. i = [0,7]
		virtual const char *KeyName(int i) {
			switch (i) {
				default: return 0;
			};
		}

		// Should return to a NULL terminated array of GameSettings structures.
		// For example you could define the array like this:
		//
		//	GameSettings my_settings[] = {
		//		{ "name1", "option 1", "first|second|third", 1 },
		//		{ "name2", "option 2", "first|second|third", 0 },
		//		{ "name3", "option 3", "first|second|third", 2 },
		//		{ 0, 0, 0, 0 }
		//	};
		//
		// and then return my_settings in this function.
		virtual GameSettings* GetSettings() {
			return 0;
		}
};

#endif		//__GAME_INTERFACE_H__
