#include "IGame.h"

GameSettings tetris_settings[] = {
	{ "level", "start level", "1|2|3|4|5|6|7|8|9|10", 0 },
	{ 0, 0, 0, 0 }
};

static volatile int timer = 0;
static void timer_f() {
	++timer;
}
END_OF_STATIC_FUNCTION(timer_f);

class Tetris;

class MyColor {
	private:
		int Make(float h, float s, float v) {
			int r,g,b;
			hsv_to_rgb(h,s,v, &r,&g,&b);
			return makecol(r,g,b);
		}

	public:
		int c1, c2, c3;

		MyColor() {
			c1 = c2 = c3 = -1;
		}

		void Make() {
			float h,s,v;
			int r = getr(c1);
			int g = getg(c1);
			int b = getb(c1);
			rgb_to_hsv(r, g, b, &h, &s, &v);
			c1 = Make(h, s*0.7f, v*1.0f);
			c2 = Make(h, s*1.0f, v*0.9f);
			c3 = Make(h, s*0.7f, v*0.8f);
		}
};

#define 	NPIECES		19
static int PIECESIZE = 16;
#define		PLAYWIDTH	10
#define		PLAYHEIGHT	20

#define		white		makecol(255,255,255)
#define		black		makecol(  0,  0,  0)
#define		red			makecol(255,  0,  0)
#define		green		makecol(  0,255,  0)
#define		blue		makecol(  0,  0,255)
#define		cyan		makecol(  0,255,255)
#define		magenta		makecol(255,  0,255)
#define		yellow		makecol(255,255,  0)
#define		orange		makecol(255,170,  0)

class Piece {
	public:
		bool matrix[4][4];
		int wide, high, type;
		MyColor col;
		int posx, posy;
		Tetris *map;

	protected:
		bool CanMove();

	public:
		Piece();
		void Create(int type);
		void CreateRandomPiece();
		void RotateRight();
		void RotateLeft();
		void MoveLeft();
		void MoveRight();
		bool MoveDown();
		bool CanMoveLeft();
		bool CanMoveRight();
		bool CanMoveDown();
		void Draw(BITMAP *canvas);
		void Draw(BITMAP *canvas, int x, int y);
};



class Tetris : public IGame {
	private:
		int highscore;

		int x, y, w, h;

	    MyColor tile[PLAYWIDTH][PLAYHEIGHT];
		Piece *curPiece, *nextPiece;
		bool playing;
		int lines, score, level;
		int frameInterval;
		bool game_over;
		bool paused;
		int key_timer[8];

		void MsgTimer();
		void MsgChar(int c);

		void DropCurrentPiece();
		void CheckLines();
		bool LineFull(int j);
		void ClearLine(int j);
		void ClearPlayArea();
		void GameOver();

	public:
		Tetris();
		~Tetris();

		char *GetName() {
			return "Tetris";
		}

		char *GetAuthor() {
			return "Miran Amon";
		}

		char *GetDescription() {
			return "Just a very simple Tetris clone.";
		}

		char *GetVersion() {
			return "1.01";
		}

		char *GetIconPath() {
			return "tetris.bmp";
		}

		bool Init() {
			if (!IGame::Init()) {
				return false;
			}

			// custom initialization goes here
			score = 0;
			highscore = 0;

			LOCK_VARIABLE(timer);
			LOCK_FUNCTION(timer_f);

			PIECESIZE = (SCREEN_H-8)/PLAYHEIGHT;
			PIECESIZE = MID(10, PIECESIZE, 32);

			w = PLAYWIDTH*PIECESIZE;
			h = PLAYHEIGHT*PIECESIZE;
			x = 3*SCREEN_W/10 - w/2;
			y = (SCREEN_H - h)/2;

			game_over = false;
			paused = false;
			for (int i=0; i<8; i++) {
				key_timer[i] = 0;
			}

			return true;
		}

		void Deinit() {
			// custom deinitialization goes here
			// ...

			// no more Allegro calls after this line!
			IGame::Deinit();
		}

		void Draw(BITMAP *canvas);

		int Logic(int keys[8], int[4], float dT) {
			// Do all input processing and game logic here. Refer to the
			// comments in IGame.h for details on the format of the input
			// parameters and the requirements for the output.
			if (!game_over) {
				if (playing) {
					if (score > highscore) {
						highscore = score;
					}

					CheckLines();

					if (!paused) {
						for (int i=0; i<8; i++) {
							if (key_timer[i] == 0) {
								if (keys[i]) {
									MsgChar(i);
									static float delay[] = { 80, 80, 180, 20, 500, 500, 500, 500 };
									key_timer[i] = (int)(delay[i] / dT);
								}
							}
							else {
								--key_timer[i];
							}
						}
					}

					while (timer > 0) {
						if (!paused) {
							MsgTimer();
						}
						--timer;
					}
				}
				else {
					//if (keys[4] || keys[5] || keys[6] || keys[7]) {
						for (int i=0; i<8; i++) {
							key_timer[i] = 50;
						}
						timer = 0;
						//Play(1);
						Play(tetris_settings[0].value+1);
						timer = 0;
					//}
				}
			}

			if (game_over) {
				return 1;
			}
			else {
				return 0;
			}
		}

		int Score() {
			// Return -1 if this game doesn't have points, otherwise the last
			// valid score where more points is better.
			return highscore;
		}

		bool UseSystemMouse() {
			// Return false if the game draws its own cursor or doesn't use
			// the mouse at all.
			return false;
		}

		void Pause() {
			paused = true;
		}

		void Resume() {
			paused = false;
		}

		// Should return true if keys can be redefined (otherwise
		// global keys are used)
		bool RedefinesKeys() {
			return true;
		}

		// Should return names of redefinable keys or NULL if
		// key can't be redefined. i = [0,7]
		const char *KeyName(int i) {
			switch (i) {
				case 0:		return "Move left";		break;
				case 1:		return "Move right";	break;
				case 2:		return "Rotate";		break;
				case 3:		return "Down";			break;
				case 4:		return "Drop piece";	break;
			};

			return 0;
		}

		GameSettings* GetSettings() {
			return tetris_settings;
		}

		MyColor GetTile(int x, int y);
		void SetTile(int x, int y, const MyColor &val);
		void Play(int level);
};



extern "C" DLLEXPORT IGame* GetPlugin() {
	return new Tetris;
}



Piece::Piece() :wide(4), high(4), posx(rand()%(PLAYWIDTH-4)), posy(PLAYHEIGHT-1), map(NULL) {
	col.c1 = makecol(0,0,0);
}

void Piece::CreateRandomPiece() {
	int i = rand()%7;
	int type = i;
	switch (i) {
		case 0:		type = i;					break;
		case 1:		type = rand()%2 + 1;		break;
		case 2:		type = rand()%2 + 3;		break;
		case 3:		type = rand()%4 + 5;		break;
		case 4:		type = rand()%4 + 9;		break;
		case 5:		type = rand()%2 + 13;		break;
		case 6:		type = rand()%4 + 15;		break;
	};
	Create(type);
}


void Piece::Create(int type) {
	for (int i=0; i<4; i++) {
		for (int j=0; j<4; j++) {
			matrix[i][j] = false;
		}
	}

	this->type = type;
	switch (type) {
		case 0:
			matrix[0][0] = matrix[1][0] = matrix[0][1] = matrix[1][1] = true;
			wide = 2;
			high = 2;
			col.c1 = red;
			break;

		case 1:
			matrix[0][0] = matrix[0][1] = matrix[1][1] = matrix[1][2] = true;
			wide = 2;
			high = 3;
			col.c1 = green;
			break;

		case 2:
			matrix[0][1] = matrix[1][0] = matrix[1][1] = matrix[2][0] = true;
			wide = 3;
			high = 2;
			col.c1 = green;
			break;

		case 3:
			matrix[0][1] = matrix[0][2] = matrix[1][0] = matrix[1][1] = true;
			wide = 2;
			high = 3;
			col.c1 = cyan;
			break;

		case 4:
			matrix[0][0] = matrix[1][0] = matrix[1][1] = matrix[2][1] = true;
			wide = 3;
			high = 2;
			col.c1 = cyan;
			break;

		case 5:
			matrix[0][0] = matrix[0][1] = matrix[0][2] = matrix[1][0] = true;
			wide = 2;
			high = 3;
			col.c1 = yellow;
			break;

		case 6:
			matrix[0][0] = matrix[0][1] = matrix[1][1] = matrix[2][1] = true;
			wide = 3;
			high = 2;
			col.c1 = yellow;
			break;

		case 7:
			matrix[0][2] = matrix[1][0] = matrix[1][1] = matrix[1][2] = true;
			wide = 2;
			high = 3;
			col.c1 = yellow;
			break;

		case 8:
			matrix[0][0] = matrix[1][0] = matrix[2][0] = matrix[2][1] = true;
			wide = 3;
			high = 2;
			col.c1 = yellow;
			break;

		case 9:
			matrix[0][0] = matrix[1][0] = matrix[1][1] = matrix[1][2] = true;
			wide = 2;
			high = 3;
			col.c1 = orange;
			break;

		case 10:
			matrix[0][0] = matrix[0][1] = matrix[1][0] = matrix[2][0] = true;
			wide = 3;
			high = 2;
			col.c1 = orange;
			break;

		case 11:
			matrix[0][0] = matrix[0][1] = matrix[0][2] = matrix[1][2] = true;
			wide = 2;
			high = 3;
			col.c1 = orange;
			break;

		case 12:
			matrix[0][1] = matrix[1][1] = matrix[2][1] = matrix[2][0] = true;
			wide = 3;
			high = 2;
			col.c1 = orange;
			break;

		case 13:
			matrix[0][0] = matrix[0][1] = matrix[0][2] = matrix[0][3] = true;
			wide = 1;
			high = 4;
			col.c1 = magenta;
			break;

		case 14:
			matrix[0][0] = matrix[1][0] = matrix[2][0] = matrix[3][0] = true;
			wide = 4;
			high = 1;
			col.c1 = magenta;
			break;

		case 15:
			matrix[0][0] = matrix[1][0] = matrix[2][0] = matrix[1][1] = true;
			wide = 3;
			high = 2;
			col.c1 = blue;
			break;

		case 16:
			matrix[0][0] = matrix[0][1] = matrix[0][2] = matrix[1][1] = true;
			wide = 2;
			high = 3;
			col.c1 = blue;
			break;

		case 17:
			matrix[0][1] = matrix[1][1] = matrix[2][1] = matrix[1][0] = true;
			wide = 3;
			high = 2;
			col.c1 = blue;
			break;

		case 18:
			matrix[1][0] = matrix[1][1] = matrix[1][2] = matrix[0][1] = true;
			wide = 2;
			high = 3;
			col.c1 = blue;
			break;
	};

	col.Make();
}

void Piece::RotateRight() {
	int newType = type;
	int oldType = type;
	switch (type) {
		case 0:		newType = 0;	break;
		case 1:		newType = 2;	break;
		case 2:		newType = 1;	break;
		case 3:		newType = 4;	break;
		case 4:		newType = 3;	break;
		case 5:		newType = 6;	break;
		case 6:		newType = 7;	break;
		case 7:		newType = 8;	break;
		case 8:		newType = 5;	break;
		case 9:		newType = 10;	break;
		case 10:	newType = 11;	break;
		case 11:	newType = 12;	break;
		case 12:	newType = 9;	break;
		case 13:	newType = 14;	break;
		case 14:	newType = 13;	break;
		case 15:	newType = 16;	break;
		case 16:	newType = 17;	break;
		case 17:	newType = 18;	break;
		case 18:	newType = 15;	break;
	};

	Create(newType);

	if (posx+wide > PLAYWIDTH) Create(oldType);

	if (!CanMove()) RotateLeft();
}

void Piece::RotateLeft() {
	int newType = type;
	switch (type) {
		case 0:		newType = 0;	break;
		case 1:		newType = 2;	break;
		case 2:		newType = 1;	break;
		case 3:		newType = 4;	break;
		case 4:		newType = 3;	break;
		case 5:		newType = 8;	break;
		case 6:		newType = 5;	break;
		case 7:		newType = 6;	break;
		case 8:		newType = 7;	break;
		case 9:		newType = 12;	break;
		case 10:	newType = 9;	break;
		case 11:	newType = 10;	break;
		case 12:	newType = 11;	break;
		case 13:	newType = 14;	break;
		case 14:	newType = 13;	break;
		case 15:	newType = 18;	break;
		case 16:	newType = 15;	break;
		case 17:	newType = 16;	break;
		case 18:	newType = 17;	break;
	};

	Create(newType);
	if (!CanMove()) RotateRight();
}

void Piece::MoveLeft() {
	if (posx <= 0) return;
	posx--;
	if (!CanMove()) posx++;
}

void Piece::MoveRight() {
	if (posx+wide >= PLAYWIDTH) return;
	posx++;
	if (!CanMove()) posx--;
}

bool Piece::MoveDown() {
	if (!CanMoveDown()) return false;
	posy--;
	return true;
}

bool Piece::CanMove() {
	int mx, my;
	for (int j=0; j<high; j++) {
		for (int i=0; i<wide; i++) {
			mx = posx + i;
			my = posy - j;

			if (matrix[i][j] && map->GetTile(mx, my).c1 != -1) {
				return false;
			}
		}
	}

	return true;
}

bool Piece::CanMoveLeft() {
	if (posx <= 0) return false;
	posx--;
	bool ret = CanMove();
	posx++;
	return ret;
}

bool Piece::CanMoveRight() {
	if (posx+wide >= PLAYWIDTH) return false;
	posx++;
	bool ret = CanMove();
	posx--;
	return ret;
}

bool Piece::CanMoveDown() {
	if (posy-high < 0) return false;
	posy--;
	bool ret = CanMove();
	posy++;
	return ret;
}

void Piece::Draw(BITMAP *canvas) {
	Draw(canvas, posx*PIECESIZE, (PLAYHEIGHT - 1 - posy)*PIECESIZE);
}

void Piece::Draw(BITMAP *canvas, int x, int y) {
	x += posx*PIECESIZE;
	y += (PLAYHEIGHT - 1 - posy)*PIECESIZE;

	int px, py;
	for (int j=0; j<high; j++) {
		for (int i=0; i<wide; i++) {
			if (matrix[i][j]) {
				px = i*PIECESIZE + x;
				py = j*PIECESIZE + y;
				rectfill(canvas, px,   py,   px+PIECESIZE-1, py+PIECESIZE-1, col.c1);
				rect(canvas, px+1, py+1, px+PIECESIZE-2, py+PIECESIZE-2, col.c2);
				rect(canvas, px+3, py+3, px+PIECESIZE-4, py+PIECESIZE-4, col.c3);
			}
		}
	}
}



Tetris::Tetris() : IGame() {
	ClearPlayArea();

	curPiece = NULL;
	nextPiece = NULL;
	playing = false;
	lines = 0;
	score = 0;
}


Tetris::~Tetris() {
	if (playing) {
		GameOver();
	}

	if (curPiece) {
		delete curPiece;
		curPiece = NULL;
	}

	if (nextPiece) {
		delete nextPiece;
		nextPiece = NULL;
	}
}

void Tetris::ClearPlayArea() {
	MyColor c;
	c.c1 = -1;
	for (int y=0; y<PLAYHEIGHT; y++) {
		for (int x=0; x<PLAYWIDTH; x++) {
			SetTile(x, y, c);
		}
	}
}

MyColor Tetris::GetTile(int x, int y) {
	return tile[x][y];
}

void Tetris::SetTile(int x, int y, const MyColor &val) {
	tile[x][y] = val;
}

void Tetris::Draw(BITMAP *canvas) {
	clear_to_color(canvas, makecol(212,212,212));
	rectfill(canvas, this->x, this->y, this->x+w-1, this->y+h-1, black);
	rect(canvas, this->x-2, this->y-2, this->x+w+1, this->y+h+1, black);

	int x = this->x;
	int y = this->y + PIECESIZE*PLAYHEIGHT;
	MyColor c;
	for (int j=0; j<PLAYHEIGHT; j++) {
		x = this->x;
		y -= PIECESIZE;
		for (int i=0; i<PLAYWIDTH; i++) {
			c = GetTile(i,j);
			if (c.c1 != -1) {
				rectfill(canvas, x,   y,   x+PIECESIZE-1, y+PIECESIZE-1, c.c1);
				rect(canvas,x+1, y+1, x+PIECESIZE-2, y+PIECESIZE-2, c.c2);
				rect(canvas,x+3, y+3, x+PIECESIZE-4, y+PIECESIZE-4, c.c3);
			}
			x += PIECESIZE;
		}
	}

	if (curPiece) {
		curPiece->Draw(canvas, this->x, this->y);
	}


	x = this->x + 2*w + 22;
	x = MIN(SCREEN_W-4, x);
	y = this->y + 4;
	int dy = 12;
	int cb = makecol(192,192,192);
	int gray = makecol(64,64,64);

	textout_ex(canvas, font, "next piece", this->x + w + 20, y, gray, -1);
	rectfill(canvas, this->x + w + 18, y+dy-3, this->x + 2*w + 22, y+dy+4*PIECESIZE+2, cb);
	rect(canvas, this->x + w + 18, y+dy-3, this->x + 2*w + 22, y+dy+4*PIECESIZE+2, gray);
	if (nextPiece) {
		nextPiece->Draw(canvas, this->x + w + 20, y+=dy);
	}
	else {
		y+=dy;
	}
	y += 4*PIECESIZE;

	rectfill(canvas, x-80, y+2*dy-3, x, y+2*dy+8, cb);
	rect(canvas, x-80, y+2*dy-3, x, y+2*dy+8, gray);
	textout_right_ex(canvas, font, "score", x, y+=dy, gray, -1);
	textprintf_right_ex(canvas, font, x, y+=dy, gray, -1, "%d", score);
	y += dy;

	if (SCREEN_H > 240) {
		rectfill(canvas, x-80, y+2*dy-3, x, y+2*dy+8, cb);
		rect(canvas, x-80, y+2*dy-3, x, y+2*dy+8, gray);
		textout_right_ex(canvas, font, "lines", x, y+=dy, gray, -1);
		textprintf_right_ex(canvas, font, x, y+=dy, gray, -1, "%d", lines);
		y += dy;
	}

	rectfill(canvas, x-80, y+2*dy-3, x, y+2*dy+8, cb);
	rect(canvas, x-80, y+2*dy-3, x, y+2*dy+8, gray);
	textout_right_ex(canvas, font, "level", x, y+=dy, gray, -1);
	textprintf_right_ex(canvas, font, x, y+=dy, gray, -1, "%d", level);
	y += dy;

	/*
	rectfill(canvas, x-80, y+2*dy-3, x, y+2*dy+8, cb);
	rect(canvas, x-80, y+2*dy-3, x, y+2*dy+8, gray);
	textout_right_ex(canvas, font, "cur. high", x, y+=dy, gray, -1);
	textprintf_right_ex(canvas, font, x, y+=dy, gray, -1, "%d", highscore);
	y += dy;
	*/

	if (!playing && !game_over) {
		int bw = 180;
		int bh = 44;
		rectfill(canvas, (SCREEN_W - bw)/2, (SCREEN_H - bh)/2, (SCREEN_W - bw)/2+bw, (SCREEN_H - bh)/2+bh, cb);
		rect(canvas, (SCREEN_W - bw)/2, (SCREEN_H - bh)/2, (SCREEN_W - bw)/2+bw, (SCREEN_H - bh)/2+bh, gray);
		textout_centre_ex(canvas, font, "Press the SPACE key", SCREEN_W/2, (SCREEN_H - bh)/2+8, gray, -1);
		textout_centre_ex(canvas, font, "to start!", SCREEN_W/2, (SCREEN_H - bh)/2+24, gray, -1);
	}
}

void Tetris::Play(int level) {
	if (curPiece) {
		delete curPiece;
		curPiece = NULL;
	}

	if (nextPiece) {
		delete nextPiece;
		nextPiece = NULL;
	}

	curPiece = new Piece;
	curPiece->map = this;
	curPiece->CreateRandomPiece();
	nextPiece = new Piece;
	nextPiece->map = this;
	nextPiece->CreateRandomPiece();

	ClearPlayArea();
	frameInterval = 550 - level*40;
	install_int(timer_f, frameInterval);
	playing = true;
	lines = 0;
	score = 0;
	this->level = level;
}

void Tetris::MsgTimer() {
	if (curPiece->MoveDown() == false) {
		DropCurrentPiece();
		if (curPiece) {
			delete curPiece;
		}
		curPiece = nextPiece;
		nextPiece = new Piece;
		nextPiece->map = this;
		nextPiece->CreateRandomPiece();

		if (!curPiece->CanMoveDown()) {
			GameOver();
		}
	}
}

void Tetris::GameOver() {
	playing = false;
	remove_int(timer_f);
	timer = 0;
	if (curPiece) {
		delete curPiece;
		curPiece = NULL;
	}
	/*
	lines = 0;
	score = 0;
	*/
	game_over = true;
}



void Tetris::DropCurrentPiece() {
	int mx, my;
	for (int j=0; j<curPiece->high; j++) {
		for (int i=0; i<curPiece->wide; i++) {
			if (curPiece->matrix[i][j]) {
				mx = curPiece->posx + i;
				my = curPiece->posy - j;
				SetTile(mx, my, curPiece->col);
			}
		}
	}
}

void Tetris::MsgChar(int c) {
	if (!curPiece || !playing) return;

	switch (c) {
		case 0:
			curPiece->MoveLeft();
			break;

		case 1:
			curPiece->MoveRight();
			break;

		case 3:
			if (curPiece->MoveDown()) {
				score += ((600 - frameInterval)/50);
			}
			break;

		case 2:
			curPiece->RotateRight();
			break;

		case 4:
		case 5:
		case 6:
		case 7:
			while (curPiece->MoveDown()) {
				score += ((600 - frameInterval)/50);
			}
			break;
	};
}

void Tetris::CheckLines() {
	int fullLines = 0;
	int oldLines = lines;

	for (int j=0; j<PLAYHEIGHT; j++) {
		if (LineFull(j)) {
			++fullLines;
			ClearLine(j);
			--j;
		}
	}

	lines += fullLines;
	switch (fullLines) {
		case 4:	score += 700;
		case 3:	score += 500;
		case 2:	score += 300;
		case 1:	score += 100;
	};

	if (fullLines) {
		if (lines%20 < oldLines%20 && level < 10) {
			frameInterval -= 50;
			install_int(timer_f, frameInterval);
			++level;
		}
	}
}

bool Tetris::LineFull(int j) {
	for (int i=0; i<PLAYWIDTH; i++) {
		if (GetTile(i, j).c1 == -1) {
			return false;
		}
	}

	return true;
}

void Tetris::ClearLine(int j) {
	int i=0;
	for (; j<PLAYHEIGHT-1; j++) {
		for (i=0; i<PLAYWIDTH; i++) {
			SetTile(i, j, GetTile(i, j+1));
		}
	}
	MyColor c;
	c.c1 = -1;
	for (i=0; i<PLAYWIDTH; i++) {
		SetTile(i, j, c);
	}
}
