////////////////////////////////////////////////////////////////////////////
// Project:			ProjectZ Game
// Author:			Andrew Deren
// Date Started:	7/01/98
//
// File:			game.h
////////////////////////////////////////////////////////////////////////////

#ifndef __GAME_H
#define __GAME_H

#include "defs.h"
#include "map.h"

void startGame(void);

inline bool checkBoxes(int x1, int y1, int w1, int h1,
					   int x2, int y2, int w2, int h2) 
{
	return (! ((x1>=x2+w2) || (x2>=x1+w1) || (y1>=y2+h2) || (y2>=y1+h1)) );
}

/////////////////////////////////////////////////////
/////////////// StarField class//////////////////////
/////////////////////////////////////////////////////
#define MAX_STARS 100
class StarField {
public:
	inline StarField(int stars = MAX_STARS, int mx = X_TILES * TILE_SIZE, int my = Y_TILES * TILE_SIZE) 
		: numStars(stars), maxX(mx), maxY(my), timer(0), color(0), speed(0), hasObject(false),
		objectX(0), objectY(0), object(NULL) {
		starX = new int[numStars];
		starY = new int[numStars];

		for (int i=0; i<numStars; i++) {
			starX[i] = Random(maxX);
			starY[i] = Random(maxY);
		}
	}

	~StarField() {
		delete [] starX;
		delete [] starY;
	}

	inline void increment() {
		for (int i=0; i<numStars; i++) {
			starX[i]--;
			if (starX[i] < 0) {
				starX[i] += maxX;
				starY[i] = Random(maxY);
			}
		}
		objectX--;
		if (objectX + object->w < 0)
			hasObject = false;
	}

	inline void draw(BITMAP *bmp, int tilePart) {
		int i;
		for (i=0; i<numStars; i++)
			putpixel(bmp, starX[i]+tilePart, starY[i], color);
		if (hasObject)
			draw_sprite(bmp, object, objectX+tilePart, objectY);
	}

	inline void operator++(int temp) {
		timer++;
		if (timer >= speed) {
			increment();
			timer = 0;
		}
	}

	inline void addObject(BITMAP* obj, int x, int y) {
		if (!hasObject) {
			objectX = x;
			objectY = y;
			object = obj;
			hasObject = true;
		}
	}

	int speed;
	int timer;
	int color;

protected:

	const int numStars;
	int *starX;
	int *starY;

	BITMAP *object;
	int objectX;
	int objectY;
	bool hasObject;

	const int maxX;	// maximum x resolution of stars
	const int maxY;

};


class GameMap;
class Game;
/////////////////////////////////////////////////////
///////////////////// Bullet class //////////////////
/////////////////////////////////////////////////////
struct Bullet {
	inline Bullet() : x(0), y(0), w(0), h(0), l(0){}
	inline Bullet(int tx, int ty, int tw, int th, int tl) {
		x = tx; y = ty; w = tw; h = th; l = tl;
	}

	int x;
	int y;
	int w;
	int h;
	int l;	// level, and for shield bullets it's the position in the lookup array
};

/////////////////////////////////////////////////////
///////////////////// Weapon class //////////////////
/////////////////////////////////////////////////////
class Weapon {
public:
	Weapon(int maxBulls = 0);
	virtual ~Weapon();

	// draws the bullets onto bmp, also increment bullets count using the counter from
	// GameMap and location of the player
	virtual void draw(BITMAP* bmp, GameMap* game) = 0;

	// change the level of the weapon
	virtual void changeLevel(int newLevel);

protected:
	friend class GameMap;

	int level;				// each weapon has four levels, level 1 is 0

	int numBullets;			// number of bullets currently used
	int maxBullets;			// maximum number of bullets there can be
	int maxSpeed;			// speed at which the bullets are ejected

	int counter;			// counter to know when to advance the bullet

	Bullet *loc;			// location of the bullet

	int width;				// width of each bullet
	int height;				// height of each bullet

	BITMAP *sprite[4];		// graphics of the weapons (4 levels)

	int change;				// game asked for the change of the weapon by setting this to 1
							// if change can occur Weapon sets this to 2

};

/////////////////////////////////////////////////////
///////////////////// Weapon1 class //////////////////
/////////////////////////////////////////////////////
class Weapon1: public Weapon {
public:
	Weapon1();
	~Weapon1();

	// draws the bullets onto bmp, also increment bullets count using the counter from
	// GameMap and location of the player
	virtual void draw(BITMAP* bmp, GameMap* game);

	// change the level of the weapon
	virtual void changeLevel(int newLevel);

protected:
	int numShieldBullets;	// number of bullets in the shield
	Bullet sLoc[2];			// shield bullets, there is one for level 3 and 2 for level 4

	// lookup table for the bullets to circle around the ship
	// they hold the offsets to ship x and y location
#define AROUND_SHIP_COUNT 32

	const static int aroundShipX[AROUND_SHIP_COUNT];	
	const static int aroundShipY[AROUND_SHIP_COUNT];

	int counter2;			// used to know when to release next bullet


};

/////////////////////////////////////////////////////
///////////////////// Weapon2 class //////////////////
/////////////////////////////////////////////////////
// bullets going in different directions
// level 1: right and left
// level 2: right, left, up and down
// level 3: level 2 + up-right + down-right
// level 4: level 3 + up-left + down-left
class Weapon2: public Weapon {
public:
	Weapon2();
	~Weapon2();

	// draws the bullets onto bmp, also increment bullets count using the counter from
	// GameMap and location of the player
	virtual void draw(BITMAP* bmp, GameMap* game);

protected:
	enum DIRS {D_UP, D_UP_RIGHT, D_RIGHT, D_DOWN_RIGHT,
		D_DOWN, D_DOWN_LEFT, D_LEFT, D_UP_LEFT};
};

/////////////////////////////////////////////////////
///////////////////// Weapon3 class //////////////////
/////////////////////////////////////////////////////
class Weapon3: public Weapon {
public:
	Weapon3();
	~Weapon3();

	// draws the bullets onto bmp, also increment bullets count using the counter from
	// GameMap and location of the player
	virtual void draw(BITMAP* bmp, GameMap* game);

protected:

};

/////////////////////////////////////////////////////
///////////////////// Weapon4 class /////////////////
/////////////////////////////////////////////////////
// Displays a column of bullets, for level 1 there are 3
// level 2 - 4, level 3 - 5 and level 4 - 6
class Weapon4: public Weapon {
public:
	Weapon4();
	~Weapon4();

	// draws the bullets onto bmp, also increment bullets count using the counter from
	// GameMap and location of the player
	virtual void draw(BITMAP* bmp, GameMap* game);

protected:

};



/////////////////////////////////////////////////////
/////////////////// GameMap class ///////////////////
/////////////////////////////////////////////////////

// by how many pixels is the map scrolled
#define PART_INCREMENT 1


class GameMap : public Map {
public:
	GameMap();
	~GameMap();
	void draw(void);

	void moveShip(int xdir, int ydir);

	virtual bool loadMap(char *name) { Map::loadMap(name); 
		shipX = 0; shipY = (Y_TILES * TILE_SIZE) / 2 ; power = 100; speedX = 10; speedY = 10;
		tilePart = 0; 
	}

	void scroll(void) { 
		if (currentX < sizeX - X_TILES - 1) {
			tilePart += PART_INCREMENT; 
			if (tilePart >= TILE_SIZE) {
				tilePart = 0;
				currentX++;
				
				// if there is object waiting to be released,
				// release one
				if (objects[currentObject].x == currentX+X_TILES) {
					int level = objects[currentObject].level;
					
					if (level == 1) stars1.addObject((BITMAP*)datObjects[objects[currentObject].type*4+objects[currentObject].size].dat, (X_TILES)*TILE_SIZE, objects[currentObject].y*TILE_SIZE);
					else if (level == 2) stars2.addObject((BITMAP*)datObjects[objects[currentObject].type*4+objects[currentObject].size].dat, (X_TILES)*TILE_SIZE, objects[currentObject].y*TILE_SIZE);
					else if (level == 3) stars3.addObject((BITMAP*)datObjects[objects[currentObject].type*4+objects[currentObject].size].dat, (X_TILES)*TILE_SIZE, objects[currentObject].y*TILE_SIZE);

					if (currentObject + 1 < numObjects)
						currentObject++;
				}
			}
		}
		
		stars1++;
		stars2++;
		stars3++;
		checkFront();
	}

	inline void changeWeapon(int toWhich) {
		if (toWhich < 1 || toWhich > 4) return;

		// make request for weapon change
		currentWeapon->change = 1;
		change = toWhich;
	}

protected:
	int tilePart;	// which scrolling stage it is in

	//////////////////////////////////////////////////////
	// ship stuff
	//////////////////////////////////////////////////////
	int shipX;			// location of the ship
	int shipY;

	int speedX;			// speed of the ship
	int speedY;

	int width;			// size of the ship
	int height;

	int power;			// power 

	void checkFront();
	void checkBack();
	void checkTop();
	void checkBottom();

	// weapons used 
	Weapon1 weapon1;
	Weapon2 weapon2;
	Weapon3 weapon3;
	Weapon4 weapon4;
	Weapon *currentWeapon;

	int change;	// user wants to change a weapon to this type

	//////////////////////////////////////////////////////
	// starfield stuff
	// there are 3 levels of the stars, 
	// 1 is the first level (closest) and fastest moving and 
	// brightest
	//////////////////////////////////////////////////////
	StarField stars1;
	StarField stars2;
	StarField stars3;

	int currentObject;	// current object used on the starfield

	//////////////////////////////////////////////////////

	BITMAP *bmp;	// doulbe buffer
	int blitCounter;	// to keep the frame rate constant

	friend class Weapon;	// make Weapon access the members of this class
	friend class Weapon1;
	friend class Weapon2;
	friend class Weapon3;
	friend class Weapon4;
	friend class Game;
};

///////////////////// Game class ////////////////////
#define SCROLL_SPEED 1

class Game {
public:
	Game();

	void playGame();

protected:
	GameMap map;
	int lastScroll;
	int lastMove;
};



#endif // __GAME_H