////////////////////////////////////////////////////////////////////////////
// Project:			ProjectZ Game
// Author:			Andrew Deren
// Date Started:	7/01/98
//
// File:			agui.h
////////////////////////////////////////////////////////////////////////////

#include <stdarg.h>
#include "defs.h"

#ifndef __AGUI_H
#define __AGUI_H


//////////////////////////////////////////////////////////////////////////////
// class status_object
//		Like text object but can set text via printf style
//////////////////////////////////////////////////////////////////////////////
#define STATUS_OBJECT_BUFFER_SIZE 500

class status_object : public dialog_object {
public:
	inline status_object() : dialog_object() { _text = new char[STATUS_OBJECT_BUFFER_SIZE]; clear();}
	~status_object() { delete [] _text;}

	// writes stuff to the buffer via printf style
	inline void write(va_list arg_list, ...) {
		va_list arg_ptr;
		char *format;

		va_start(arg_ptr, arg_list);
		format = (char*)arg_list;
		vsprintf(_text,format,arg_ptr);
		redraw();
	}	

	inline void clear(){ _text[0] = '\0'; redraw();}

	virtual void msg_draw() {
		text_mode(degui_light_shad_color);
		rectfill(screen, x(), y(), x2(), y2(), degui_light_shad_color);
		textout(screen, font, _text, x()+1, y()+1, degui_fore_color);
		_redraw = false;
	}

	char *getText(void) { return _text;}

protected:
	char* _text;		// buffer
};

extern status_object debug;


// forward declaration of classes
class select_bitmap_object;

//////////////////////////////////////////////////////////////////////////
// class close_dialog_object
//		Closes dialog when clicked on
//////////////////////////////////////////////////////////////////////////
class close_dialog_object : public dialog_object {
public:
	close_dialog_object(int theKey = 0) : dialog_object() { keyClose = theKey;}

	virtual void msg_click(void) { parent()->close(); }
	virtual bool msg_xchar(int k) { 
		if (k == keyClose) {
			parent()->close();
			return D_USED_CHAR;
		}
		else return D_O_K;
	}

protected:
	int keyClose;		// optionally can also use the key to close the dialog
};


//////////////////////////////////////////////////////////////////////////
// class h_select_bitmap_object
//		helper class for select_bitmap_object
//		used as a dialog object when poping up the 
//		list of graphics
//////////////////////////////////////////////////////////////////////////
class h_select_bitmap_object : public panel_raised_object {
public:
	h_select_bitmap_object(select_bitmap_object* s = NULL) : panel_raised_object() {
		_selectObject = s;
		_current = -1;
		_oldCurrent = -1;

		newColors.set_fore(makecol(255, 255, 255));
		newColors.set_back(makecol(0, 0, 0));
		newColors.set_select(makecol(0,0,255));

		color = &newColors;
	}

	virtual void msg_draw(void);
	virtual void msg_click(void);
	virtual void msg_start(void);
	
	inline void setCurrent(int cur) { _current = cur;}
	inline int getCurrent(void) const { return _current;}

	// checks whether mouse is inside the object
	inline bool hasMouse(void) {
		if (mouse_x > x() && mouse_x < x2() && mouse_y > y() && mouse_y < y2())
			return true;
		else return false;
	}


protected:
	select_bitmap_object *_selectObject;

	int _current;		// current tile selected
	int _oldCurrent;	// old current tile

	color_scheme newColors;	// colors for the object
};


//////////////////////////////////////////////////////////////////////////
// class select_bitmap_object
//		When clicked on executes a callback and
//		when clicked in the corner or right clicked
//		a dialog with bitmaps to select from popups
//		and puts selected bitmap as the main one
//////////////////////////////////////////////////////////////////////////
class select_bitmap_object : public dialog_object {
public:
	// constructor takes a pointer to datafile
	// start is the first bitmap to display and numGraphics is
	// number of images used from datafile
	select_bitmap_object(const char* name, DATAFILE *data, int start, int numGraphics);
	select_bitmap_object(void);

	// member access functions
	inline void setDatafile(DATAFILE *data) { _datafile = data;}
	inline DATAFILE *getDatafile(void) const { return _datafile;}

	inline void setStart(int start) { _start = start;}
	inline int getStart(void) const { return _start;}

	inline void setNumGraphics(int num) { _numGraphics = num;}
	inline int getNumGraphics(void) const { return _numGraphics;}

	inline void setCurrent(int current) { _current = current;}
	inline int getCurrent(void) const { return _current;}

	inline void setNumColumns(int cols) { _numCols = cols;}
	inline int getNumColumns(void) const { return _numCols;}

	inline void setName(const char* name) { strcpy(_name, name);}
	inline char* getName(void) { return _name;}

	inline void setGroup(const int group) { _group = group;}
	inline int getGroup(void) const { return _group;}

	// set callback functions
	inline void setClickedCallback(void (*callback)(select_bitmap_object*)) {
		cbClicked = callback;
	}

	inline void setChangedCallback(void (*callback)(select_bitmap_object*)) {
		cbChanged = callback;
	}

	
	// checks whether mouse is inside the object
	inline bool hasMouse(void) {
		if (mouse_x > x() && mouse_x < x2() && mouse_y > y() && mouse_y < y2())
			return true;
		else return false;
	}

	inline static void setFont(FONT* f) { _font = f;}
	inline static FONT* getFont(void) { return _font;}

	// virtual function from dialog_object specific to this object
	virtual void msg_draw(void);
	virtual void msg_click(void);
	virtual void msg_gotmouse(void);
	virtual void msg_lostmouse(void);
	virtual void msg_radio(int g);

	inline int getSelection(void) const { return _current - _start + 1;}

protected:
	DATAFILE *_datafile;		// datafile where graphics are taken from
	int _start;					// first graphics located in datafile
	int _numGraphics;			// number of graphics used

	char _name[26];				// name to display
	static FONT* _font;			// font to use to display name (same for all instances)
	int _group;					// group that the object belongs to
	int _current;				// currently selected graphic
	int _numCols;				// number of columns to display in dialog

	// callback function called when the button with graphic is clicked
	// pointer to the object is sent as a paramter to the function
	void (*cbClicked)(select_bitmap_object*);

	// callback function called when the graphic is changed
	void (*cbChanged)(select_bitmap_object*);

	// make h_select_bitmap_object be a friend
	// so that it can access all the members
	friend class h_select_bitmap_object;
};


////////////////////////////////////////////////////////////////////////////////////////
// class degui_colors_object
//		used to show the colors of degui
////////////////////////////////////////////////////////////////////////////////////////
class degui_colors_object : public dialog_object {
public:
	degui_colors_object() : dialog_object() {}

	virtual void msg_draw(void);
};


inline void msgbox(char *msg1, char *msg2 = NULL, char *msg3 = NULL) {
	alert(msg1, msg2, msg3, "OK", NULL, KEY_ESC, KEY_ENTER);
}



#endif