////////////////////////////////////////////////////////////////////////////
// Project:			ProjectZ Game
// Author:			Andrew Deren
// Date Started:	7/01/98
//
// File:			editor.h
////////////////////////////////////////////////////////////////////////////
#ifndef __EDITOR_H

#include "agui.h"
#include "defs.h"
#include "map.h"

class Editor;
class ScrollMapObject;

//////////////////////////////////////////////////////////////////////
// class EditorMap is for handling the drawing and other stuff for the
// map used in editor.
//////////////////////////////////////////////////////////////////////
class EditorMap : public Map, public dialog_object {
public:
	EditorMap() : Map(), dialog_object(), bmp(NULL), gridOn(true),
		gridColor(makecol(0, 160, 0)), scroll(NULL)
	{
		setSize(100);	// default map size is 100
	}
	virtual ~EditorMap() {}

	// event handlers inherited from dialog_object
	virtual void msg_start(void);
	virtual void msg_end(void);
	virtual void msg_click(void);
	virtual void msg_idle(void);
	virtual bool msg_xchar(int k);
	virtual void msg_draw(void);

	virtual void msg_getmouse(void) { redraw();}
	virtual void msg_lostmouse(void) { redraw();}
	virtual bool msg_wantfocus(void) { return true;}

	// checks whether mouse is in map area
	inline isInMap(void) { 
		if (mouse_x > x() && mouse_x < x2() && mouse_y > y() && mouse_y < y2())
			return true;
		else return false;
	}

	// sets the tileX and tileY according to the position of the mouse
	// it's relative to the screen not currentX.
	// sets tileX-Y to -1 if mouse is out of map
	inline bool setTiles(void) {
		if (!isInMap()) {
			tileX = -1;
			tileY = -1;
			return false;
		}
		tileX = (mouse_x - (x() + 1)) / TILE_SIZE;
		tileY = (mouse_y - (y() + 1)) / TILE_SIZE;
		return true;
	}

	// set and get status object which displays info about map
	void setStatusObject(status_object* s) { status = s;}
	status_object* getStatusObject(void) { return status;}
	
	// set and get scroll map object
	void setScrollMapObject(ScrollMapObject* s) { scroll = s;}
	ScrollMapObject* getScrollMapObject(void) { return scroll;}

	// filename is used for saving and loading of a map
	void setFilename(const char *newFile) { strcpy(filename, newFile);}
	char *getFilename(void) { return filename;}

	// when fileToSave or fileToLoad is NULL then the current filename is used
	bool saveMap(char* fileToSave = NULL);
	virtual bool loadMap(char* fileToLoad = NULL);

	bool newMap(int size);	// create new map
	bool resize(int size);	// resize current map keeping old values

	inline void toggleGrid(void) { if (gridOn) gridOn = false; else gridOn = true; redraw();}

	virtual void redraw(void);

protected:
	friend class ScrollMapObject;

	BITMAP *bmp;		// double buffer where stuff is drawn

	int visible;		// number of tiles visible at one time

	int tileX;			// current X tile
	int tileY;			// current Y tile
	int oldX;			// old x tile
	int oldY;			// old y tile

	bool gridOn;		// is grid on?
	int gridColor;		// color of the grid

	char filename[256];	// file name of the map

	status_object* status;	// object that displays the status
	ScrollMapObject *scroll;	// object used for scrolling the map

	// displays information in status object
	void drawStatus(void);
	void drawGrid(void);

	// pretty put tile checking the tiles next to it
	bool isEmpty(int xloc, int yloc);
	void putTile(int xloc, int yloc, int tile);

	// scroll map to the left when draging 
	inline void scrollLeft() {
		if (mouse_x < x() && currentX > 0) {
			currentX--;
			redraw();
			parent()->dialog_message(MSG_IDLE, 0);
			rest(50);
		}
	}
	// scroll map to the right
	inline void scrollRight() {
		if (mouse_x > x2() && currentX+X_TILES<sizeX) {
			currentX++;
			redraw();
			parent()->dialog_message(MSG_IDLE, 0);
			rest(50);
		}
	}

	enum PUT_MODE {P_NONE, P_TILE, P_OBJECT, P_UNIT};

	// puts a planet onto the map
	void putObject(int x, int y, int which);

	// finds the object at x and y and returns the position of that
	// object in the array, returns -1 otherwise
	int findObject(int x, int y);


	// display object properties dialog box
	void objectProperties(int objectID);
};

//////////////////////////////////////////////////////////////////////
// class ScrollMapObject, displays the scrollbar at the bottom
// of EditorMap used for scrolling
//////////////////////////////////////////////////////////////////////
class ScrollMapObject : public dialog_object {
public:
	ScrollMapObject(EditorMap* edMap) : map(edMap), start(0), current(0) {}

	virtual void msg_click(void);
	virtual void msg_draw(void);
protected:
	EditorMap *map;
	int start;
	int current;
};

//////////////////////////////////////////////////////////////////////
// class Editor, holds the ojbects used by the editor
//////////////////////////////////////////////////////////////////////
class Editor  
{
public:
	Editor();
	virtual ~Editor() {}

	void execute();
	void exit() { edtDialog.close();}

	inline void saveMap(char *file = NULL) { if (!map.saveMap(file)) msgbox("Error Saving map.");}
	inline void loadMap(char *file = NULL) { if (!map.loadMap(file)) msgbox("Error Loading map.");}
	inline void newMap(int size) { if (!map.newMap(size)) msgbox("Error creating new map.");}
	inline void resizeMap(int size) { if (!map.resize(size)) msgbox("Error resizing map.");}

	inline int getMapSize(void) const { return map.getSize();}
	inline void toggleGrid(void) { map.toggleGrid();}
protected:
	friend class EditorMap;

	// objects
	clear_screen_object clrScreen;	// clear screen object
	dialog edtDialog;				// main dialog
	select_bitmap_object selTiles;		// object to select tiles
	select_bitmap_object selObjects;	// object to select planets
	status_object status;			// status object for the map
	EditorMap map;					// the map
	ScrollMapObject scroll;			// used for scrolling the map

};

extern Editor *editor;

void startEditor(void);


#endif // __EDITOR_H
