////////////////////////////////////////////////////////////////////////////
// Project:			ProjectZ Game
// Author:			Andrew Deren
// Date Started:	7/01/98
//
// File:			editor.cpp - editor object
////////////////////////////////////////////////////////////////////////////

#include "Editor.h"
#include "defs.h"

// editor object
Editor *editor;

// debug object
status_object debug;

//////////////////////////////////////////////////////////////////////
// class EditorMap is for handling the drawing and other stuff for the
// map used in editor.
//////////////////////////////////////////////////////////////////////

void EditorMap::redraw(void) 
{ 
	dialog_object::redraw(); 
	if (scroll) 
		scroll->redraw();
}

///// when dialog started set it's size
void EditorMap::msg_start(void)
{
	//set width
	set_w(MAP_WIDTH + 2);
	set_h(MAP_HEIGHT + 2);

	bmp = create_bitmap(w(), h());

	tileX = 0;
	tileY = 0;
	oldX = -1;
	oldY = -1;

	strcpy(filename, "untitled.map");
}

//// when object is done deallocate the bmp
void EditorMap::msg_end(void)
{
	destroy_bitmap(bmp);
}

//// handle mouse clicks
void EditorMap::msg_click(void)
{
	int temp;
	
	if (!mouse_b) 
		return;
	
	// first check what we are putting
	int putMode;
	
	if (editor->selTiles.selected()) 
		putMode = P_TILE;
	else if (editor->selObjects.selected())
		putMode = P_OBJECT;
	else putMode = P_NONE;
	
	if (!setTiles()) return;
	
	if (putMode == P_TILE) {
		int vtile = editor->selTiles.getCurrent();
		int toldX = -1;
		int toldY = -1;
		
		while (mouse_b) {
			temp = setTiles();
			
			if (!temp) {
				if (mouse_x < x() && currentX > 0) 
					scrollLeft();
				if (mouse_x > x2() && currentX+X_TILES<sizeX) 
					scrollRight();
				
				continue;
			}
			else {
				if (toldX != tileX || toldY != tileY) {
					toldX = tileX;
					toldY = tileY;
					if (mouse_b & 1) {
						putTile(currentX+tileX,		tileY,		vtile);
						putTile(currentX+tileX,		tileY-1,	-1);
						putTile(currentX+tileX+1,	tileY-1,	-1);
						putTile(currentX+tileX+1,	tileY,		-1);
						putTile(currentX+tileX+1,	tileY+1,	-1);
						putTile(currentX+tileX,		tileY+1,	-1);
						putTile(currentX+tileX-1,	tileY+1,	-1);
						putTile(currentX+tileX-1,	tileY,		-1);
						putTile(currentX+tileX-1,	tileY-1,	-1);
						
					}
					//line[currentX+tileX].tile[tileY] = vtile;
					else 
						line[currentX+tileX].tile[tileY] = 0;
					
					redraw();
					parent()->dialog_message(MSG_IDLE, 0);
				}
			}
			//debug.write("%d,%d - %d,%d", tileX, tileY, oldX, oldY);
		} // end of while (mouse_b)
	} // end of if putMode == P_TILE

	// putting an object
	else if (putMode == P_OBJECT) {
		setTiles();
		int objectID = findObject(currentX+tileX, tileY);

		// if left mouse button pressed put planet there
		// and if there is no other object in that place
		if (mouse_b & 1) {
			if (objectID != -1)
				return;

			int objType = editor->selObjects.getCurrent();
			if (setTiles()) {
				putObject(currentX + tileX, tileY, objType);
			}
			while (mouse_b) {}
		}
		else {
			if (objectID == -1) 
				return;				
			objectProperties(objectID);
		}
	}
}

///// handle idle periods
void EditorMap::msg_idle(void)
{
	if (setTiles()) {
		if (oldX != tileX || oldY != tileY) {
			dialog_object::redraw();
			oldX = tileX;
			oldY = tileY;
		}
	}
}


////// handle key presses
bool EditorMap::msg_xchar(int k)
{
	k = k >> 8;
	switch (k) {
	case KEY_LEFT: 
		if (currentX > 0) { 
			currentX--; 
			redraw();
		}; 
		return true;
	case KEY_RIGHT: 
		if (currentX+X_TILES<sizeX) {
			currentX++; 
			redraw();
		} 
		return true;
	case KEY_HOME: 
		currentX = 0; 
		redraw(); 
		return true;
	case KEY_END: 
		currentX = sizeX - X_TILES; 
		redraw(); 
		return true;
	case KEY_PGUP: 
		currentX += X_TILES; 
		if (currentX + X_TILES > sizeX) 
			currentX = sizeX - X_TILES;
		redraw();
		return true;
	case KEY_PGDN: 
		currentX -= X_TILES;
		if (currentX < 0)
			currentX = 0;
		redraw();
		return true;
		default: return false;
	}
}

//// handle drawing of the map
void EditorMap::msg_draw(void)
{
	draw_3d_frame(bmp, 0, 0, w()-1, h()-1, degui_fore_color, degui_dark_shad_color,degui_back_color);		

	int i, column, row;
	// draw the actual map
	int t;

	// draw all visible planets
	for (i=0; i<numObjects; i++) {
		if (objects[i].x > currentX + TILE_SIZE)
			break;

		if (objects[i].x >= currentX)
			draw_sprite(bmp, (BITMAP*)datObjects[objects[i].type * 4 + objects[i].size].dat,
						(objects[i].x - currentX)*TILE_SIZE, objects[i].y * TILE_SIZE);
	}

	for (i=0, column = currentX; i < X_TILES && column < sizeX; column++, i++) {
		for (row = 0; row < Y_TILES; row++) {
			t = line[column].tile[row];
			if (t > 0 && t < 155)
				draw_sprite(bmp, (BITMAP*)datTiles[t].dat, i*TILE_SIZE, row*TILE_SIZE);
		}
	}

	drawGrid();
	rect(bmp, tileX*TILE_SIZE, tileY*TILE_SIZE, tileX*TILE_SIZE+TILE_SIZE, tileY*TILE_SIZE+TILE_SIZE, makecol(0, 0, 255));
	blit(bmp, screen, 0, 0, x(), y(), bmp->w, bmp->h);

	drawStatus();
}


////// show some info about map //////////////////
void EditorMap::drawStatus(void)
{
	if (!status) return;

	status->write("%s - current: %d (%d)    %d,%d", get_filename(filename), currentX, sizeX, tileX, tileY);
}

/////// draws the grid /////////////////////////////
void EditorMap::drawGrid(void)
{
	if (!gridOn) return;

	int i;
	for (i=1; i<X_TILES; i++)
		vline(bmp, i*TILE_SIZE, 1, h()-2, gridColor);
	for (i=1; i<Y_TILES; i++)
		hline(bmp, 1, i*TILE_SIZE, w()-2, gridColor);

}

////////// save a map ///////////////////////////////
bool EditorMap::saveMap(char* fileToSave)
{
	// if passed fileToSave then used that file name to save the map
	char *fileName;
	if (fileToSave)
		fileName = fileToSave;
	else fileName = filename;

	FILE *file = fopen(fileName, "wb");
	if (!file) return false;

	// write id and version of the map
	fwrite(FILE_ID, sizeof(char), 5, file);
	fwrite(&FILE_VERSION, sizeof(int), 1, file);
	
	// now write map data
	fwrite(&sizeX, sizeof(int), 1, file);	// number of lines

	// write line by line
	int x, y;
	for (x = 0; x < sizeX; x++) 
		fwrite(line[x].tile, sizeof(char), Y_TILES, file);

	// save objects (planets)
	fwrite(&numObjects, sizeof(int), 1, file);
	fwrite(objects, sizeof(MapObject), numObjects, file);

	fclose(file);

	if (fileToSave)
		setFilename(fileToSave);
	
	redraw();


	fclose(file);

	return true;
}

////////// load a map ///////////////////////////////
bool EditorMap::loadMap(char* fileToLoad)
{
	// if passed fileToLoad then used that file name to load the map
	char *fileName;
	if (fileToLoad)
		fileName = fileToLoad;
	else fileName = filename;

	FILE *file = fopen(fileName, "rb");
	if (!file) {
		debug.write("Cannot open file.");
		return false;
	}

	int checkVersion;	// used to read the verion
	char checkID[5];	// used to check the file id
	int width;			// size of map

	// write id and version of the map
	fread(checkID, sizeof(char), 5, file);
	fread(&checkVersion, sizeof(int), 1, file);

	if (strcmp(checkID, FILE_ID) != 0 || checkVersion != FILE_VERSION) {
		debug.write("Invalid id or version.");
		return false;
	}
	
	// now write map data
	fread(&width, sizeof(int), 1, file);	// number of lines

	if (!setSize(width)) {
		debug.write("Invalid size.");
		return false;
	}

	// write line by line
	int x, y;
	for (x = 0; x < sizeX; x++) 
		fread(line[x].tile, sizeof(char), Y_TILES, file);

	// save objects (planets)
	fread(&numObjects, sizeof(int), 1, file);
	if (numObjects >= maxObjects) {
		delete [] objects;
		maxObjects = numObjects * 2;
		objects = new MapObject[maxObjects];
	}
	fread(objects, sizeof(MapObject), numObjects, file);


	fclose(file);

	if (fileToLoad)
		setFilename(fileToLoad);

	redraw();

	return true;
}

// create new map
bool EditorMap::newMap(int size)
{
	if (!setSize(size))
		return false;

	tileX = 0;
	tileY = 0;
	currentX = 0;
	strcpy(filename, "untitled.map");
	oldX = -1;
	oldY = -1;
	numObjects = 0;

	return true;
}

// resize current map keeping old values
bool EditorMap::resize(int size)	
{
	int i;
	int oldSize = sizeX;
	// copy the values into temp array of MapLine's
	MapLine* newLine = new MapLine[sizeX];
	for (i=0; i<sizeX; i++)
		newLine[i] = line[i];

	// resize current line
	if (!setSize(size)) {
		delete [] newLine;
		return false;
	}

	for (i=0; i<oldSize && i<sizeX; i++) 
		line[i] = newLine[i];

	currentX = 0;
	redraw();

	return true;
}

// return true if the cell is non-occupied and true
// if the cell is occupied or out of bounds
bool EditorMap::isEmpty(int xloc, int yloc)
{
	if (xloc < 0 || yloc < 0 || xloc >= sizeX || yloc >= Y_TILES)
		return false;
	else {
		if (line[xloc].tile[yloc] == 0) return true;
		else return false;
	}
}

// pretty put tile
// if tile is -1 get the tile from the passed cell location
void EditorMap::putTile(int xloc, int yloc, int tile)
{
	bool temp;
	if (tile == -1)
		temp = false;
	else temp = true;

	if (tile == -1) {
		if (xloc < 0 || yloc < 0 || xloc >= sizeX || yloc >= Y_TILES)
			tile = 0;
		else tile = line[xloc].tile[yloc];
	}

	if (tile >= TILE12_00_BMP && tile <= TILE12_12_BMP) {
		int newtile;

		/*      x1
			  x4  x2
			    x3
		*/
		int x1 = isEmpty(xloc, yloc-1);
		int x2 = isEmpty(xloc+1, yloc);
		int x3 = isEmpty(xloc, yloc+1);
		int x4 = isEmpty(xloc-1, yloc);

		if (temp)
			debug.write("%d %d %d %d", x1, x2, x3, x4);

		// if has squares on all sides
		if (!x1 && !x2 && !x3 && !x4) {
			/*		x1  x2
					
					x4  x3
			*/
			x1 = isEmpty(xloc-1, yloc-1);
			x2 = isEmpty(xloc+1, yloc-1);
			x3 = isEmpty(xloc+1, yloc+1);
			x4 = isEmpty(xloc-1, yloc+1);

			if (temp)
				debug.write("%d %d %d %d", x1, x2, x3, x4);

			if (x1)
				newtile = TILE12_07_BMP;
			else if (x2)
				newtile = TILE12_08_BMP;
			else if (x3)
				newtile = TILE12_05_BMP;
			else if (x4)
				newtile = TILE12_06_BMP;
			else newtile = TILE12_00_BMP;
		}
		else if (!x1 && x2 && !x3 && !x4) // 1 0 1 1
			newtile = TILE12_01_BMP;
		else if (!x1 && !x2 && x3 && !x4) // 1 1 0 1
			newtile = TILE12_02_BMP;
		else if (!x1 && !x2 && !x3 && x4) // 1 1 1 0
			newtile = TILE12_03_BMP;
		else if (x1 && !x2 && !x3 && !x4) // 0 1 1 1
			newtile = TILE12_04_BMP;

		else if (x1 && !x2 && !x3 && x4) // 0 1 1 0
			newtile = TILE12_09_BMP;
		else if (x1 && x2 && !x3 && !x4) // 0 0 1 1
			newtile = TILE12_10_BMP;
		else if (!x1 && x2 && x3 && !x4) // 1 0 0 1
			newtile = TILE12_11_BMP;
		else if (!x1 && !x2 && x3 && x4) // 1 1 0 0
			newtile = TILE12_12_BMP;
		else newtile = TILE12_13_BMP;
		
		line[xloc].tile[yloc] = newtile;
	}
	else
		line[xloc].tile[yloc] = tile;
}

// put which planet at x and y location of the map
void EditorMap::putObject(int xLoc, int yLoc, int which)
{
	int i;
	
	// if the array of objects if already full, create new one
	if (numObjects == maxObjects) {
		MapObject *tArray = objects;
		objects = new MapObject[maxObjects*2];
		for (i=0; i<numObjects; i++)
			objects[i] = tArray[i];

		maxObjects *= 2;
	}

	for (i=0; i<numObjects; i++) {
		// found new location for the object
		if (xLoc < objects[i].x) {
			int j;
			for (j=numObjects; j>i;j--) 
				objects[j] = objects[j-1];
			break;
		}
	}
	objects[i].x = xLoc;
	objects[i].y = yLoc;
	objects[i].size = 1;
	objects[i].level = 1;
	objects[i].type = which;

	numObjects++;

	// Object definitions - DEGUI 1.24 beta
	char sX[5];
	char sY[5];
	char sL[2];
	char sS[2];
	sprintf(sX, "%d", xLoc);
	sprintf(sY, "%d", yLoc);
	sprintf(sL, "%d", 1);
	sprintf(sS, "%d", 1);

	close_dialog_object clsObject;
	panel_raised_object object_1;
	text_object lblX("x");
	text_object lblY("y");
	text_object lblSize("size");
	text_object lblLevel("level");
	edittext_object edtX(sX,4);
	edittext_object edtY(sY,4);
	edittext_object edtSize(sS,1);
	edittext_object edtLevel(sL,1);
	bitmap_object bmpView((BITMAP*)datEdObj[which].dat);
	winbutton_object cmdClose("OK", KEY_ESC, true);

	int locX = (xLoc - currentX) * TILE_SIZE + x();
	int locY = (yLoc * TILE_SIZE) + y();

	// Add Objects to the dialog
	dialog the_dialog;
	the_dialog.add(clsObject, 0, 0, SCREEN_W, SCREEN_H);
	the_dialog.add(object_1,	locX,locY,90,105);
	the_dialog.add(lblX,		locX+5,locY+10,8,8); 
	the_dialog.add(lblY,		locX+5,locY+25,8,8);
	the_dialog.add(lblSize,		locX+5,locY+40,32,8);
	the_dialog.add(lblLevel,	locX+5,locY+55,40,8);
	the_dialog.add(edtX,		locX+45,locY+5,40,13);
	the_dialog.add(edtY,		locX+45,locY+20,40,13);
	the_dialog.add(edtSize,		locX+45,locY+35,40,15);
	the_dialog.add(edtLevel,	locX+45,locY+50,40,15);
	the_dialog.add(bmpView,		locX+15,locY+70,32,32);
	the_dialog.add(cmdClose,	locX+60, locY+88, 25, 14);

	if (mouse_b)
		while (mouse_b) {}

	the_dialog.popup();

	int newX = atoi(edtX.get_text());
	int newY = atoi(edtY.get_text());
	int newL = atoi(edtLevel.get_text());
	int newS = atoi(edtSize.get_text());

	if (newX >= 0 && newX < sizeX)
		objects[i].x = newX;
	if (newY >= 0 && newY < Y_TILES)
		objects[i].y = newY;
	if (newL > 0 && newL < 4)
		objects[i].level = newL;
	if (newS >= 0 && newS < 4)
		objects[i].size = newS;
}

// finds the object at x and y and returns the position of that
// object in the array, returns -1 otherwise
int EditorMap::findObject(int x, int y)
{
	int i;
	for (i=0; i<numObjects; i++) 
		if (objects[i].x == x && objects[i].y == y)
			return i;
	return -1;
}

// display object properties dialog box
void EditorMap::objectProperties(int objectID)
{
	int xLoc = objects[objectID].x;
	int yLoc = objects[objectID].y;
	int level = objects[objectID].level;
	int size = objects[objectID].size;
	int type = objects[objectID].type;

	// Object definitions - DEGUI 1.24 beta
	char sX[5];
	char sY[5];
	char sL[2];
	char sS[2];
	sprintf(sX, "%d", xLoc);
	sprintf(sY, "%d", yLoc);
	sprintf(sL, "%d", level);
	sprintf(sS, "%d", size);

	close_dialog_object clsObject;
	panel_raised_object object_1;
	text_object lblX("x");
	text_object lblY("y");
	text_object lblSize("size");
	text_object lblLevel("level");
	edittext_object edtX(sX,4);
	edittext_object edtY(sY,4);
	edittext_object edtSize(sS,1);
	edittext_object edtLevel(sL,1);
	bitmap_object bmpView((BITMAP*)datEdObj[type].dat);
	winbutton_object cmdDelete("Del", KEY_DEL, true);
	winbutton_object cmdClose("OK", KEY_ESC, true);

	int locX = (xLoc - currentX) * TILE_SIZE + x();
	int locY = (yLoc * TILE_SIZE) + y();

	// Add Objects to the dialog
	dialog the_dialog;
	the_dialog.add(clsObject, 0, 0, SCREEN_W, SCREEN_H);
	the_dialog.add(object_1,	locX,locY,90,105);
	the_dialog.add(lblX,		locX+5,locY+10,8,8); 
	the_dialog.add(lblY,		locX+5,locY+25,8,8);
	the_dialog.add(lblSize,		locX+5,locY+40,32,8);
	the_dialog.add(lblLevel,	locX+5,locY+55,40,8);
	the_dialog.add(edtX,		locX+45,locY+5,40,13);
	the_dialog.add(edtY,		locX+45,locY+20,40,13);
	the_dialog.add(edtSize,		locX+45,locY+35,40,15);
	the_dialog.add(edtLevel,	locX+45,locY+50,40,15);
	the_dialog.add(bmpView,		locX+15,locY+70,32,32);
	the_dialog.add(cmdDelete,	locX+60, locY+70, 25, 14);
	the_dialog.add(cmdClose,	locX+60, locY+88, 25, 14);

	if (mouse_b)
		while (mouse_b) {}

	dialog_object* retType = the_dialog.popup();

	if (retType == &cmdClose) {
		int newX = atoi(edtX.get_text());
		int newY = atoi(edtY.get_text());
		int newL = atoi(edtLevel.get_text());
		int newS = atoi(edtSize.get_text());

		if (newX >= 0 && newX < sizeX)
			objects[objectID].x = newX;
		if (newY >= 0 && newY < Y_TILES)
			objects[objectID].y = newY;
		if (newL > 0 && newL < 4)
			objects[objectID].level = newL;
		if (newS >= 0 && newS < 4)
			objects[objectID].size = newS;
	}
	else if (retType == &cmdDelete) {
		// deleting the object, shift all above it by one down
		int i;
		for (i=objectID; i<numObjects-1; i++)
			objects[i] = objects[i+1];
		numObjects--;
	}
}

//////////////////////////////////////////////////////////////////////
// class ScrollMapObject, displays the scrollbar at the bottom
// of EditorMap used for scrolling
//////////////////////////////////////////////////////////////////////
void ScrollMapObject::msg_click(void)
{


}

void ScrollMapObject::msg_draw(void)
{
	current = map->currentX;
	start = 0;

	// find which page of the view are we using
	if (map->sizeX > MAP_WIDTH) {
		if (current < MAP_WIDTH / 2)
			start = 0;
		else if (map->sizeX - current < MAP_WIDTH / 2) 
			start = map->sizeX - MAP_WIDTH;
		else 
			start = current - (MAP_WIDTH / 2);

	}

	text_mode(degui_light_shad_color);
	sprintf(textBuffer, "%04d", start);
	textout(screen, font, textBuffer, x(), y()-12, makecol(0, 0, 255));
	sprintf(textBuffer, "%04d", start+MAP_WIDTH);
	textout(screen, font, textBuffer, x2()-30, y()-12, makecol(0, 0, 255));

	int i, j;
	int visible = w()-2;
	draw_3d_frame(screen, x(), y(), x2(), y2(), degui_dark_shad_color, degui_dark_shad_color, degui_back_color);	

	for (i = 0; i<visible && i < map->sizeX; i++) 
		for (j=0; j<Y_TILES; j++)
			putpixel(screen, x()+1+i, y()+1+j, map->line[i+start].tile[j] == 0 ? degui_fore_color : degui_back_color);

	rect(screen, x()+current+1-start, y()+1, x()+current+X_TILES+1-start, y()+1+Y_TILES, makecol(0, 0, 255));
}

//////////////////////////////////////////////////////////////////////
// callback functions
//////////////////////////////////////////////////////////////////////
int cbkExitEditor(void)
{
	exit(0);
}

int cbkSaveMap(void)
{
	editor->saveMap();
}

int cbkLoadMap(void)
{
	//open file dialog
	textBuffer[0] = '\0';

	if (file_select("Save Map (*.map)", textBuffer, "map") != 0) {
		editor->loadMap(textBuffer);
	}
}

int cbkSaveAsMap(void)
{
	//open file dialog
	textBuffer[0] = '\0';

	if (file_select("Save Map (*.map)", textBuffer, "map") != 0) {
		editor->saveMap(textBuffer);
	}
}

int cbkResizeMap(void)
{
	char newSize[5];
	sprintf(newSize, "%d", 0);
	sprintf(textBuffer, "%d", editor->getMapSize());

   // Object definitions - DEGUI 1.24 beta
   panel_raised_object window;
   text_object txtNewSize("New Size:");
   text_object txtCurrentSize("Current Size:");
   text_object txtCurSize(textBuffer);
   edittext_object edtNewSize(newSize,4);
   winbutton_object cmdOK("OK",0,true);
   winbutton_object cmdCancel("Cancel",0,true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(window,100,100,185,110);
   the_dialog.add(txtNewSize,110,160,72,8);
   the_dialog.add(txtCurrentSize,110,130,104,8);
   the_dialog.add(txtCurSize,225,130,24,8);
   the_dialog.add(edtNewSize,225,155,40,13);
   the_dialog.add(cmdOK,120,180,65,20);
   the_dialog.add(cmdCancel,200,180,70,20);
   the_dialog.center();
   dialog_object* res = the_dialog.popup();

   if (res == &cmdOK) {
		int size = atoi(edtNewSize.get_text());
		editor->resizeMap(size);
   }
}

int cbkNewMap(void)
{
	sprintf(textBuffer, "%d", 100);

   // Object definitions - DEGUI 1.24 beta
   panel_raised_object window;
   text_object txtNewSize("Map Size:");
   edittext_object edtNewSize(textBuffer,4);
   winbutton_object cmdOK("OK",0,true);
   winbutton_object cmdCancel("Cancel",0,true);

   // Add Objects to the dialog
   dialog the_dialog;
   the_dialog.add(window,100,100,185,110);
   the_dialog.add(txtNewSize,110,160,72,8);
   the_dialog.add(edtNewSize,225,155,40,13);
   the_dialog.add(cmdOK,120,180,65,20);
   the_dialog.add(cmdCancel,200,180,70,20);
   the_dialog.center();
   dialog_object* res = the_dialog.popup();

   if (res == &cmdOK) {
		int size = atoi(edtNewSize.get_text());
		editor->newMap(size);
   }
}


int cbkGrid(void)
{
	editor->toggleGrid();
}

//////////////////////////////////////////////////////////////////////
// class Editor, holds the ojbects used by the editor
//////////////////////////////////////////////////////////////////////
Editor::Editor()
:selTiles("Tile", datTiles, TILE12_00_BMP, TILE12_13_BMP),
 selObjects("Obj", datEdObj, EDOBJ01, EDOBJ04+1),
 scroll(&map)
{
}

void Editor::execute()
{
	// do menus
	menu mnuFile;
	menu mnuEdit;
	menu mnuHelp;
	menu mnuMainMenu;

	degui_colors_object showColors;

	// build file menu
	mnuFile.add("New", &cbkNewMap);
	mnuFile.add("Load", &cbkLoadMap);
	mnuFile.add("Save", &cbkSaveMap);
	mnuFile.add("Save As", &cbkSaveAsMap);
	mnuFile.add();
	mnuFile.add("Exit Editor", &cbkExitEditor);

	mnuEdit.add("Resize", &cbkResizeMap);
	mnuEdit.add("Toggle Grid", &cbkGrid);

	// build help menu
	mnuHelp.add("Help", NULL);
	mnuHelp.add();
	mnuHelp.add("About", NULL);

	mnuMainMenu.add("&File", mnuFile);
	mnuMainMenu.add("&Edit", mnuEdit);
	mnuMainMenu.add("&Help", mnuHelp);

	menu_object dialogMenu(mnuMainMenu);     //main menu

	selTiles.setGroup(1);
	selObjects.setGroup(1);
	selTiles.select();
	selObjects.setNumColumns(4);

	selTiles.setFont((FONT*)datMain[FNT_EDITOR].dat);

	map.setStatusObject(&status);
	map.setScrollMapObject(&scroll);

	edtDialog.add(clrScreen);
	edtDialog.add(dialogMenu);
	edtDialog.add(map, 4, 16, 0, 0);
	edtDialog.add(scroll, 4, 480, MAP_WIDTH, Y_TILES+3);
	edtDialog.add(selTiles, 600, 16, 34, 48);
	edtDialog.add(selObjects, 600, 76, 34, 48);
	edtDialog.add(status, 4, 500, MAP_WIDTH, 12);
	edtDialog.add(debug, 200, 1, 300, 12);
	edtDialog.add(showColors, 660, 16, 100, 100);

	edtDialog.execute();

}

void startEditor(void)
{
	Editor mainEditor;
	editor = &mainEditor;

	editor->execute();
}