// Space Invaders Game
// written by:
// Costas Vandikas (vandikas@csd.uoc.gr)
// Stavros Michael (michael@csd.uoc.gr)

#pragma once
#include "gamesession.h"
#include "statesfacade.cpp"

namespace Application
{
	namespace Sessions
	{
		GameSession::GameSession( Levels Level ):
		_level( NULL ), _playerPanel( new PlayerPanel() )
		{
			setLevel( Level );
		}

		void GameSession::InitLevel()
		{
			_scene = new GameScene( _level->getAlienType() );
			_platformManager = new PlatformManager();
			_bulletManager = new BulletManager();
			_alienManager = new AlienManager( _level->getBulletChanceLevelMultiplier(), _level->getAlienMoveDelay() );
			_bonusShipManager = new BonusShipManager( _level->getBonusShips() );
		}

		Level* GameSession::getLevel() const
		{
			return _level;
		}

		void GameSession::EndLevel()
		{
			delete _scene;
			delete _alienManager;
			delete _bulletManager;
			delete _bonusShipManager;
			delete _platformManager;
		}

		void GameSession::Destruct()
		{
			EndLevel();
			delete _playerPanel;
		}

		void GameSession::setLevel( Levels LevelID )
		{
			Level* next = LevelLoader::getInstance()->Load( LevelID );
			if ( next != NULL )
			{
				if ( _level != NULL )
					EndLevel();
				_level = next;
				InitLevel();
				if ( _level != 0 )
					_playerPanel->getScoreBoard()->changeScore( LevelHit );
				SessionManager::getInstance()->DrawAnnouncement( _level->getBitmap() );
				play_midi( SoundDataFile::getInstance()->getMidi(won), false );
				Pause();
			}
			else
			{
				play_midi( SoundDataFile::getInstance()->getMidi(won), false );
				SessionManager::getInstance()->setNextSession( new MainMenuSession() );
			}
		}	

		void GameSession::setNextLevel()
		{
			int id = _level->getID();
			setLevel( (Levels)++id );
		}

		void GameSession::Pause()
		{
			SessionManager::getInstance()->Pause();
		}

		void GameSession::HandleUserInput()
		{
			Player* player = _scene->getPlayer();
			
			if ( key[KEY_ESC] )
			{
				SessionManager::getInstance()->DrawAnnouncement( ScreenDataFile::getInstance()->getScreen( Sure ) );
				SessionManager::getInstance()->StopTimer();
				SessionManager::getInstance()->ResetKeyboard();
				do
				{
					if ( key[KEY_Y] )
					{
						SessionManager::getInstance()->StartTimer();
						SessionManager::getInstance()->setNextSession( new MainMenuSession() );
						break;
					}

					if ( key[KEY_N] )
					{
						SessionManager::getInstance()->StartTimer();
						SessionManager::getInstance()->ResetKeyboard();
						break;
					}
				}
				while ( true );
			}
			
			switch( player->getActorState() )
			{
				case PlayerAlive:
				{
					if ( key[KEY_LEFT] ) 
						player->Move( MoveLeft );

					if ( key[KEY_RIGHT] )
						player->Move( MoveRight );

					if ( key[KEY_SPACE] )
						_bulletManager->addPlayerBullet( player->getBulletStartLocation() );
					else
						_bulletManager->setPlayerFired(false);
					break;
				}
				case PlayerDead:
				{
					_playerPanel->DecreasePlayerLives();
					if ( _playerPanel->getPlayerLives() != 0 )
					{
						SessionManager::getInstance()->DrawAnnouncement( ScreenDataFile::getInstance()->getScreen( PressAny ) );
						Pause();
						player->setActorState( PlayerAlive );
					}
					else
					{
						SessionManager::getInstance()->DrawAnnouncement( ScreenDataFile::getInstance()->getScreen( GameOver ) );
						SessionManager::getInstance()->ResetKeyboard();
						play_midi( SoundDataFile::getInstance()->getMidi(lost), false );
						readkey();

						//list<ScoreElement>* scoreList = HighscoreManager::getInstance()->getScoreList();
						//scoreList->sort();
						//scoreList->reverse();
						//list<ScoreElement>::iterator iterator = scoreList->end();
						//if ( ScoreBoard::getInstance()->getCurrentScore() >= iterator->getScore() )
							//SessionManager::getInstance()->setNextSession( new InputSession() );
						SessionManager::getInstance()->setNextSession( new MainMenuSession() );
					}

					break;
				}
			}
		}

		void GameSession::HandleMovement()
		{
			_alienManager->HandleMovement();
			_bonusShipManager->HandleMovement();
			_bulletManager->HandleMovement();
			_scene->getPlayer()->Move();
		}

		void GameSession::PopulateRenderingBuffer()
		{
			_scene->PopulateRenderingBuffer( RenderingEngine::getInstance()->getBuffer() );
			_playerPanel->Render( RenderingEngine::getInstance()->getBuffer() );
		}

		BulletManager* GameSession::getBulletManager() const
		{
			return _bulletManager;
		}

		AlienManager* GameSession::getAlienManager() const
		{
			return _alienManager;
		}

		BonusShipManager* GameSession::getBonusShipManager() const
		{
			return _bonusShipManager;
		}

		PlatformManager* GameSession::getPlatformManager() const
		{
			return _platformManager;
		}

		Scene* GameSession::getScene() const
		{
			return _scene;
		}

		PlayerPanel* GameSession::getPlayerPanel() const
		{
			return _playerPanel;
		}
	}
}