#include <stdio.h>
#include "wave.h"
#include "random.h"
#include "vector.h"
#include "globals.h"

wave_element wavetable[] =
{
    // name,             choppers, troopers, interval, bonus
    { "wave one",           3,  4,     1, 3,   20, 30, 100,  200 },
    { "wave two",           4,  5,     1, 3,   19, 29, 110,  250 },
    { "wave three",         5,  6,     1, 4,   18, 28, 120,  300 },
    { "wave four",          7,  8,     1, 5,   17, 27, 130,  350 },
    { "wave five",          9, 10,     2, 5,   16, 25, 140,  400 },
    { "wave six",          11, 13,     2, 6,   15, 24, 150,  450 },
    { "wave seven",        14, 16,     2, 7,   14, 23, 160,  500 },
    { "wave eight",        17, 19,     2, 8,   13, 22, 170,  550 },
    { "wave nine",         20, 21,     3, 8,   12, 20, 180,  600 },
    { "wave ten",          20, 22,     3, 8,   11, 19, 190,  650 },
    { "wave eleven",       20, 23,     3, 8,   10, 18, 200,  700 },
    { "wave twelve",       20, 24,     3, 8,    9, 17, 220,  750 },
    { "wave thirteen",     22, 25,     4, 8,    8, 15, 240,  800 },
    { "wave fourteen",     25, 30,     4, 8,    7, 14, 260,  950 },
    { "wave fifteen",      25, 30,     4, 8,    6, 13, 280, 1000 },
    { "wave sixteen",      25, 30,     4, 8,    5, 12, 300, 1050 },
    { "wave seventeen",    25, 30,     5, 8,    4, 10, 320, 1100 },
    { "wave eighteen",     25, 30,     5, 8,    3,  9, 340, 1150 },
    { "wave nineteen",     25, 30,     5, 8,    2,  8, 360, 1200 },
    { "ad infinitum",   10000, 10001,  5, 8,    1,  7, 380, 1250 },
};


Wave::Wave()
{
    max_level = sizeof(wavetable) / sizeof(wave_element);
    level = 0;
    wave_pause = 100;
    state = INITIALIZING;
}

Wave::~Wave()
{
    // do nothing
}

void Wave::init(int new_level)
{
    if (new_level >= max_level)
    {
	cerr << "Warning: max_level exceeded!" << endl;
	new_level = max_level - 1;
    }
    
    level = new_level;
    chopper_timer = 0;        // create first immediately
    choppers = random_int_range(wavetable[level].min_choppers,
				wavetable[level].max_choppers);

    // intro message
    message_start(level);

    // air alarm
    play_sample(samp_alarm, 128, 255, 1000, 0);
    
    // we're all set
    state = RUNNING;
}

void Wave::update()
{
    if (wave_pause > 0)
    {
	wave_pause--;
	return;
    }

    if (state == INITIALIZING)
    {
	init(level);
	return;
    }

    if (state == RUNNING)
    {
	chopper_timer--;

	if (chopper_timer < 0)
	{
	    if (choppers > 0)
	    {
		// generate a new one
		generate_chopper();
	    }
	}

	// level ends when there are no more choppers in play.
	// can't do this with troopers as there may be troopers on the
	// ground still.
	if (choppers_dead())
	{
	    // level completed
	    message_finish(level);
	    level++;
	    wave_pause = 500;
	    state = INITIALIZING;
	}
    }
}

void Wave::generate_chopper()
{
    Chopper *ch = new Chopper;
    ch->next = ::choppers;
    ::choppers = ch;

    ch->ntroopers = random_int_range(wavetable[level].min_troopers,
				     wavetable[level].max_troopers);
    
    // reset timer
    chopper_timer = random_int_range(100*wavetable[level].min_interval,
				     100*wavetable[level].max_interval);
    // count
    choppers--;
}

void Wave::message_start(int lev)
{
    Message *ms = new Message(wavetable[lev].name);
    vector_set(ms->scale, 0.15, 0.15, 1.0);
    ms->color = makecol(0, 50, 150);
    ms->rightleft();
    ms->next = messages;
    messages = ms;
}

void Wave::message_finish(int lev)
{
    char buf[256];
    sprintf(buf, "%s completed!", wavetable[lev].name); 
    Message *me = new Message(buf);
    vector_set(me->scale, 0.15, 0.15, 1.0);
    me->color = makecol(0, 150, 50);
    me->rightleft();
    me->next = messages;
    messages = me;
}

bool Wave::choppers_dead()
{
    Chopper *ch = ::choppers;

    while (ch)
    {
	if (ch->state != Chopper::DEAD && ch->state != Chopper::WRECK)
	    return false;
	ch = ch->next;
    }

    return true;
}
