#include "paratrooper.h"
#include "trooper.h"
#include "vector.h"
#include "triangle.h"
#include "globals.h"

Trooper::Trooper()
{
    vector_set(pos, 1, 6, 10);
    vector_set(v, 0, 0, -0.1);
    state = FLOAT;
    angle = 10.0;
    shoulders = 1.8;
    next = 0;
}

// draw function
void Trooper::draw()
{
    // save the matrix
    glPushMatrix();
    
    // translate to the right position
    glTranslatefv(pos);

    // rotate
    glRotatef(angle, 0, 0, 1);
    
    // assume that we may call GL calls right away
    if (state == FLOAT)
	draw_canopy();

    draw_man();

    glPopMatrix();
}

void Trooper::draw_man()
{
    // color blue
    glColor3f(0,0,1);

    // cube around 0,0,0
    // stretch it a bit
    glScalef(0.15, 0.15, 1);
    
    glBegin(GL_QUAD_STRIP);

    // sides of the cube
    glVertex3f(-1,-1,0);
    glVertex3f(-1,-1,shoulders);

    glVertex3f(1,-1,0);
    glVertex3f(1,-1,shoulders);

    glVertex3f(1,1,0);
    glVertex3f(1,1,shoulders);

    glVertex3f(-1,1,0);
    glVertex3f(-1,1,shoulders);

    glVertex3f(-1,-1,0);
    glVertex3f(-1,-1,shoulders);

    glEnd();

    // end caps of the cube
    glBegin(GL_QUADS);

    glVertex3f(-1,-1,0);
    glVertex3f(-1,1,0);
    glVertex3f(1,1,0);
    glVertex3f(1,-1,0);

    glVertex3f(-1,-1,shoulders);
    glVertex3f(1,-1,shoulders);
    glVertex3f(1,1,shoulders);
    glVertex3f(-1,1,shoulders);

    glEnd();
}

void Trooper::draw_canopy()
{
    // triangle strip
    // first vertex is the center one
    // canopy is a sphere: translate before drawing to position it
    // above the man.

    float d = 0.707;
    float
	green[3] = {0.5, 1.0, 0.5},
	white[3] = {1.0, 0.9, 0.9},
        gray[3] = {0.8, 0.8, 0.8};
	
    
    float points[9][3] = {      // 1 top, 8 edge points
	{0, 0, 1},
	{1, 0, 0},
	{d, d, 0},
	{0, 1, 0},
	{-d, d, 0},
	{-1, 0, 0},
	{-d, -d, 0},
	{0, -1, 0},
	{d, -d, 0}
    };
    

    glPushMatrix();
    
    // sphere centre: 5 meters above man head
    // that's 5 + shoulders meters above his feet
    
    glTranslatef(0, 0, 5 + shoulders);

    // scale: wider, not deeper
    glScalef(2.0, 2.0, 1.0);
    
    // color cream white slashed with green
    // circle around, of subdivided triangles

    glColor3fv(white);
    divide_triangle_sphere(points[0], points[1], points[2], 2);
    divide_triangle_sphere(points[0], points[3], points[4], 2);
    divide_triangle_sphere(points[0], points[5], points[6], 2);
    divide_triangle_sphere(points[0], points[7], points[8], 2);

    glColor3fv(green);
    divide_triangle_sphere(points[0], points[2], points[3], 2);
    divide_triangle_sphere(points[0], points[4], points[5], 2);
    divide_triangle_sphere(points[0], points[6], points[7], 2);
    divide_triangle_sphere(points[0], points[8], points[1], 2);

    // strings
    glColor3fv(gray);
    
    glBegin(GL_QUADS);
    for (int l = 1; l < 9; l++)
    {
	float offset[3];
	float res[3];
	float center[3] = {0, 0, -5};
	
	// compute offset orthogonal to points[l]
	offset[0] = -points[l][1] * 0.01;
	offset[1] = points[l][0] * 0.01;
	offset[2] = points[l][2] * 0.01;

	vector_add(points[l], offset, res);
	glVertex3fv(res);
	vector_subtract(points[l], offset, res);
	glVertex3fv(res);
	vector_add(center, offset, res);
	glVertex3fv(res);
	vector_subtract(center, offset, res);
	glVertex3fv(res);
	
	// compute offset parallel to points[l]
	offset[0] = points[l][0] * 0.01;
	offset[1] = points[l][1] * 0.01;
	offset[2] = points[l][2] * 0.01;

	vector_add(points[l], offset, res);
	glVertex3fv(res);
	vector_subtract(points[l], offset, res);
	glVertex3fv(res);
	vector_add(center, offset, res);
	glVertex3fv(res);
	vector_subtract(center, offset, res);
	glVertex3fv(res);
    }
    glEnd();
    
    glPopMatrix();
    
}

int Trooper::intersect_canopy(float *l1, float *l2, int n)
{
    // in n steps, check if points on line (l1, l2) intersect the canopy

    float factor = 0;
    float diff[3];      // distance from l1 to l2
    float diff2[3];     // distance from point to canopy center
    float dd[3];
    float canopy_center[3];
    float dist;
    
    vector_subtract(l2, l1, diff);    // line vector

    vector_assign(canopy_center, pos);
    canopy_center[2] += 5 + shoulders;
    
    for (int i = 0; i <= n; i++)
    {
	// compute partial line vector
	factor = i / (float)n;
	vector_assign(dd, diff);
	vector_multiply(dd, factor);
	vector_add(dd, l1);

	// dd should now be a point on the line somewhere between l1 and l2

	// check if we hit the canopy
	vector_subtract(canopy_center, dd, diff2);
	dist = vector_length(diff2);

	// canopy is not round so just radius won't do.
	// check vertical range as well
	if (dist < 2.0
	    && dd[2] > canopy_center[2]
	    && dd[2] < canopy_center[2] + 1)
	{
	    return 1;
	}
    }

    return 0;
}

// update function
void Trooper::update()
{
    if (state == DEAD)
    {
	return;
    }
    
    if (state == FLOAT)
    {
	angle += 0.2;
    }

    if (state == FALL || state == JUMP)
    {
	v[2] -= 0.001;    // 10 m/s/s = 10cm/0.01s/s  0.001 m/0.01s/0.01s
    }

    vector_add(pos, v);

    if (state == JUMP)
    {
	if (pos[2] < open_height)
	{
	    state = FLOAT;
	    v[2] = -0.04;
	}
    }

    if (state == WALK)
    {
	float dist = vector_length(pos);

	if (dist < 10)
	{
	    // bit strange, but make troopers disappear when they
	    // damage the gun. Looks funny otherwise, a whole row of
	    // blue blocks around you.
	    gun->activists++;
	    vector_set(v, 0, 0, 0);
	    state = DEAD;
	}
    }
    
    if (pos[2] < 0 && state != WALK)
    {
	if (v[2] < -0.05)
	{
	    // scrash boom
	    state = DEAD;
	    return;
	}
	
	state = WALK;
	pos[2] = 0.0;

	// walk towards gun
//	float angle = atan2(pos[1], pos[0]);
//	v[0] = -cos(angle);
//	v[1] = -sin(angle);

	v[0] = -pos[0];
	v[1] = -pos[1];
	v[2] = 0.0;  // (on the ground)

	vector_normalize(v);
	vector_multiply(v, 0.03);    //  3m/s (slightly fast but looks better)
    }
}
