#include "paratrooper.h"
#include "globals.h"
#include "vector.h"
#include "texture.h"
#include "health.h"
#include "tree.h"
#include "radar.h"

char const *progname;

void usage()
{
    cerr << "usage: " << progname << endl;
}

// initialise allegro
// initialise allegrogl
// set grafics mode
// set audio modes

int init()
{
    // allegro
    allegro_init();

    // allegrogl
    install_allegro_gl();

    // display init
    allegro_gl_clear_settings();
    allegro_gl_set(AGL_COLOR_DEPTH, 32);
    allegro_gl_set(AGL_Z_DEPTH, 16);
//    allegro_gl_set(AGL_WINDOWED, TRUE);
    allegro_gl_set(AGL_FULLSCREEN, TRUE);
    allegro_gl_set(AGL_DOUBLEBUFFER, 1);
    allegro_gl_set(AGL_SUGGEST, AGL_COLOR_DEPTH | AGL_Z_DEPTH |
		   AGL_DOUBLEBUFFER 
		   | AGL_FULLSCREEN);
//		   | AGL_WINDOWED);
		   
    set_color_depth(32);
    if (set_gfx_mode(GFX_OPENGL, 640, 480, 0, 0) < 0)
    {
	allegro_message("Error setting graphics mode:\n%s\n", allegro_error);
	return 0;
    }

//    allegro_gl_screen_mode(AGL_MODE_DUPLEX);

    text_mode(-1);
    
    install_keyboard();
    install_timer();
    install_mouse();

    // sound init
    reserve_voices(8, -1);
    if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, 0) < 0)
    {
	allegro_message("Error installing sound:\n%s\n", allegro_error);
	return 0;
    }
    
    // install interrupt handler (every 10 milliseconds)
    LOCK_VARIABLE(loop);
    LOCK_FUNCTION(timer_func);
    install_int(timer_func, 10);
    
    return 1;
}

void init_gl()
{
    // set every gl setting that we can do in advance
    allegro_gl_begin();

    glShadeModel(GL_SMOOTH);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
//    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glEnable(GL_DEPTH_TEST);
//    glCullFace(GL_BACK);
//    glEnable(GL_CULL_FACE);
    glClearColor(sky[0], sky[1], sky[2], 0);

    // set viewing frustum
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glFrustum(-0.8, 0.8, -0.6, 0.6, 2.0, MAX_DISTANCE);
    glMatrixMode(GL_MODELVIEW);

    // set fog
    glEnable(GL_FOG);
    glFogfv(GL_FOG_COLOR, sky);
    glFogf(GL_FOG_DENSITY, 0.1);
    glFogi(GL_FOG_MODE, GL_LINEAR);
    glFogf(GL_FOG_START, MAX_DISTANCE / 2.0);
    glFogf(GL_FOG_END, MAX_DISTANCE);

    // texture settings
    glDisable(GL_DITHER);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
//    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
    allegro_gl_set_texture_format(GL_RGBA8);
//    allegro_gl_set_texture_format(GL_ALPHA8);
    
    // transparency settings
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    // end of gl settings
    allegro_gl_end();
}

void camera()
{
    // turn in the direction of the gun
    // translate up 2 meters so we're not in the ground
    glRotatef(-90 - gun->theta, 1, 0, 0);
    glRotatef(gun->phi, 0, 0, 1);
    glTranslatef(0, 0, -2);
}   


void draw_ground()
{
    float brown[3] = {0.3, 0.2, 0.05};
    
    glPushMatrix();
    glColor3fv(brown);
    
    for (int i = 0; i < 8; i++)
    {
	// triangle from 0 to far clipping plain, and 45 degrees to the right
	glBegin(GL_TRIANGLES);
	glVertex3f(0, 0, 0);
	glVertex3f(0, MAX_DISTANCE, 0);
	glVertex3f(710, 710, 0);
	glEnd();

	// rotate 45 degrees
	glRotatef(45, 0, 0, 1);
    }

    glPopMatrix();
}

void delete_bullets(bool always)
{
    Bullet
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = bullets;

    while (b)
    {
	if (b->state == Bullet::DEAD || always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    bullets = new_b;
}

void delete_trees(bool always)
{
    Tree
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = trees;

    while (b)
    {
	if (always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    trees = new_b;
}


void delete_choppers(bool always)
{
    Chopper
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = choppers;

    while (b)
    {
	if (b->state == Chopper::DEAD || always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    choppers = new_b;
}

void delete_troopers(bool always)
{
    Trooper
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = troopers;	

    while (b)
    {
	if (b->state == Trooper::DEAD || always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    troopers = new_b;
}

void delete_explosions(bool always)
{
    Explosion
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = explosions;	

    while (b)
    {
	if (b->state == Explosion::DEAD || always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    explosions = new_b;
}


void delete_messages(bool always)
{
    Message
	*b,
	*del,
	*keep,
	*new_b;

    new_b = 0;
    b = messages;	

    while (b)
    {
	if (b->state == Message::DEAD || always)
	{
	    // throw it away
	    del = b;
	    b = b->next;
	    delete del;
	}
	else
	{
	    // keep it
	    keep = b;
	    b = b->next;
	    keep->next = new_b;
	    new_b = keep;
	}
    }

    messages = new_b;
}

void logic_loop()
{
    Bullet *b;
    Trooper *tr;
    Chopper *ch;
    Explosion *ex;
    Message *msg;
    int mx, my;
//    int k;

    // don't use readkey as it will limit the steering rate to the 
    // keyboard repeat rate. Use the key array instead
    
    if (key[KEY_P])
    {
	while (key[KEY_P]);
	paused = !paused;
    }
	
    if (key[KEY_G])
    {
	while (key[KEY_G]);
	game_over = 1;
    }
    
    if (key[KEY_SPACE])
    {
	if (paused)
	    paused = 0;
	else
	    gun->fire();
    }
    
    if (key[KEY_LEFT])
	gun->left();
	
    if (key[KEY_RIGHT])
	gun->right();
    
    if (key[KEY_DOWN])
	gun->up();
    
    if (key[KEY_UP])
	gun->down();
    
    if (mouse_b & 1)
    {
	gun->fire();
    }

    // read mouse mickeys
    get_mouse_mickeys(&mx, &my);
    // move the gun accordingly
    gun->move(mx, my);
	
    // update gun
    gun->update();

    // update the wave generator
    wave->update();
    
    // update bullets
    b = bullets;
    while (b)
    {
	if (!paused)
	    b->update();
	b = b->next;
    }

    // delete dead bullets
    delete_bullets();
    
    // update choppers
    ch = choppers;
    while (ch)
    {
	if (!paused)
	    ch->update();
	ch = ch->next;
    }
    
    // delete dead choppers
    delete_choppers();
    
    // update troopers
    tr = troopers;
    while (tr)
    {
	if (!paused)
	    tr->update();
	tr = tr->next;
    }
    
    // delete dead troopers
    delete_troopers();

    // update explosions
    ex = explosions;
    while (ex)
    {
	if (!paused)
	    ex->update();
	ex = ex->next;
    }

    // delete dead explosions
    delete_explosions();

    // update messages
    msg = messages;
    while (msg)
    {
	// do this even when paused, otherwise game over messages won't work
	msg->update();
	msg = msg->next;
    }

    // delete dead messages
    delete_messages();
    
    // check for game over
    if (gun->activists > MAX_ACTIVISTS)
    {
	game_over = 1;
	return;
    }
    
    // count a gameloop
    loop--;
}

void draw_frame()
{
    Bullet *b;
    Trooper *tr;
    Chopper *ch;
    Tree *tree;
    Explosion *ex;
    Message *msg;
    
    allegro_gl_begin();
    // reset
    glLoadIdentity();
    // clear
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    // gun and crosshair and score have to be in front, but drawn last.
    // (for alpha) this requires saving the state and drawing them afterwards.

    glPushMatrix();
    
    camera();

    draw_ground();

    // draw bullets
    b = bullets;
    while (b)
    {
	b->draw();
	b = b->next;
    }

    // draw choppers
    ch = choppers;
    while (ch)
    {
	ch->draw();
	ch = ch->next;
    }
    
    // draw troopers
    tr = troopers;
    while (tr)
    {
	tr->draw();
	tr = tr->next;
    }

    // draw trees
    tree = trees;
    while (tree)
    {
//	printf("drawing tree at [%f %f %f]\n", tree->pos[0],tree->pos[1],tree->pos[2]);
	tree->draw();
	tree = tree->next;
    }

    // draw explosions
    ex = explosions;
    while (ex)
    {
	ex->draw();
	ex = ex->next;
    }

    glPopMatrix();     // done drawing the world

    // gun, crosshair and other screen stuff
    gun->draw();
    draw_score();
    draw_fps();
    draw_health_bar();
    draw_radar();
    
    if (game_over)
    {
 	shutdown_globals();
 	init_globals();
	game_over = 0;
	paused = 1;
	draw_game_over();
    }
    else
	draw_crosshair();

    // draw messages
    msg = messages;
    while (msg)
    {
	msg->draw();
	msg = msg->next;
    }
    
//    static int i = 0;
//    printf("update %d.\n", ++i);
    
    glFlush();
    allegro_gl_flip();
    allegro_gl_end();

    if (game_over)
    {
 	clear_keybuf();

    }
}


int init_data()
{
    data = load_datafile("avantgarde.dat");
    if (!data)
    {
	cerr << "Error loading 'avantgarde.dat'." << endl;
	return 1;
    }

    sound = load_datafile("sound.dat");
    if (!sound)
    {
	cerr << "Error loading 'sound.dat'." << endl;
    }
    
    return 0;
}


int main(int argc, char **argv)
{

    int
	frames = 0,
	csecs = 0;
    
    progname = argv[0];
    
    if (!init())
	return 1;

    init_gl();

    if (init_data())
	return 0;
    
    init_globals();
    
    while (!key[KEY_ESC])
    {
	// draw a frame
	draw_frame();
	frames++;
	csecs += loop;

	// compute fps once per second
	if (csecs > 100)
	{
	    fps = frames * 100 / csecs;
	    frames = 0;
	    csecs = 0;
	}
	
	// catch strange bugs: if loop < 0 || > 100,
	// report and set to a sane value
	if (loop < 0 || loop > 100)
	{
	    cerr << "warning: fishy stuff happening. "
		 << "logic loop got value " << loop << "." << endl;
	}
	
	// loop until we're back up to snuff with the game logic
	// do a maximum of 100 game loops, give up after that
	// (computer is too slow or strange stuff happening)
	int i = 100;
	while (loop > 0 && i-- > 0)
	{
	    logic_loop();
	}
	loop = 0;              // declare us done, even if we're not.
    }
    
    return 0;
}
END_OF_MAIN();
