#include <stdlib.h>
#include <math.h>
#include "paratrooper.h"
#include "globals.h"
#include "gun.h"
#include "cylinder.h"
#include "vector.h"

#define DELTA_H 0.1
#define DELTA_V 0.1
#define KEY_FACTOR 2

Gun::Gun()
{
    theta = 10.0;   // up angle
    phi = 0.0;      // directional angle

    reloading = 0;
    activists = 0;   // no troopers at the gun yet

    // init_data happens before init_globals (important!)
    // samples must be loaded _before_ init globals too.
    voice_fire = allocate_voice(samp_gun);
    voice_reload = allocate_voice(samp_reload);

    if (voice_fire < 0 || voice_reload < 0)
    {
	cerr << "Error: not enough voices available for Gun.";
	exit(1);
    }

    voice_set_priority(voice_fire, 255);
    voice_set_priority(voice_reload, 254);
    voice_set_volume(voice_fire, 255);
    voice_set_volume(voice_reload, 255);
}

Gun::~Gun()
{
    release_voice(voice_fire);
    release_voice(voice_reload);
}

void Gun::up()
{
    theta += KEY_FACTOR * DELTA_V;
    if (theta > 85)
	theta = 85;
}

void Gun::down()
{
    theta -= KEY_FACTOR * DELTA_V;
    if (theta < 5)
	theta = 5;
}
    
void Gun::left()
{
    phi -= KEY_FACTOR * DELTA_H;
    if (phi < 0)
	phi += 360;
}

void Gun::right()
{
    phi += KEY_FACTOR * DELTA_H;
    if (phi > 360)
	phi -= 360;
}

void Gun::move(int mx, int my)
{
    phi += mx * DELTA_H;
    while (phi > 360)
	phi -= 360;
    while (phi < 0)
	phi += 360;

    theta -= my * DELTA_V;
    if (theta < 5)
	theta = 5;
    if (theta > 85)
	theta = 85;
}

void Gun::fire()
{
    float ph, th;

    if (reloading > 0)
	return;

    // set reloading (0.8 second)
    reloading = 80;

    // make a bullet and add it to the list
    Bullet *b = new Bullet;
    b->next = bullets;
    bullets = b;
    
    // copy angles from cannon
    b->phi = phi;
    b->theta = theta;

    // gun is 0.7 meters above the ground (2 eyeheight - 1.3m)
    // put the bullet there too
    vector_set(b->pos, 0, 0, 0.65);
    
    // compute bullet velocities
    // phi = 0 -> gun looks along the y axis
    // also, positive angles mean the gun has turned to the right

    // compute the 'proper' angles first
    ph = 2 * M_PI * (90 - phi) / 360;
    th = 2 * M_PI * theta / 360;

    b->v[0] = cos(ph) * cos(th);
    b->v[1] = sin(ph) * cos(th);
    b->v[2] = sin(th);

    // normalize at bullet speed
    // 2 meters per time step
    // keep this in mind when intersecting
    vector_multiply(b->v, 2.0);

//    printf("bullet fired. pos [%f %f %f] v [%f %f %f] angles %f %f\n",
//	   b->pos[0],b->pos[1],b->pos[2],b->v[0],b->v[1],b->v[2],
//	   b->phi, b->theta);

    // play sound
    voice_start(voice_fire);
    voice_start(voice_reload);
}

void Gun::draw()
{
    // draw gun (and crosshair?)
    // these are performed _before_ the camera position is set,
    // as they need to move with the camera.
    glPushMatrix();

    // we need to draw the barrel in the z direction (draw_cylinder)
    // as the camera is still looking in the negative z direction,
    // that's not so hard.
    draw_barrel();
    
    glPopMatrix();
}

void Gun::draw_barrel()
{
    // draw a cylinder, on the negative z axis from 0 to -4 m

    // we need to scale to get the radius right (default 1)
    // translate down a bit so we don't look _through_ the barrel
    glTranslatef(0, -1.3, 0);
    glScalef(0.1, 0.1, 1.0);
    glColor3fv(barrel);
    draw_cylinder(-4, -8, 2);
}

void Gun::update()
{
    if (reloading > 0)
	reloading--;
}
