#include <math.h>
#include <stdio.h>
#include "paratrooper.h"
#include "chopper.h"
#include "vector.h"
#include "triangle.h"
#include "box.h"
#include "globals.h"
#include "cylinder.h"
#include "random.h"

Chopper::Chopper()
{
    // position: 1 km away from the LZ
    // height: 200m +- 50m

    // compute a random start and target point
    float
	from[3],
	to[3],
	r = MAX_DISTANCE - 1.0,  // without -1 it gets removed right away
//	r = 100,
	approach;

    next = NULL;
    
    // angle between 0 and 2pi
    approach = random_range(0.0, 2 * M_PI);
//    approach = 0.5 * M_PI;
    
    // compute x, y, and vx, vy
    from[0] = r * cos(approach);
    from[1] = r * sin(approach);

    from[2] = random_range(JUMP_HEIGHT, JUMP_HEIGHT + 50.0);
//    from[2] = 20;
    
    // set position
    vector_assign(pos, from);

//    printf("heli pos: %f, %f, %f\n", pos[0], pos[1], pos[2]);
    
    // pick random target (reuse approach angle)
    // anywhere within 100 m from the gun
    approach = random_range(0.0, 2*M_PI);
    r = random_range(0.0, 100.0);

    to[0] = r * cos(approach);
    to[1] = r * sin(approach);
    to[2] = from[2];

//    to[0] = 0;
//    to[1] = 0;
//    to[2] = 10;
    
    // compute velocity (30 m/s)
    vector_subtract(to, from, v);
    vector_normalize(v);
    vector_multiply(v, 0.3);
//    vector_multiply(v, 0.02);

    // compute turn angle of the chopper for drawing and collisions.
    // keep in mind that the body points towards the positive y axis,
    // so when flying in direction 0, the body really has direction
    // -90 or it will fly sideways.
    dir_radian = atan2(v[1], v[0]) - 0.5 * M_PI;
    dir_degree = dir_radian * 360 / (2 * M_PI);
    
    // set remaining variables
    angle = 0.0;
    state = FLYING;
    ntroopers = 1;          // will be set by the wave generator
    time_to_drop = 0;

    // allocate a voice
    voice_chopper = allocate_voice(samp_chopper);
    
    // this may fail if we get too many choppers. Check for that !
    if (voice_chopper >= 0)
    {
	voice_set_playmode(voice_chopper, PLAYMODE_LOOP);
	voice_set_volume(voice_chopper, 0);
	voice_start(voice_chopper);
    }
}

Chopper::~Chopper()
{
    if (voice_chopper >= 0)
	deallocate_voice(voice_chopper);
}

void Chopper::draw()
{
    glPushMatrix();

    // translate to chopper position
    glTranslatefv(pos);

    // rotate so we're facing forwards
    // subtract 90 degrees for chopper orientation
    glRotated(dir_degree, 0, 0, 1);

//    printf("dir = %lf, v = [%f, %f, %f]\n", dir, v[0], v[1], v[2]);
    
    draw_body();

    // translate up a little bit for the rotor
    // helicopter roof 2.0m + 0.3m axis
    // save state for doing the tail rotor afterwards

    glPushMatrix();
    glTranslatef(0, 0, 2.3);
    // turn the rotor
    glRotatef(angle, 0, 0, 1);
    draw_rotor();
    glPopMatrix();

    // now do the tail rotor
    glPushMatrix();
    // translate to tail
    glTranslatef(-0.3, -6, 1.8);
    // rotate (position, and turning of the rotor)
    glRotatef(-90, 0, 1, 0);
    glRotatef(angle, 0, 0, 1);
    // draw
    draw_tail_rotor();
    glPopMatrix();
    
    glPopMatrix();
}

void Chopper::draw_body()
{
    // start with a box

    // leave some space at the bottom for struts
    glColor3fv(chopper);
    draw_box(-1, -2, 0.5, 1, 2, 2);

    // tail boom
    // use a cylinder. need to rotate first, and scale for the radius
    // draw_box(-0.25, -2, 1.5, 0.25, -6, 1.9);
    glPushMatrix();
    glTranslatef(0, 0, 1.6);
    glRotatef(90, 1, 0, 0);
    glScalef(0.2, 0.2, 1.0);
    draw_cylinder(2, 6, 1);    // 1 = 1 subdivision level
    glPopMatrix();

    // struts
    glColor3fv(dark_gray);
    draw_box(-1.2, -1.5, 0, -1.25, 1.5, 0.05);
    draw_box(1.2, -1.5, 0, 1.25, 1.5, 0.05);

}

void Chopper::draw_rotor()
{
    // vertical axis
    glColor3fv(dark_gray);
    draw_box(-0.05,-0.05,-0.3, 0.05, 0.05, 0);

    // blades

    for (int blade = 0; blade < 6; blade++)
    {
	glBegin(GL_QUADS);
	
	glVertex3f(0.05, 0.05, 0);
	glVertex3f(0.05, -0.25, 0);
	glVertex3f(5, -0.25, 0);
	glVertex3f(5, 0.05, 0);
	
	glEnd();
	
	// next blade
	glRotatef(60, 0, 0, 1);
    }
}

void Chopper::draw_tail_rotor()
{
    glColor3fv(dark_gray);
    // tail rotor is simply 3 quads, no axis.

    for (int blade = 0; blade < 3; blade++)
    {
	glBegin(GL_QUADS);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(0.1, 0.0, 0.0);
	glVertex3f(0.1, 1.0, 0.0);
	glVertex3f(0.0, 1.0, 0.0);
	glEnd();

	glRotatef(120, 0, 0, 1);
    }
}

void Chopper::update()
{
    float dist;

    if (state == DEAD || state == WRECK)
	return;

    vector_add(pos, v);
    angle += 6;
    dist = vector_length_xy(pos);

//    printf("chopper at [%f %f %f] flying in dir [%f %f %f] state %d\n",
//	   pos[0], pos[1], pos[2], v[0], v[1], v[2], state);
    
    // check if we should start dropping troopers
    if (state == FLYING && ntroopers > 0)
    {
	// check distance to LZ

	if (dist < DROP_ZONE)
	{
	    // start dropping
	    state = DROPPING;

	    // set time to drop (0 to 3 seconds)
	    time_to_drop = random_int_range(100, 300);
	}
    }
    
    if (state == DROPPING)
    {
	time_to_drop--;
	if (!time_to_drop)
	{
	    if (dist < DROP_ZONE)
	    {
		drop_trooper();
		
		if (ntroopers > 0)
		{
		    time_to_drop = random_int_range(100, 300);
		}
		else
		{
		    time_to_drop = 0;
		    state = FLYING;
		}
	    }
	    else
	    {
		// flew out of the dropzone
		time_to_drop = 0;
		state = FLYING;
	    }
	}
    }

    if (state == FALLING)
    {
	v[2] -= 0.001;   // 10m/s/s = 0.001 m/0.01s/0.01s
	
	if (pos[2] < 0)
	{
	    // crash
	    state = WRECK;
	    vector_set(v, 0, 0, 0);

	    explode();
	}
    }

    // check if we're out of range. Don't count z coordinate
    if (dist > MAX_DISTANCE)
    {
	state = DEAD;
    }

    // update our volume
    if (voice_chopper >= 0)
    {
	float
	    reldist = 5 * dist / MAX_DISTANCE,       // 5 = fudge factor
	    factor = 1.0 / (1.0 + reldist*reldist);
	
	voice_set_volume(voice_chopper, (int)(255 * factor));
    }
    
//    printf("chopper state now %d\n", state);
}

void Chopper::drop_trooper()
{
    Trooper *tr = new Trooper;

    // give it our position
    vector_assign(tr->pos, pos);
    tr->state = Trooper::JUMP;
    tr->open_height = random_range(JUMP_HEIGHT - 20.0, JUMP_HEIGHT);
 
    // put it in the list
    tr->next = troopers;
    troopers = tr;

    // count 1 trooper
    ntroopers--;
}

// collision detection: does p fall within our body
// intersect bullet line with body
int Chopper::contains(float *p1, float *p2)
{
    // compute our flight angle
    // rotate so we're facing forwards
    float
	diff1[3],
	diff2[3],
	b1[3],
	b2[3];

    // compute difference vector
    vector_subtract(p1, pos, diff1);
    vector_subtract(p2, pos, diff2);
    
    // convert it to chopper coordinates  (rotate by -dir)
    vector_rotate_xy(diff1, -dir_radian);
    vector_rotate_xy(diff2, -dir_radian);

    // now check if it falls within our body
    vector_set(b1, -1, -2, 0.5);
    vector_set(b2, 1, 2, 2);

    // intersect in 20 steps
    // step size is 2m
    // check in 5 steps, 40cm should catch most cases
    if (intersect_box(diff1, diff2, b1, b2, 5))
    {
//	printf("converted bullet coordinates: [%f %f %f] [%f %f %f]\n",
//	       diff1[0],diff1[1],diff1[2],diff2[0],diff2[1],diff2[2]);
	return 1;
    }

    // check the tail
    // we want to hit the tail boom too so use enough steps
    // 40 cm is not enough here, use 10 cm
    vector_set(b1, -0.2, -2, 1.4);
    vector_set(b2, 0.2, -6, 1.8);
    if (intersect_box(diff1, diff2, b1, b2, 20))
    {
	return 1;
    }
   
    return 0;
}

void Chopper::explode(float radius)
{
    float dist, reldist, factor;
    
    // make an explosion at our current location
    Explosion *ex = new Explosion;
    vector_assign(ex->pos, pos);
    ex->radius = radius;
    ex->duration = 50;        // 0.5 seconds, slightly longer than normal
    
    // add it to the list
    ex->next = explosions;
    explosions = ex;

    // stop our voice and play an explosion
    deallocate_voice(voice_chopper);
    voice_chopper = -1;

    dist = vector_length(pos);
    reldist = 5 * dist / MAX_DISTANCE;       // 5 = fudge factor
    factor = 1.0 / (1.0 + reldist*reldist);

    // reduce frequency a bit for a longer explosin
    play_sample(samp_explode, (int)(255 * factor), 128, 900, 0);

    // crash sound should follow slightly later
    play_sample(samp_crash, (int)(255 * factor), 128, 1000, 0);
}
