#include "bullet.h"
#include "paratrooper.h"
#include "cylinder.h"
#include "vector.h"
#include "globals.h"
#include "box.h"
#include <stdio.h>

Bullet::Bullet()
{
    vector_set(pos, 0, 0, 0);
    vector_set(v, 0, 0, 0);
    state = FLYING;
    phi = theta = 0.0;
    next = NULL;
}   

void Bullet::draw()
{
    glPushMatrix();

    // translate to our position
    glTranslatefv(pos);

    // compute flight direction and rotate accordingly
    // use the angles copied from the gun for this
    // keep in mind that the bullet is vertical to start with
    glRotatef(-phi, 0, 0, 1);
    glRotatef(theta, 1, 0, 0);
    glRotatef(-90, 1, 0, 0);
    
    glColor3fv(barrel);
    
    // as we're using draw_cylinder, we need to scale the radius.
    // radius of the gun is 10cm, bullet radius is 9 cm
    glScalef(0.09, 0.09, 1.0);

    // draw the cylinder. bullet length: 0.5 m
    draw_cylinder(0, 0.5, 1);

    // draw endcap (orange)

    glColor3fv(orange);

    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(0, 0, 0);
    glVertex3f(-1, 0, 0);
    glVertex3f(-0.707, -0.707, 0);
    glVertex3f(0, -1, 0);
    glVertex3f(0.707, -0.707, 0);
    glVertex3f(1, 0, 0);
    glVertex3f(0.707, 0.707, 0);
    glVertex3f(0, 1, 0);
    glVertex3f(-0.707, 0.707, 0);
    glVertex3f(-1, 0, 0);
    glEnd();

    // bullet point (in red :-)
    glColor3fv(red);

    glBegin(GL_TRIANGLE_FAN);
    glVertex3f(0, 0, 0.6);
    glVertex3f(-1, 0, 0.5);
    glVertex3f(-0.707, -0.707, 0.5);
    glVertex3f(0, -1, 0.5);
    glVertex3f(0.707, -0.707, 0.5);
    glVertex3f(1, 0, 0.5);
    glVertex3f(0.707, 0.707, 0.5);
    glVertex3f(0, 1, 0.5);
    glVertex3f(-0.707, 0.707, 0.5);
    glVertex3f(-1, 0, 0.5);
    glEnd();
    
    glPopMatrix();
}

void delete_bullet(Bullet *b)
{
    // find in the list
    Bullet *f = bullets;
    Bullet *last = 0;
    
    while (f)
    {
	if (f == b)
	{
	    if (last)
	    {
		last->next = f->next;
	    }
	    else
	    {
		bullets = f->next;
	    }
	}

	last = f;
	f = f->next;
    }
}

int Bullet::collide(float *oldpos, float *newpos)
{
    Trooper *t;
    Chopper *c;

    // collide with troopers
    t = troopers;
    while (t)
    {
	float  m1[3] = {t->pos[0] - 0.3, t->pos[1] - 0.3, t->pos[2]};
	float  m2[3] = {t->pos[0] + 0.3, t->pos[1] + 0.3, t->pos[2] + t->shoulders};

	// don't shoot walking troopers
	if (t->state == Trooper::WALK || t->state == Trooper::STAND)
	{
	    t = t->next;
	    continue;
	}
	
	// check if we hit it directly
	
	// use intersect_box in 20 steps (man is small, stepsize is 2m)
	if (intersect_box(oldpos, newpos, m1, m2, 20))
	{
	    // hit man
	    t->state = Trooper::DEAD;
	    score += SCORE_MAN;

	    explode();
	    return 1;
	}

	if (t->intersect_canopy(oldpos, newpos, 10))
	{
	    // hit the canopy
	    t->state = Trooper::FALL;
	    score += SCORE_CANOPY;
	    return 1;
	}
	
	t = t->next;
    }

    // collide with choppers
    c = choppers;
    while (c)
    {
	// box collisions
	// keep in mind the boxes are rotated (urgh).
	// pos1 is old bullet position, pos2 is new
	
	if (c->contains(oldpos, newpos))
	{
//	    printf("chopper hit. bullet and chopper positions: [%f %f %f] [%f %f %f] [%f %f %f]\n",
//		   oldpos[0],oldpos[1],oldpos[2],
//		   newpos[0],newpos[1],newpos[2],
//		   c->pos[0],c->pos[1],c->pos[2]);
		   
	    c->state = Chopper::FALLING;
	    score += SCORE_CHOPPER;
	    score += c->ntroopers * SCORE_CASUALTY;

	    explode();
	    return 1;
	}
	
	c = c->next;
    }
    
    return 0;
}

void Bullet::update()
{
    float dist;

    float
	oldpos[3],
	newpos[3];

    vector_assign(oldpos, pos);
    vector_add(oldpos, v, newpos);
    
    // check if we will collide
    if (collide(oldpos, newpos))
    {
	state = DEAD;
	return;
    }
    
    // move
    vector_assign(pos, newpos);
    
    // check if we're out of range
    dist = vector_length(pos);
    if (dist > MAX_DISTANCE)
    {
	state = DEAD;
    }
}

void Bullet::explode(float radius)
{
    float reldist, factor, dist;
    
    // make an explosion at our current location
    Explosion *ex = new Explosion;
    vector_assign(ex->pos, pos);
    ex->radius = radius;

    // add it to the list
    ex->next = explosions;
    explosions = ex;

    // play explosion sound
    // compute distance of the bullet to determine the volume

    dist = vector_length(pos);
    reldist = 5 * dist / MAX_DISTANCE;       // 5 = fudge factor
    factor = 1.0 / (1.0 + reldist*reldist);

    // increase frequency a bit for a shorter explosion
    play_sample(samp_explode, (int)(255 * factor), 128, 1300, 0);
}
