/*  Source file for the Edit3d program by Robert Parker using the Allegro
    and Bgui2 - see credits else where - also see BasicGUI source code.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "solid.h"
#include "drawline.h"
#include "obj3ctrl.h"

SolidClass::SolidClass() : ObjectClass()
{ P = NULL;
  X = Xb = Y = Z = 0;  // Xb is X dimension in bytes
  Element = 0;  // elements are 1 unit in size
  Vol = NULL;
}

SolidClass::~SolidClass()
{
  if(Vol)
    free(Vol);
}

void SolidClass::CreateVolume(fixed dx, fixed dy, fixed dz)
{
  fixed es = 1<<(16 + Element);
  X = dx/es;
  Y = dy/es;
  Z = dz/es;
  int Xb = X/8;
  if(Xb*8 < X)
    Xb++;
  long ts = (long)Xb * Y * Z;  // 8 bit aligned
  Vol = (char*)realloc(Vol,ts);
  if(Vol)
    memset(Vol,0xFF,ts);   // starts as solid block?
}

bool SolidClass::State(int x, int y, int z)
{
  return(Vol[x/8+((z*Y)+y)*Xb] & (1<<(x%8))!= 0);
}


struct cubestruct { unsigned X :1;
                    unsigned Y :1;
                    unsigned Z :1;
                  };

const cubestruct CubeLine[12][2] = {{{0,0,0},{1,0,0}},
                                    {{0,1,0},{1,1,0}},
                                    {{0,0,1},{1,0,1}},
                                    {{0,1,1},{1,1,1}},
                                    {{0,0,0},{0,1,0}},
                                    {{0,0,1},{0,1,1}},
                                    {{1,0,0},{1,1,0}},
                                    {{1,0,1},{1,1,1}},
                                    {{0,0,0},{0,0,1}},
                                    {{0,1,0},{0,1,1}},
                                    {{1,0,0},{1,0,1}},
                                    {{1,1,0},{1,1,1}},
                                   };

SolidControlClass SolidControl;

#define SOLID ((SolidClass*)object)

SolidControlClass:: SolidControlClass() : Object3DControlClass()
{ Name = "Solid"; Properties = 5;
  Color = LIGHT_GREEN;
}

ObjectClass* SolidControlClass::Create()
{
  return(new SolidClass);
}

void SolidControlClass::Remove(ObjectClass* object)
{
  // remove any remote references
}

long SolidControlClass::Size()
{
  return(sizeof(SolidClass));
}

void SolidControlClass::NewForOld( ObjectClass* object, ObjectClass* oldobj, ObjectClass* newobj)
{
  if(SOLID->P == (VertexClass*)oldobj)
    SOLID->P = (VertexClass*)newobj;
}

bool SolidControlClass::Match(ObjectClass* object,BlockClass* block ,LayerClass* layer)
{ if(SOLID->P)
     return(SOLID->P->Block == block && (AllLayers || SOLID->P->Layer == layer));
  else
     return false;
}

bool SolidControlClass::DependantOn(ObjectClass* object, ObjectClass* dependant)
{ return(dependant == SOLID->P);
}

void SolidControlClass::GetOtherVertex(VTX* p, ObjectClass* object) {
  // mixed types - be caarrrreeeefull
  fixed es = 1<<(16+SOLID->Element);
  p->X = fixadd(SOLID->P->Pos.X,es*SOLID->X);
  p->Y = fixadd(SOLID->P->Pos.Y,es*SOLID->Y);
  p->Z = fixadd(SOLID->P->Pos.Z,es*SOLID->Z);
}

bool SolidControlClass::Include(ObjectClass* object, int, int, int, int)
{
  if(SOLID->P && SOLID->P->Block == Base_Block)
    return(Current_Visual->Include(Type,SOLID->P->Layer));
  return false;
}

bool SolidControlClass::Inside(ObjectClass* object, int mx, int my)
{
  if(!SOLID->P || SOLID->P->Block != Base_Block)
    return false;
// only comes here after passing rough check on limits
  long x1, x2, y1, y2, z;
  Current_Visual->RealToFlat(&SOLID->P->Pos,x1,y1,z);
  VTX p;
  GetOtherVertex(&p,object);
  Current_Visual->RealToFlat(&p,x2,y2,z);
  return((mx > x1) && (mx < x2) && (my > y1) && (my < y2));
}

#include "visulist.h"

// should be called Get Pointer to and Name of Property
int SolidControlClass::GetProperty(ObjectClass* object, int n, void** p, char** name,
                                               void** select_function)
{
  *select_function = NULL;
  VertexClass* vtx;
  if(n == 0)
  { vtx = SOLID->P;
    // user wants to have a better reference than a hexdecimal pointer so convert to link count
    static int vertex;
    vertex = TargetList->Find_OfType(vtx,&VertexControl);
    *p = &vertex;
  }
  switch(n)
  { case 0:
      *name = "P:";
      return(INT_TYPE);
    case 1:
      *name = "X:";
      return(INT_TYPE);
    case 2:
      *name = "Y:";
      return(INT_TYPE);
    case 3:
      *name = "Z:";
      return(INT_TYPE);
    case 4:
      *p = &(SOLID->Element);
      *name = "Element^2:";
      return(INT_TYPE);
  }
  return(-1);
}

int SolidControlClass::Summary(ObjectClass* object,char* str, int maxstr, int flags)
{
  if((flags & 1) && SOLID->P && SOLID->P->Block != Base_Block) {
    str[0] = 0;
    return 0;
  }
  int p = TargetList->Find_OfType(SOLID->P,&VertexControl);
  int i = uszprintf(str,maxstr,"%s P=%d X=%d Y=%d Z=%d",Name,p,
       SOLID->X,SOLID->Y,SOLID->Z);
  i += uszprintf(str+i,maxstr-i,"E^2=%d",SOLID->Element);
  i += VertexControl.Summary_Layer(SOLID->P,str+i,maxstr-i);
  i += VertexControl.Summary_Pos(SOLID->P,str+i,maxstr-i);
  return(i);
}

#include "visulist.h"

int SolidControlClass::SetProperty(ObjectClass* object, int n, void* p)
{
  VertexClass* vtx;
  SolidClass* solid;
  int x = SOLID->X;
  int y = SOLID->Y;
  int z = SOLID->Z;
  int e = SOLID->Element;
  switch(n)
  { case 0:
      vtx = (VertexClass*)TargetList->Find_OfType(*(int*)p,&VertexControl);
      SOLID->P = vtx;
      break;
    case 1:
      x = *((int*)p);
      break;
    case 2:
      y = *((int*)p);
      break;
    case 3:
      z = *((int*)p);
      break;
    case 4:
      e = *((int*)p);
      break;
    default:
      return(-1);
  }
  if(x != SOLID->X || y != SOLID->Y || z != SOLID->Z || e != SOLID->Element) {
    // dimensions have changed
    // ************ need to finish
  }
  return(0);
}

void SolidControlClass::GetRealLimits(ObjectClass* object, fixed& x1, fixed& y1, fixed& z1,
                                                          fixed& x2, fixed& y2, fixed& z2)
{
  VTX p;
  GetOtherVertex(&p,object);
  x1 = MIN(SOLID->P->Pos.X,p.X);
  x2 = MAX(SOLID->P->Pos.X,p.X);
  y1 = MIN(SOLID->P->Pos.Y,p.Y);
  y2 = MAX(SOLID->P->Pos.Y,p.Y);
  z1 = MIN(SOLID->P->Pos.Z,p.Z);
  z2 = MAX(SOLID->P->Pos.Z,p.Z);
}

bool SolidControlClass::GetLimits(ObjectClass* object,int& x,int& y,int &z,
                                                       int& w,int& h,int &d)
{
  // use current visual mode to map object limits onto a flat plane
  int x1,y1,x2,y2,z1,z2;
  Current_Visual->RealToFlat(&SOLID->P->Pos,x1,y1,z1);
  VTX p;
  GetOtherVertex(&p,object);
  Current_Visual->RealToFlat(&p,x2,y2,z2);
  x = MIN(x1,x2);
  y = MIN(y1,y2);
  z = MIN(z1,z2);
  w = MAX(x1,x2)-x + 1;
  h = MAX(y1,y2)-y + 1;
  d = MAX(z1,z2)-z + 1;
  // don't use x,y,w,h for any solid drawing nosense
  return(Current_Visual->Show(Type,SOLID->P->Layer));
}

void SolidControlClass::SetLimits(ObjectClass* object,int x,int y,
                                                       int w,int h)
{
  // not sure if changing the vertex settings makes any sense
  // use current visual mode to map flat plane onto object limits
  //  VertexControl.SetLimits(SOLID->P1,x,y,w,h);
  //  VertexControl.SetLimits(SOLID->P2,x,y,w,h);
}

void SolidControlClass::Draw(BITMAP* bmp,ObjectClass* object)
{
  int x1,x2,y1,y2,z1,z2;
  int x,y,z;
  VTX p,q;
  GetOtherVertex(&q,object);
  // get limits of solid in current view
  for(int i = 0; i < 12; i++) {
    for(int j = 0; j < 2; j++) {
      if(CubeLine[i][j].X)
        p.X = SOLID->P->Pos.X;
      else
        p.X = q.X;
      if(CubeLine[i][j].Y)
        p.Y = SOLID->P->Pos.Y;
      else
        p.Y = q.Y;
      if(CubeLine[i][j].Z)
        p.Z = SOLID->P->Pos.Z;
      else
        p.Z = q.Z;
      Current_Visual->RealToFlat(&p,x,y,z);
      if(i==0 && j==0) {  // first check
        x1 = x2 = x;
        y1 = y2 = y;
        z1 = z2 = z;
      } else {
        if(x < x1)
          x1 = x;
        if(x > x2)
          x2 = x;
        if(y < y1)
          y1 = y;
        if(y > y2)
          y2 = y;
        if(z < z1)
          z1 = z;
        if(z > z2)
          z2 = z;
      }
    }
  }
  if(x2 < Current_Visual->X1 ||
     x1 > Current_Visual->X2 ||
     y2 < Current_Visual->Y1 ||
     y1 > Current_Visual->Y2)
     return;  // nothing to draw
  // clip these limits to the limits of view
  if(x1 < Current_Visual->X1)
    x1 = Current_Visual->X1;
  if(x2 > Current_Visual->X2)
    x2 = Current_Visual->X2;
  if(y1 < Current_Visual->Y1)
    y1 = Current_Visual->Y1;
  if(y2 > Current_Visual->Y2)
    y2 = Current_Visual->Y2;
  // convert z limits to near and far
  int zn;  // nearest solid
  int zf;  // farthest solid
  if(z1 > z2) {
    zf = z1;
    zn = z2;
  } else {
    zf = z2;
    zn = z1;
  }

  // diagnostic
  rect(bmp,x1,y1,x2,y2,WHITE);  // check that limits are correct

  // scan the solid limits in view
  for(y = y1; y <= y2; y++) {
    int xc = 0;
    for(x = x1; x <= x2; x++) {
      VTX p1,p2;
      fixed dpx,dpy,dpz;
      fixed dxx,dxy,dxz;
      int mi;
      if(Current_Visual->Mode || xc < 2) {
        // convert scan position back to real positions for near and far
        // first save current depth and alter to limits
        fixed d = Current_Visual->Depth;
        Current_Visual->Depth = itofix(zn);
        Current_Visual->FlatToReal(&p1,x,y);
        if(Current_Visual->Mode || (x == x1 && y == y1))
        { // in isometric mode, only the first run needs to find the increment vector
          Current_Visual->Depth = itofix(zf);
          Current_Visual->FlatToReal(&p2,x,y);

          dpx = p2.X - p1.X;
          dpy = p2.Y - p1.Y;
          dpz = p2.Z - p1.Z;
          // find maximum distance traveled in any axis
          fixed mm = 0;
          if(dpx < 0) {
            d = -dpx;
            if(d > mm)
              mm = d;
          } else if(dpx > mm)
            mm = dpx;
          if(dpy < 0) {
            d = -dpy;
            if(d > mm)
              mm = d;
          } else if(dpy > mm)
            mm = dpy;
          if(dpz < 0) {
            d = -dpz;
            if(d > mm)
              mm = d;
          } else if(dpz > mm)
            mm = dpz;
          // mm should now equal the maximum distance
          // divide by 2 exponent of Element size
          if(SOLID->Element < 0)
            mm = (mm<<(-SOLID->Element));
          else
            mm = (mm>>SOLID->Element);
          mi = fixtoi(mm);
          // mm now describes the number of crawl increments
          // find crawl increment
          dpx = fdiv(dpx,mm);
          dpy = fdiv(dpy,mm);
          dpz = fdiv(dpz,mm);
        }
        // restore depth
        Current_Visual->Depth = d;
      } else {
        p1.X = fadd(p1.X,dxx);
        p1.Y = fadd(p1.Y,dxy);
        p1.Z = fadd(p1.Z,dxz);
      }
      // ray trace - crawl along ray to find object
      // start at a position one step back
      // only needed for illumination calc - abandoned for the moment
      fixed fx = fsub(fsub(p1.X,dpx),SOLID->P->Pos.X);
      fixed fy = fsub(fsub(p1.Y,dpy),SOLID->P->Pos.Y);
      fixed fz = fsub(fsub(p1.Z,dpz),SOLID->P->Pos.Z);
      fixed es = 1<<(16+SOLID->Element);
      int lx,ly,lz;
      for(int m = -1; m < mi; m++) {
        // get index into solid array
        int ex = fixtoi(fdiv(fx,es));
        int ey = fixtoi(fdiv(fy,es));
        int ez = fixtoi(fdiv(fz,es));
        // do -1 increment first just to get direction
        if(m >= 0 &&
           ex >= 0 && ex < SOLID->X &&    // within bounds of SOLID's array
           ey >= 0 && ey < SOLID->Y &&
           ez >= 0 && ez < SOLID->Z) {

           //diagnostic to check bounds cover
           // int col = makecol((long)ex*255/SOLID->X,(long)ey*255/SOLID->Y,(long)ez*255/SOLID->Z);
           //  putpixel(bmp,x,y,col);
           //  break;
           if(SOLID->State(ex,ey,ez)) {
             int col;
             col = EGAColor[ez%16];
             /* for simple side color illumination
             if(ex > lx)
               col = WHITE;
             else if(ex < lx)
               col = BLACK;
             else if(ey > ly)
               col = LIGHT_GREEN;
             else if(ey < ly)
               col = GREEN;
             else if(ez > lz)
               col = LIGHT_BLUE;
             else
               col = BLUE;
             */
             putpixel(bmp,x,y,col);  // should alter color depending on direction?
             break; // no point going any further unless it were translucent
           }
        }
        lx = ex;
        ly = ey;
        lz = ez;
        fx += dpx;
        fy += dpy;
        fz += dpz;
      }
      if(xc == 0) {
        // save p1 for calc of change of p1
        p2.X = p1.X;
        p2.Y = p1.Y;
        p2.Z = p1.Z;
      } else if(xc == 1) {
        // calc change of p1 with x increment
        dxx = fsub(p1.X,p2.X);
        dxy = fsub(p1.Y,p2.Y);
        dxz = fsub(p1.Z,p2.Z);
      }
      xc++;
    }
  }
}

void SolidControlClass::Indicate(BITMAP* bmp, ObjectClass* object,
     int,int,int,int, int fg, int bg)
{
  int x1,y1,x2,y2, z;
  VTX p[2];
  VTX q;
  GetOtherVertex(&q,object);
  for(int i = 0; i < 12; i++) {
    for(int j = 0; j < 2; j++) {
      if(CubeLine[i][j].X)
        p[j].X = SOLID->P->Pos.X;
      else
        p[j].X = q.X;
      if(CubeLine[i][j].Y)
        p[j].Y = SOLID->P->Pos.Y;
      else
        p[j].Y = q.Y;
      if(CubeLine[i][j].Z)
        p[j].Z = SOLID->P->Pos.Z;
      else
        p[j].Z = q.Z;
    }
    Current_Visual->RealToFlat(&p[0],x1,y1,z);
    Current_Visual->RealToFlat(&p[1],x2,y2,z);
    dotted_line(bmp,x1,y1,x2,y2,fg,bg);
  }
}

