/*  Source file for the Edit3d program by Robert Parker using the Allegro
    and Bgui2 - see credits else where - also see BasicGUI source code.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "visual.h"

VisualClass* Current_Visual;

#ifndef M_PI
  #define M_PI   205887   // fixed PI = 3.1415862 instead of 3.1415927
#endif

#ifndef Fdeg
  #define Fdeg
  #define deg90  (64<<16)
  #define deg180 (128<<16)
  #define deg270 (192<<16)
#endif


CameraClass::CameraClass()
{
  Sc.Z = -20<<16;
  FOV = 32<<16;
  Aspect = 1<<16;
}

void CameraClass::MakeMatrix()
{
  // rotations angles to greater than 0 and less than 360
  Sc.Normalise();
  
  MATRIX roller;
//   int x, y, w, h;
  fixed xfront, yfront, zfront;
  fixed xup, yup, zup;

  /* calculate the in-front vector */
  xfront = fsin(Sc.Ry) * fcos(Sc.Rx);
  yfront = fsin(Sc.Rx);
  zfront = fcos(Sc.Ry) * fcos(Sc.Rx);

  /* rotate the up vector around the in-front vector by the roll angle */
  get_vector_rotation_matrix(&roller, xfront, yfront, zfront, Sc.Rz*128<<16/M_PI);
  apply_matrix(&roller, 0, -1, 0, &xup, &yup, &zup);

  /* build the camera matrix */
  get_camera_matrix(&Matrix,
  //     -Sc.X, -Sc.Y, -Sc.Z,        /* camera position */
       0, 0, 0,
       xfront, yfront, zfront,  /* in-front vector */
       xup, yup, zup,           /* up vector */
       FOV,                     /* field of view */
       Aspect);                 /* aspect ratio */
}

int Views = 0;   // doesn't keep track of deletions but will always be unique

VisualClass::VisualClass()
{ X=Y=X1=Y1=X2=Y2=0;
  Depth = 0;   // distance from camera to drawing plane (found in view.cpp)
  Scale = 0;   // 0 is equal 1:1
  Mode = 0;
  Numeric = 0; // free float report
  Linked = 1;
  ID = Views;
  Views++;
  IncludeTypes  = -1;  // ie. All
  ShowTypes     = -1;
  Background_Buffer = NULL;
  BitmapBackground.File_Name[0] = 0;
  BitmapBackground.X = BitmapBackground.Y = BitmapBackground.W = BitmapBackground.H = 0;
  BitmapBackground.Ref1 = 0;
  BitmapBackground.Ref2 = MAXUNSIGNED;
  Background_Visible = true;
  Background_Fixed_Aspect = false;
  Link_Background = true;
  Frame_Advance = 1;
  Joystick_Enabled = false;
  Refresh = false;
  Render = false;
}

VisualClass::~VisualClass()
{ // for some reason destroying bitmap in a static destructor is a no no.
  // so use Unload_Background below during a deinit process if visualclass is a static
  if(Background_Buffer)
    destroy_bitmap(Background_Buffer);
}

void VisualClass::SetView(int x1, int y1, int x2, int y2)
{
  X1 = x1;
  Y1 = y1;
  X2 = x2;
  Y2 = y2;
  X = (X1+X2)/2;
  Y = (Y1+Y2)/2;
  fX = itofix(X);
  fY = itofix(Y);
}

int VisualClass::Save(char* path)
{ PACKFILE* file = pack_fopen(path,"wp");
  if(!file)
    return(1);
  if( pack_fwrite(&Camera.Sc,       sizeof(Camera.Sc),       file) != sizeof(Camera.Sc) ||
      pack_fwrite(&Camera.FOV,      sizeof(Camera.FOV),      file) != sizeof(Camera.FOV) ||
      pack_fwrite(&Camera.Aspect,   sizeof(Camera.Aspect),   file) != sizeof(Camera.Aspect) ||
      pack_fwrite(&BitmapBackground,sizeof(BitmapBackground),file) != sizeof(BitmapBackground) )
  { pack_fclose(file);
    return(2);
  }
  pack_fclose(file);
  return(0);
}

int VisualClass::Load(char* path)
{ PACKFILE* file = pack_fopen(path,"rp");
  if(!file)
    return(1);
  if( pack_fread(&Camera.Sc,       sizeof(Camera.Sc),       file) != sizeof(Camera.Sc) ||
      pack_fread(&Camera.FOV,      sizeof(Camera.FOV),      file) != sizeof(Camera.FOV) ||
      pack_fread(&Camera.Aspect,   sizeof(Camera.Aspect),   file) != sizeof(Camera.Aspect) ||
      pack_fread(&BitmapBackground,sizeof(BitmapBackground),file) != sizeof(BitmapBackground) )
  { pack_fclose(file);
    return(2);
  }
  pack_fclose(file);
  return(0);
}

void VisualClass::Set_Origin()
{
  // calc. axis points on screen andgrid offsets
  VTX origin = { 0, 0, 0};
  long z;
  RealToFlat(&origin,Xa,Ya,z);
  int gx = Grid_W[Grid];
  Xg = X1 + (Xa-X1)%gx;
//  Xg = Xa + (X1 - Xa)/gx*gx;
  int gy = Grid_H[Grid];
  Yg = Y1 + (Ya-Y1)%gy;
//  Yg = Ya + (Y1 - Ya)/gy*gy;
}

void VisualClass::AxisSign(BITMAP* bmp)
{ // if(!Mode) // should work regardless of mode
  // work out which axis is where
    QUAT axis_quat;
    get_rotation_quat(&axis_quat,
      fixtof(Camera.Sc.Rx),fixtof(Camera.Sc.Ry),fixtof(Camera.Sc.Rz));
    float xs[3], ys[3], zs[3];
    for(int a = 0; a < 3; a++)
    { int x = 0, y = 0, z = 0;
      switch(a)
      { case 0:
          x = 16;
          break;
        case 1:
          y = 16;
          break;
        case 2:
          z = 16;
          break;
      }
      apply_quat(&axis_quat,x,y,z,xs+a,ys+a,zs+a);
    }
                   //    0       1       2      n/a     n/a      5       6       7
    char order[24] = { 0,1,2,  1,0,2,  0,2,1,  0,0,0,  0,0,0,  1,2,0,  2,0,1,  2,1,0 };
    char flag = (zs[1] > zs[0]) | ((zs[2] > zs[1])<<1)  | ((zs[2] > zs[0])<<2);
    text_mode(-1);   // transparent
    for(int a = 0; a < 3; a++)
    { char ac[2];
      ac[1] = 0;
      int o = order[flag*3+a];
      switch(o)
      { case 0:
          ac[0] = 'X';
          break;
        case 1:
          ac[0] = 'Y';
          break;
        case 2:
          ac[0] = 'Z';
          break;
      }
      line(bmp, X1 + 24, Y1 + 24, X1 + 24 + xs[o], Y1 + 24 - ys[o], Axis_Colors[o]);
      textout(bmp, font, ac, X1 + 20 + xs[o], Y1 + 20 - ys[o], Axis_Colors[o]);
    }
}

int VisualClass::Load_Background(char* path, int frame)
{ BITMAP *temp_buffer;
  RGB pal[256];
  int frames = 1;
  if(strcmp(path + strlen(path) - 4,".avi"))
  { temp_buffer = load_bitmap(path, pal);   // not avi so assume .bmp or .pcx etc.
    // convert to current palette
    if(temp_buffer)
    { for(int x = 0; x < temp_buffer->w; x++)
      {  for(int y = 0; y < temp_buffer->h; y++)
         { int c = getpixel(temp_buffer,x,y);
           putpixel(temp_buffer,x,y,makecol(pal[c].r<<2,pal[c].g<<2,pal[c].b<<2));
         }
      }
    }
  }
  else
    temp_buffer = Load_Frame(path, frame, &frames, &AVI_Status);
  if(!temp_buffer)
    return(1);
    
  int ow = 0;
  int oh = 0;
  if(Background_Buffer)
  { ow = Background_Buffer->w;
    oh = Background_Buffer->h;
    destroy_bitmap(Background_Buffer);
  }
  Background_Buffer = temp_buffer;
  BitmapBackground.Frames = frames;
  if(!BitmapBackground.W)
  { BitmapBackground.X = 0;
    BitmapBackground.Y = 0;
  }
  // if new image is the same source size then retain same enlargement
  if(ow != Background_Buffer->w)
    BitmapBackground.W = FlatToReal(Background_Buffer->w);
  if(oh != Background_Buffer->h)
    BitmapBackground.H = FlatToReal(Background_Buffer->h,true);
  return(0);
}

int VisualClass::Previous_Frame()
{ if(BitmapBackground.Frames <= 1 || !BitmapBackground.Frame)
    return(1);
  if(!AVI_Status.uSecPerFrame || Frame_Advance < 100)   // advance by frames
  { if(BitmapBackground.Frame < Frame_Advance)
      BitmapBackground.Frame = 0;
    else
      BitmapBackground.Frame -= Frame_Advance;
    AVI_Position = BitmapBackground.Frame*AVI_Status.uSecPerFrame;
  }
  else
  { if(AVI_Position > Frame_Advance)
    { AVI_Position -= Frame_Advance;
      BitmapBackground.Frame = AVI_Position / AVI_Status.uSecPerFrame;
    }
    else
      BitmapBackground.Frame = 0;
  }
  return(Load_Background(BitmapBackground.File_Name,BitmapBackground.Frame));
}

int VisualClass::Next_Frame()
{ if(BitmapBackground.Frames <= 1 || BitmapBackground.Frame >= BitmapBackground.Frames)
    return(1);
  if(!AVI_Status.uSecPerFrame || Frame_Advance < 100)   // advance by frames
  { BitmapBackground.Frame += Frame_Advance;
    AVI_Position = BitmapBackground.Frame*AVI_Status.uSecPerFrame;
  }
  else
  { AVI_Position += Frame_Advance;
    BitmapBackground.Frame = AVI_Position / AVI_Status.uSecPerFrame;
  }
  if(BitmapBackground.Frame >= BitmapBackground.Frames)
    BitmapBackground.Frame = BitmapBackground.Frames - 1;
  return(Load_Background(BitmapBackground.File_Name,BitmapBackground.Frame));
}

// Backgrounds need to be destroyed before static destructor
void VisualClass::Unload_Background()
{ if(Background_Buffer)
    destroy_bitmap(Background_Buffer);
  Background_Buffer = NULL;
  BitmapBackground.W = 0;
  BitmapBackground.File_Name[0] = 0;
}

void VisualClass::Get_Background_Corners(long &x1, long &y1, long &x2, long &y2)
{ x1 = Xa + RealToFlat(BitmapBackground.X);
  x2 = x1 + RealToFlat(BitmapBackground.W);
  y2 = Ya - RealToFlat(BitmapBackground.Y,true);
  y1 = y2 - RealToFlat(BitmapBackground.H,true);
}

void VisualClass::Set_Background_Corners(long x1, long y1, long x2, long y2)
{ BitmapBackground.X = FlatToReal(x1-Xa);
  BitmapBackground.Y = FlatToReal(Ya-y2,true);
  fixed w = FlatToReal(x2-x1);
  fixed h = FlatToReal(y2-y1,true);
  if(Background_Fixed_Aspect != 0)
  { fixed eh = fmul(Background_Fixed_Aspect,w);
    fixed ew = fdiv(h,Background_Fixed_Aspect);
    // work out the smaller fit
    if(ew < w)
    { BitmapBackground.W = ew;
      BitmapBackground.H = h;
    }
    else
    { BitmapBackground.W = w;
      BitmapBackground.H = eh;
    }
  }
  else
  { BitmapBackground.W = w;
    BitmapBackground.H = h;
  }
}

void VisualClass::Reset_Bitmap_Pos()
{ BitmapBackground.X = 0; // or FlatToReal(Xa-X);
  BitmapBackground.Y = 0; // or FlatToReal(Ya-Y,true);
}

void VisualClass::Reset_Bitmap_Size()
{ if(Background_Buffer)
  { BitmapBackground.W = FlatToReal(Background_Buffer->w);
    BitmapBackground.H = FlatToReal(Background_Buffer->h,true);
  }
}

int VisualClass::Draw_Background(BITMAP* bmp)
{ if(!Background_Buffer)
    return(0);
  if(!BitmapBackground.W)
  { Reset_Bitmap_Pos();
    Reset_Bitmap_Size();
  }
  // source co-ordinates
  long x1 = 0;               // default points within buffer
  long x2 = Background_Buffer->w;
  long y1 = 0;
  long y2 = Background_Buffer->h;
  // destination co-ordinates for infinitely large screen
  long vx1,vy1,vx2,vy2;
  Get_Background_Corners(vx1,vy1,vx2,vy2);
  if(vx1 > X2 || vx2 < X1 || vy1 > Y2 || vy2 < Y1)
    return(0);  // out of range
  double scale = fixtof(Camera.Sc.Scale);
  double aspect = fixtof(Camera.Aspect);
  if(vx1 < X1)
  { x1 = (X1 - vx1)/scale;
    vx1 = X1;
  }
  if(vx2 > X2)
  { x2 += (X2 - vx2)/scale;
    vx2 = X2;
  }
  if(vy1 < Y1)
  { y1 = (Y1 - vy1)/scale/aspect;
    vy1 = Y1;
  }
  if(vy2 > Y2)
  { y2 += (Y2 - vy2)/scale/aspect;
    vy2 = Y2;
  }
  stretch_blit(Background_Buffer, bmp, x1, y1, x2-x1, y2-y1, vx1, vy1, vx2-vx1, vy2-vy1);
  return(1);
}

//fixed Dx, Dy, Dz;

void VisualClass::MakeTransMatrix()
{
//  MATRIX mx, my, mz, mo;
  //get_translation_matrix(&mt,-Camera.Sc.X,-Camera.Sc.Y,-Camera.Sc.Z);
  // reverse rotations to z, y, and then x
//  get_rotation_matrix(&mr,-Camera.Sc.Rx,-Camera.Sc.Ry,-Camera.Sc.Rz);
/*  get_x_rotate_matrix(&mx,Camera.Sc.Rx);  // opposite to rev translate?
  get_y_rotate_matrix(&my,Camera.Sc.Ry);
  get_z_rotate_matrix(&mz,Camera.Sc.Rz);
  matrix_mul(&mz,&my,&mo);
  matrix_mul(&mo,&mx,TransMatrix);
  // can use transition in RevMatrix
  qscale_matrix(&TransMatrix,Camera.Sc.Scale);
*/
//    get_transformation_matrix(&TransMatrix,Camera.Sc.Scale,
//      Camera.Sc.Rx, Camera.Sc.Ry, Camera.Sc.Rz,
//      0, 0, 0);
//      -Camera.Sc.X, -Camera.Sc.Y, -Camera.Sc.Z);
    get_rotation_matrix(&TransMatrix,Camera.Sc.Rx, Camera.Sc.Ry, Camera.Sc.Rz);
//    apply_matrix(&TransMatrix,Camera.Sc.X,Camera.Sc.Y,Camera.Sc.Z,&Dx,&Dy,&Dz);
    qscale_matrix(&TransMatrix,Camera.Sc.Scale);
}

void VisualClass::MakeRevMatrix()
{
  MATRIX mt, mx, my, mz, mo, mr;
  fixed revscale = fdiv((1<<16),Camera.Sc.Scale);
  get_translation_matrix(&mt,Camera.Sc.X,Camera.Sc.Y,Camera.Sc.Z); //Dx,Dy,Dz);
//    fmul(Camera.Sc.X,revscale),
//    fmul(Camera.Sc.Y,revscale),
//    fmul(Camera.Sc.Z,revscale));
  // reverse rotations to z, y, and then x
  get_x_rotate_matrix(&mx,Camera.Sc.Rx);  // opposite to normal translate?
  get_y_rotate_matrix(&my,Camera.Sc.Ry);
  get_z_rotate_matrix(&mz,Camera.Sc.Rz);
  matrix_mul(&mx,&my,&mo);
  matrix_mul(&mo,&mz,&mr);
  // can use transition in RevMatrix
  qscale_matrix(&mr,revscale);
  matrix_mul(&mr,&mt,&RevMatrix);
}

#include "errno.h"  // for errno

void VisualClass::Draw(BITMAP* bmp, bool focus, VTX* DepthVTX)
{
  Refresh = false;
  if(focus)
    Rect(bmp,WHITE);
  else
    Rect(bmp,BLACK);

  // calculate camera scale base on log2 ratio in Scale
  Camera.Sc.Scale= 1<<16;
  if(Scale > 0)
    Camera.Sc.Scale *= Scale;
  else if(Scale < 0)
    Camera.Sc.Scale /= -Scale;
//  MakeTransMatrix();
  if(Mode > 1) {
    Camera.MakeMatrix();
    TransMatrix = Camera.Matrix;
  } else
    MakeTransMatrix();

  // RevMatrix is not used in Draw, but rather in subsequent edit FlatToReal()
  MakeRevMatrix();
//  get_transformation_matrix(&RevMatrix,revscale,
//       -Camera.Sc.Rx,-Camera.Sc.Ry,-Camera.Sc.Rz,Camera.Sc.X,Camera.Sc.Y,Camera.Sc.Z);
    
  if(Mode < 2)
  { if(!(Camera.Sc.Rx%deg90 | Camera.Sc.Ry%deg90 | Camera.Sc.Rz%deg90))
    { // draw axis
      Set_Origin();  // now that scale and matrix'are set
      Draw_Grid(bmp, X2-Xg, Y2-Yg, Xg, Yg);  // should use scale or indicate grid scale
/*      if(Xa > X1 && Xa < X2)
        vline(bmp,Xa,Y1,Y2,LIGHT_GREEN);
      if(Ya > Y1 && Ya < Y2)
        hline(bmp,X1,Ya,X2,LIGHT_GREEN);
*/
    }
//    else
//      Mode = 0;  // just something to break on
    // else new to find grid point, dot by dot and axis, line by line
  }
  if(Mode == 2)  // prespective mode is kept simple
  { set_projection_viewport(X1, Y1, X2, Y2);
    return;
  }

  if(Background_Visible)
    Draw_Background(bmp);

    
  VTX maxlim,minlim;
  Extremity(&maxlim,&minlim);
  errno = 0;
  VTX axisvert;
  // draw X axis
  long x1,y1,x2,y2,z;
  axisvert.X = maxlim.X;
  axisvert.Y = 0;
  axisvert.Z = 0;
  RealToFlat(&axisvert,x1,y1,z);
  axisvert.X = minlim.X;
  RealToFlat(&axisvert,x2,y2,z);
  if(!errno)     // if there has been an error, better not to show axis
    line(bmp,x1,y1,x2,y2,Axis_Colors[0]);
  errno = 0;
  // draw y axis
  axisvert.X = 0;
  axisvert.Y = maxlim.Y;
  RealToFlat(&axisvert,x1,y1,z);
  axisvert.Y = minlim.Y;
  RealToFlat(&axisvert,x2,y2,z);
  if(!errno)
    line(bmp,x1,y1,x2,y2,Axis_Colors[1]);
  errno = 0;
  // draw z axis
  axisvert.Y = 0;
  axisvert.Z = maxlim.Z;
  RealToFlat(&axisvert,x1,y1,z);
  axisvert.Z = minlim.Z;
  RealToFlat(&axisvert,x2,y2,z);
  if(!errno)
    line(bmp,x1,y1,x2,y2,Axis_Colors[2]);
  errno = 0;
  // put axis indicator in top left corner
  AxisSign(bmp);
  if(!focus && DepthVTX != NULL)  // show depth relevant for the current visual
  { axisvert.Z = 0;
    RealToFlat(&axisvert,x1,y1,z);  // origin
    RealToFlat(DepthVTX,x2,y2,z);
    if(!errno)
      line(bmp,x1,y1,x2,y2,Axis_Colors[3]);  // is there a Color[3]
    errno = 0;
  }
}

void VisualClass::Rect(BITMAP* bmp, int color)
{
  set_clip(bmp,X1,Y1,X2,Y2);
  rect(bmp,X1,Y1,X2,Y2, color);
}

void VisualClass::RealToFixed(VTX* vtx, fixed& fx, fixed& fy, fixed& fz)
{   // don't try passing an array object to x, y, or z
  // the simple solution - kept incase the need for speed arises
    // try for the simple solution first
    // may not improve speed by much
//    bool notsimple = FALSE;
    fixed dx = vtx->X - Camera.Sc.X;
    fixed dy = vtx->Y - Camera.Sc.Y;
    fixed dz = vtx->Z - Camera.Sc.Z;
/*    if(!Camera.Sc.Rz)   // roll zero
    { if(!Camera.Sc.Rx) // pitch zero
      { fy = dy;
        if(!Camera.Sc.Ry)  // heading zero
        { fx = dx;
          fz = dz;
        }
        else if(Camera.Sc.Ry == deg90)
        { fx = dz;
          fz = dx;
        }
        else if(Camera.Sc.Ry == deg180)
        { fx = -dx;
          fz = dz;
        }
        else if(Camera.Sc.Ry == deg270)
        { fx = -dz;
          fz = -dx;
        }
        else
          notsimple = TRUE;
      }
      else if(Camera.Sc.Rx == deg90)
      { if(!Camera.Sc.Ry)
        { fx = dx;
          fy = dz;
          fz = dy;
        }
        else
          notsimple = TRUE;
      }
      else
        notsimple = TRUE;
    }
    else // it gets complicated
      notsimple = TRUE;
    if(notsimple)
    {  */
//      apply_matrix(&TransMatrix, vtx->X, vtx->Y, vtx->Z, &fx, &fy, &fz);
      apply_matrix(&TransMatrix, dx, dy, dz, &fx, &fy, &fz);
      if(Mode == 1) {  // dodgy perspective
        if(fz < 0) {
          fx = 0;
          fy = 0;
        } else {
          fixed div = fz/Camera.FOV + 1<<16;
          fx = fdiv(fx,div);
          fy = fdiv(fy,div);
 /// should really calc distance from center and bring it in a polar reduction
        }
      }
//        persp_project(fx,fy,fz,&fx,&fy);
//    }
  if(Mode < 2)  // convert to screen position relative to center of view
  { fx += fX;
    fy = fY - fy;
  }
/*
      flags[c] = 0;

      // set flags if this vertex is off the edge of the screen
      // this only applies for 45 degree field of view?
      if (fx < -fz)
        flags[c] |= 1;
      else if (fx > fz)
        flags[c] |= 2;

      if (fy < -fz)
        flags[c] |= 4;
      else if (fy > fz)
        flags[c] |= 8;

      if (v[c]->z < 0.1)
        flags[c] |= 16;
*/
  // should check for overflow before applying scale
/*
  if(Scale > 1)
  { fx *= Scale;
    fy *= Scale;
  }
  else if(Scale < -1)
  { fx /= -Scale;
    fy /= -Scale;
  }
*/
}

void VisualClass::RealToFlat(VTX* vtx, long& x, long& y, long& z)
{   // don't try passing an array object to x, y, or z
  fixed fx, fy, fz;
  RealToFixed(vtx,fx,fy,fz);
  x = fixtoi(fx);  // converted to int ????? then back to long ????
  y = fixtoi(fy);
  z = fixtoi(fz);
}

// this version limited to usefull int size
const long IntMax = MAXINT - 2048;
const long IntMin = MININT + 2048;

int IntLimit(long xl)
{ int x;
  if(xl > IntMax)
    x = IntMax;
  else if(xl < IntMin)
    x = IntMin;
  else
    x = xl;
  return(x);
}

void VisualClass::RealToFlat(VTX* vtx, int& x, int& y, int& z)
{
  long xl,yl,zl;
  RealToFlat(vtx,xl,yl,zl);
  x = IntLimit(xl);
  y = IntLimit(yl);
  z = IntLimit(zl);
}

void VisualClass::SetDepth(VTX* vtx)
{
  fixed x,y;
  RealToFixed(vtx,x,y,Depth);
}

void VisualClass::FlatToReal(VTX* vtx, long x, long y)  // know way of knowing z
{
  if(Mode > 1) // how do you do perspective reversal?
    return;
    
  fixed fx = itofix(x - X);
  fixed fy = itofix(Y - y);
  fixed fz = Depth;
  if(Mode == 1) {
    fixed div = fz/Camera.FOV;
    if(div < 0)
      div = 1<<16 - div;
    else
      div += 1<<16;
    fx = fmul(fx,div);
    fy = fmul(fy,div);
  }
  apply_matrix(&RevMatrix, fx, fy, fz, &(vtx->X), &(vtx->Y), &(vtx->Z));
  // RevMatrix applies position transition
  return;
/*     // the remains of the simple solution - kept in case a need for speed arises
  if(Scale > 1)
  { fx /= Scale;
    fy /= Scale;
  }
  else if(Scale < -1)
  { fx *= -Scale;
    fy *= -Scale;
  }
  // try for the simple solution first
  // may not improve speed by much
  bool notsimple = FALSE;
  if(!Camera.Sc.Rz)   // roll zero
  { if(!Camera.Sc.Rx) // pitch zero
    { vtx->Y = fy;
      if(!Camera.Sc.Ry)  // heading zero
      { vtx->X = fx;
        vtx->Z = fz;
      }
      else if(Camera.Sc.Ry == deg90)
      { vtx->X = fz;
        vtx->Z = fx;
      }
      else if(Camera.Sc.Ry == deg180)
      { vtx->X = -fx;
        vtx->Z = fz;
      }
      else if(Camera.Sc.Ry == deg270)
      { vtx->X = -fz;
        vtx->Z = -fx;
      }
      else
        notsimple = TRUE;
    }
    else if(Camera.Sc.Rx == deg90)
    { if(!Camera.Sc.Ry)
      { vtx->X = fx;
        vtx->Y = fz;
        vtx->Z = fy;
      }
      else
        notsimple = TRUE;
    }
    else
      notsimple = TRUE;
  }
  else // it gets complicated
    notsimple = TRUE;

  fixed fx,fy,fz;
  if(notsimple)
    apply_matrix_f(Camera.Matrix, x, y, z, &fx, &fy, &fz);
  else if(Mode)
  { fx = itofix(x);
    fy = itofix(y);
    fz = itofix(z);
  }
*/
/*
      flags[c] = 0;

      // set flags if this vertex is off the edge of the screen
      // this only applies for 45 degree field of view?
      if (fx < -fz)
        flags[c] |= 1;
      else if (fx > fz)
        flags[c] |= 2;

      if (fy < -fz)
        flags[c] |= 4;
      else if (fy > fz)
        flags[c] |= 8;

      if (v[c]->z < 0.1)
        flags[c] |= 16;
*/
/*
  if(!notsimple)
  { vtx->X += Camera.Sc.X;
    vtx->Y += Camera.Sc.Y;
    vtx->Z += Camera.Sc.Z;
  }
*/
}

// helper functions for conversion of projected points

long VisualClass::RealToFlat(fixed x, bool aspect)
{
  double f = fixtof(x)*fixtof(Camera.Sc.Scale);
  if(aspect)
    f *= fixtof(Camera.Aspect);
  return((long)f);
}

/*
// old version does seem to work
long VisualClass::RealToFlat(fixed x, bool aspect)
{
  fixed f = fmul(x,Camera.Sc.Scale);
  if(aspect)
    f = fmul(f,Camera.Aspect);
  return(fixtoi(f));
}
*/

fixed VisualClass::FlatToReal(long x, bool aspect)
{
  float r = (float)x/fixtof(Camera.Sc.Scale);
  if(aspect)
    r /= fixtof(Camera.Aspect);
  return(ftofix(r));
}

const VTX VisualLimitMax = { 0x3FFFFFFF,0x3FFFFFFF,0x3FFFFFFF };
const VTX VisualLimitMin = { 0xC0000000,0xC0000000,0xC0000000 };

void VisualClass::Extremity(VTX* maxlim, VTX* minlim)
{
  VTX limit[4];
  errno = 0;  // reset errno
  FlatToReal(limit,X1,Y1);
  FlatToReal(limit+1,X2,Y2);
  FlatToReal(limit+2,X1,Y2);
  FlatToReal(limit+3,X2,Y1);
  if(errno) {
    *maxlim = VisualLimitMax;
    *minlim = VisualLimitMin;
  }
  *maxlim = limit[0];
  *minlim = limit[0];
  for(int l = 1; l < 4; l++)
  { if(limit[l].X > maxlim->X)
      maxlim->X = limit[l].X;
    if(limit[l].Y > maxlim->Y)
      maxlim->Y = limit[l].Y;
    if(limit[l].Z > maxlim->Z)
      maxlim->Z = limit[l].Z;
    if(limit[l].X < minlim->X)
      minlim->X = limit[l].X;
    if(limit[l].Y < minlim->Y)
      minlim->Y = limit[l].Y;
    if(limit[l].Z < minlim->Z)
      minlim->Z = limit[l].Z;
  }
}

void VisualClass::SetCamera(long x, long y)
{ VTX vtx;
  fixed depth = Depth;  // save current depth to drawing plane
  Depth = 0;            // we're interested in moving camera at plane of camera
  FlatToReal(&vtx,x,y);
  Depth = depth;
  Camera.Sc.X = vtx.X;
  Camera.Sc.Y = vtx.Y;
  Camera.Sc.Z = vtx.Z;
}

void VisualClass::CorrectScale()
{  // restrict to 1,2,5 sequence
  int s = abs(Scale);
  while(s > 10)
  { if(s > 20 && s < 30)
      Scale = Scale/s*20;
    s /= 10;
  }
  if(s >=3 && s<= 4)
    Scale = Scale/s*5;
}

void VisualClass::In(long x, long y)
{
  SetCamera(x,y);
  if(Scale >= 10000)
    return;
  if(Scale <= -2)
    Scale /= 2;
  else if(Scale < -1)
    Scale = 1;
  else if(Scale < 1)
    Scale = 2;
  else
    Scale *= 2;
  CorrectScale();
  // will also need to RestoreDepth to all views
}

void VisualClass::Out(long x, long y)
{
  SetCamera(x,y);
  if(Scale <= -100)
    return;
  if(Scale >= 2)
    Scale /= 2;
  else if(Scale > 1)
    Scale = 1;
  else if(Scale > -1)
    Scale = -2;
  else
    Scale *= 2;
  CorrectScale();
  // will also need to RestoreDepth to all views
}

void VisualClass::ScaleIn(int& x, int& y)  // only used for extimate of new view
{
  if(Scale >= -1)
  { x /= 2;
    y /= 2;
  }
  else
  { x *= Scale/2;
    x /= Scale;
    y *= Scale/2;
    y /= Scale;
  }
}

void VisualClass::Pan(int dx, int dy)
{
  SetCamera((X1+X2)/2+dx,(Y1+Y2)/2+dy);
}

void VisualClass::Set(VisualClass* vis)
{
  Scale = vis->Scale;
//  Camera = vis->Camera;
  Camera.Sc.X = vis->Camera.Sc.X;
  Camera.Sc.Y = vis->Camera.Sc.Y;
  Camera.Sc.Z = vis->Camera.Sc.Z;
//  if(Mode)
//    Camera.MakeMatrix();
  // equalise level and block settings
}

bool VisualClass::Include(int type, LayerClass* layer)
{
  // note: IncludeTypes is temporarily altered for new_object in edit3d
  // to only find vertex
  if(type < sizeof(IncludeTypes)*8 && !(IncludeTypes & (1<<type)))
    return(FALSE);
  if(!layer)
    return(TRUE);
  return(ID > LAYER_VIEWS || (layer->ViewFlags[ID] & LAYER_VIEW_FILTER) == 0);
}

bool VisualClass::Show(int type, LayerClass* layer)
{
  if(type < sizeof(ShowTypes)*8 && !(ShowTypes & (1<<type)))
    return(FALSE);
  if(!layer)
    return(TRUE);
  return(ID > LAYER_VIEWS || (layer->ViewFlags[ID] & LAYER_VIEW_HIDE) == 0);
}

int VisualClass::Inside(int x, int y, int z)
{ // not sure what to do with z yet - possible z > 0
  return((x < X1) | ((x > X2)<<1) | ((y < Y1)<<2) | ((y > Y2)<<3));
}

bool VisualClass::Animate()  // returns true if redraw required
{
  long pos[4];
  int c = 0;
  if(Joystick_Enabled)
  { long t = clock();
    long dt;
    if(LastClock < 0)
      dt = 0;
    else
      dt = t - LastClock;
    LastClock = t;
    if(poll_joystick())
      return false;        // not installed
    for(int i  = 0; i < num_joysticks; i++) {
      int sticks = joy[i].num_sticks;
      for(int s = 0; s < sticks; s++) {
         int axises = joy[i].stick[s].num_axis;
         for(int a = 0; a < axises; a++) {
           pos[c] = joy[i].stick[s].axis[a].pos;
           pos[c] *= dt;
           c++;
         }
      }
    }
    unsigned rx = Camera.Sc.Rx >> 16;
    unsigned ry = Camera.Sc.Ry >> 16;
    Camera.Sc.Rx += pos[0]<<8;
    Camera.Sc.Ry += pos[1]<<8;
    if(c > 3) {
      unsigned rz = Camera.Sc.Rz >> 16;
      Camera.Sc.Rz += pos[2]<<8;
      Camera.Sc.Scale += pos[3];
      if((Camera.Sc.Rz >> 16) != rz)
        Refresh =  true;
//      if(Camera  || (pos[3]!=0))
//        return true;   // redraw
    }
    if((Camera.Sc.Rx >> 16) != rx)
      Refresh = true;
    if((Camera.Sc.Ry >> 16) != ry)
      Refresh = true;
  }
//  if(Mode && Refresh)
//    Camera.MakeMatrix(); // only be required for perspective
  return Refresh;
}

