/*  Source file for the Edit3d program by Robert Parker using the Allegro
    and Bgui2 - see credits else where - also see BasicGUI source code.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "view.h"
#include "edit.h"
#include "findvisu.h"
#include "help.h"
#include "open3d.h"
#include "joy.h"
#include "render.h" // for menu_render

// the following really need to be loaded as part of a project file
int HSplit = 0;
int VSplit = 0;
VisualClass Visual[4];
VTX DrawingPlane;

// visual view available area - defaults reset in init
int ViewsX1 = 1, ViewsX2 = 640, ViewsY1 = 16, ViewsY2 = 480;
int View_Gap = 2;    // actually half the gap
#define MIN_VIEW_WIDTH 128
#define MIN_VIEW_HEIGHT 128

// parameters for cursor status
int Status_Pos_X[6];
int Status_Pos_Y = 0;
int Status_Colors[12] = { WHITE,BLUE,  // this should not match axis colors
                          LIGHT_MAGENTA,BLUE,  // this either
                          LIGHT_GREEN,BLUE,   // polar
                          WHITE,GREEN,
                          LIGHT_MAGENTA,GREEN,
                          LIGHT_GREEN,GREEN };
int Status_Disp_Mode = 0;
char* Status_Color_Str[] = { "3D_Abs_Foreground","3D_Abs_Background",
                             "3D_Rel_Foreground","3D_Rel_Background",
                             "3D_Pol_Foreground","3D_Pol_Background",
                             "2D_Abs_Foreground","2D_Abs_Background",
                             "2D_Rel_Foreground","2D_Rel_Background",
                             "2D_Pol_Foreground","2D_Pol_Background"
                           };
int Last_Disp_Mode = -1;   // for adaptive refresh
int Last_Disp_Y = -1;

int Checker(void)  // generic checker also used in edit3d.cpp
{ active_menu->flags ^= D_SELECTED;
  return D_OK;
}

void ReSizeViews()
{
  // limit split point to something reasonable
  if(VSplit < ViewsX1 + MIN_VIEW_WIDTH + View_Gap)
    VSplit = ViewsX1 + MIN_VIEW_WIDTH + View_Gap;
  else if(VSplit > ViewsX2 - MIN_VIEW_WIDTH - View_Gap)
    VSplit = ViewsX2 - MIN_VIEW_WIDTH - View_Gap;
  if(HSplit < ViewsY1 + MIN_VIEW_HEIGHT + View_Gap)
    HSplit = ViewsY1 + MIN_VIEW_HEIGHT + View_Gap;
  else if(HSplit > ViewsY2 - MIN_VIEW_HEIGHT - View_Gap)
    HSplit = ViewsY2 - MIN_VIEW_HEIGHT;
    
  for(int i = 0; i < 4; i++)
  { int x1 = ViewsX1, x2 = ViewsX2, y1 = ViewsY1, y2 = ViewsY2;
    // y1 should really be the height of the menu
    if(Menu_View[SPLIT_H].flags & D_SELECTED)
    { if(i < 2)
        y2 = HSplit - View_Gap;
      else
        y1 = HSplit + View_Gap;
    }
    else if(i >= 2)
      continue;
    if(Menu_View[SPLIT_V].flags & D_SELECTED)
    { if(i & 1)
        x1 = VSplit + View_Gap;
      else
        x2 = VSplit - View_Gap;
    }
    else if(i & 1)
      continue;
    Visual[i].SetView(x1,y1,x2,y2);
  }
}

int Status_Tidy()  // helper for Staus_Mover && Status_Hider
{ ViewsY2 = SCREEN_H - 1;
  if(Status_Pos_Y > ViewsY1)
  { if(Status_Disp)
      ViewsY2 -= text_height(font) + 2;
    ReSizeViews();
    return D_REDRAW;
  }
  return D_OK;
}

extern MENU Menu_Status[];  // predefine

void Init_Views()
{
  // for current project
  // check if config file for project exists and if so switch to it for a tad
  // don' think I'll use a regular config
    ViewsX2 = SCREEN_W - 1;
    ViewsY2 = SCREEN_H - 1;
    HSplit = get_config_int("views","hsplit",SCREEN_H/2);
    VSplit = get_config_int("views","vsplit",SCREEN_W/2);
    // ReSizeViews() is done in Status_Tidy

    int tl = text_length(font,"dX:");
    int sp = tl*14;  // estimate status width
    for(int i = 0; i < 4; i++)
    { Status_Pos_X[i*2] = SCREEN_W - sp*(4-i)/3;
      Status_Pos_X[i*2+1] = Status_Pos_X[i*2] + tl;
    }
    Status_Pos_Y = get_config_int("status_display","pos",SCREEN_H - text_height(font));
    Status_Disp_Mode = 0;
    Status_Disp = Position_Status;
    for(int c = 0; c < 10; c++)
      Status_Colors[c] = get_config_int("status_display",Status_Color_Str[c],Status_Colors[c]);
    if(Status_Tidy() == D_OK)
      ReSizeViews();
    
    Set_Current_Visual(Visual); // ie #0
    Menu_Status[1].flags |= D_SELECTED;  // equates to status_mode 0

    DrawingPlane.X = DrawingPlane.Y = DrawingPlane.Z = 0;
    // can't RestoreDepth() here as RevMatrix not set - so do it after drawviews
}

void DeInit_Views()
{
  // for current project
  // check if
    set_config_int("veiws","hsplit",HSplit);
    set_config_int("veiws","vsplit",VSplit);
    for(int v = 0; v < 4; v++)
      Visual[v].Unload_Background();
}

void SetViewEnable()
{
  if(Menu_View[SPLIT_H].flags & D_SELECTED)
    Menu_View[BL_VIEW].flags = 0;
  else
    Menu_View[BL_VIEW].flags = D_DISABLED;
  if(Menu_View[SPLIT_V].flags & D_SELECTED)
    Menu_View[TR_VIEW].flags = 0;
  else
    Menu_View[TR_VIEW].flags = D_DISABLED;
  if(Menu_View[SPLIT_H].flags & D_SELECTED && Menu_View[SPLIT_V].flags & D_SELECTED)
    Menu_View[BR_VIEW].flags = 0;
  else
    Menu_View[BR_VIEW].flags = D_DISABLED;
  ReSizeViews();
}

int Split_Sizer()
{
  char hs[30], vs[30];
  usprintf(hs,"%d",HSplit);
  usprintf(vs,"%d",VSplit);
  char hst[80], vst[80];
  usprintf(hst,"Horizontal Split (Max:%3d):",SCREEN_W);
  usprintf(vst,"Vertical   Split (Max:%3d):",SCREEN_H);

  DIALOG split_dlg[] =
  {
     /* (proc)            (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                         (dp2)           (dp3) */
    { d_billwin_proc,    0,   0,   320, 80,  0,   0,   0,    0,      0,   0,   (void*)"Split Size", NULL,           NULL },
    { d_billtext_proc,   8,   20,  144, 8,   0,   0,   0,    0,      0,   0,   (void*)hst,          NULL,           NULL },
    { Xedit_proc,        220, 16,  80,  16,  0,   0,   0,    0,      9,   0,   hs,                  NULL,           NULL },
    { d_billtext_proc,   8,   40,  144, 8,   0,   0,   0,    0,      0,   0,   (void*)vst,          NULL,           NULL },
    { Xedit_proc,        220, 36,  80,  16,  0,   0,   0,    0,      9,   0,   vs,                  NULL,           NULL },
    { d_billbutton_proc, 60,  60,  48,  16,  0,   0,   13,   D_EXIT, 0,   0,   (void*)"OK",         NULL,           NULL },
    { d_billbutton_proc, 112, 60,  72,  16,  0,   0,   27,   D_EXIT, 0,   0,   (void*)"Cancel",     NULL,           NULL },
    { NULL,              0,   0,   0,   0,   0,   0,   0,    0,      0,   0,   NULL,                NULL,           NULL }
  };

  // Do the dialog
  set_dialog_color(split_dlg, gui_fg_color, gui_bg_color);
  centre_dialog(split_dlg);
  int res = popup_dialog(split_dlg, -1);  // -1 = no focus

  // Read data back
  if(res == 5)
  { HSplit = ustrtol(hs,NULL,0);
    VSplit = ustrtol(vs,NULL,0);
    ReSizeViews();
  }
  return D_REDRAW;
}

int HSplitChecker()
{ Menu_View[SPLIT_H].flags ^= D_SELECTED;
  SetViewEnable();
  return D_REDRAW;
}

int VSplitChecker()
{ Menu_View[SPLIT_V].flags ^= D_SELECTED;
  SetViewEnable();
  return D_REDRAW;
}

int Split3D()
{
  Menu_View[SPLIT_H].flags |= D_SELECTED;
  Menu_View[SPLIT_V].flags |= D_SELECTED;
  SetViewEnable();
  Visual[0].Camera.Sc.PanTo(0,0,0);  //  is plan view
  Visual[1].Camera.Sc.PanTo(0, -64<<16, 0);  // side view
  Visual[2].Camera.Sc.PanTo(64<<16,0,0);  // front view
  Visual[3].Camera.Sc.PanTo(32<<16,32<<16,0);  // isometric view
  return D_REDRAW;
}

/* Cross hairs used for locating vertex are usually limited to bounds of the view
 * However, if the adjacent view is linked as in the Split3D isometric view
 * it is usefull to continue these cross hair lines into neighboring views
 */
void CrossHair_Limits(int& x1, int& y1, int& x2, int& y2)
{
  x1 = Current_Visual->X1;
  x2 = Current_Visual->X2;
  y1 = Current_Visual->Y1;
  y2 = Current_Visual->Y2;
  if(Menu_View[SPLIT_H].flags & D_SELECTED)
  {
    if((Current_Visual == &Visual[0]) || (Current_Visual == &Visual[2]))
    { if(Visual[0].Camera.Sc.SamePos(&Visual[2].Camera.Sc) &&
         Visual[0].Camera.Sc.Orthagonal() &&
         Visual[2].Camera.Sc.Orthagonal() )
      {
        y1 = Visual[0].Y1;
        y2 = Visual[2].Y2;
      }
    } else
    { if(Visual[1].Camera.Sc.SamePos(&Visual[3].Camera.Sc) &&
         Visual[1].Camera.Sc.Orthagonal() &&
         Visual[3].Camera.Sc.Orthagonal() )
      {
        y1 = Visual[1].Y1;
        y2 = Visual[3].Y2;
      }
    }
  }
  if(Menu_View[SPLIT_V].flags & D_SELECTED)
  {
    if((Current_Visual == &Visual[0]) || (Current_Visual == &Visual[1]))
    { if(Visual[0].Camera.Sc.SamePos(&Visual[1].Camera.Sc) &&
         Visual[0].Camera.Sc.Orthagonal() &&
         Visual[1].Camera.Sc.Orthagonal() )
      {
        x1 = Visual[0].X1;
        x2 = Visual[1].X2;
      }
    } else
    { if(Visual[2].Camera.Sc.SamePos(&Visual[3].Camera.Sc) &&
         Visual[2].Camera.Sc.Orthagonal() &&
         Visual[3].Camera.Sc.Orthagonal() )
      {
        x1 = Visual[2].Y1;
        x2 = Visual[3].Y2;
      }
    }
  }
}


int Status_Color_Setter()
{ int colors[12];  // buffer for color setter
  memcpy(colors,Status_Colors,sizeof(colors)); // copy to buffer
  if(Color_Setter(colors,12,Status_Color_Str))
  { for(int c = 0; c < 12; c++)
    { if(Status_Colors[c] != colors[c]) // has this color changed
      { Status_Colors[c] = colors[c];
        // save the relevant change to config file
        set_config_int("status_display",Status_Color_Str[c],Status_Colors[c]);
      }
    }
  }
  return D_OK;
}

int Status_Checker()
{ int m = active_menu - Menu_Status;
  int lm  = Status_Disp_Mode + 1;  // last menu item - skipping the seperators
  if(lm > 3)
    lm++;
  if(m == lm)
    return D_OK;
  Menu_Status[lm].flags = 0;
  active_menu->flags |= D_SELECTED;
  m--;
  if(m >= 3)
    m--;
  Status_Disp_Mode =  m;
  return D_OK;
}

int Status_Mover()
{ active_menu->flags ^= D_SELECTED;
  int th = text_height(font);
  if(active_menu->flags & D_SELECTED)
    Status_Pos_Y = ViewsY1 - th - 2;
  else
    Status_Pos_Y = SCREEN_H - th;
  if(Status_Tidy() == D_OK)
    ReSizeViews();
  return D_REDRAW;
}

int Status_Hider()
{ active_menu->flags ^= D_SELECTED;
  if(Status_Disp)
    Status_Disp = NULL;
  else
    Status_Disp = Position_Status;
  return(Status_Tidy());
}

int Frame_Control()
{
  if(Current_Visual->BitmapBackground.Frames <= 1 ||
                !Current_Visual->AVI_Status.uSecPerFrame)
    return(D_OK);
  char fs[30], us[30];
  if(Current_Visual->Frame_Advance < 100)
  { usprintf(fs,"%lu",Current_Visual->Frame_Advance);
    us[0] = 0;
  }
  else
  { usprintf(us,"%lu",Current_Visual->Frame_Advance);
    fs[0] = 0;
  }
  float fps = 1000000.0/Current_Visual->AVI_Status.uSecPerFrame;
  char fst[80];
  usprintf(fst,"uSec/Frame:%10ul Frames/Sec %7.3f",
    Current_Visual->AVI_Status.uSecPerFrame,fps);

  DIALOG frame_dlg[] =
{
   /* (proc)            (x)  (y) (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                                     (dp2) (dp3) */
   { d_billwin_proc,    0,   0,  384, 104, 0,   0,   0,    0,      0,   0,   (void*)"Frame Advance for Current View", NULL, NULL },
   { d_billtext_proc,   8,   20, 144, 8,   0,   0,   0,    0,      0,   0,   (void*)fst,                              NULL, NULL },
   { d_billtext_proc,   8,   37, 224, 10,  0,   0,   0,    0,      0,   0,   (void*)"Advance by frames (0 to 99)",    NULL, NULL },
   { Xedit_proc,        274, 34, 80,  16,  0,   0,   0,    0,      9,   0,   fs,                                      NULL, NULL },
   { d_billtext_proc,   8,   60, 264, 8,   0,   0,   0,    0,      0,   0,   (void*)"Advance by MicroSeconds (100 up):", NULL, NULL },
   { Xedit_proc,        274, 57, 80,  16,  0,   0,   0,    0,      9,   0,   us,                                      NULL, NULL },
   { d_billbutton_proc, 135, 80, 48,  16,  0,   0,   13,   D_EXIT, 0,   0,   (void*)"OK",                             NULL, NULL },
   { d_billbutton_proc, 198, 81, 72,  16,  0,   0,   27,   D_EXIT, 0,   0,   (void*)"Cancel",                         NULL, NULL },
   { NULL,              0,   0,  0,   0,   0,   0,   0,    0,      0,   0,   NULL,                                    NULL, NULL }
};

  // Do the dialog
  set_dialog_color(frame_dlg, gui_fg_color, gui_bg_color);
  centre_dialog(frame_dlg);
  int res = popup_dialog(frame_dlg, -1);  // -1 = no focus

  // Read data back
  if(res == 6)
  { long fa = ustrtol(fs,NULL,0);
    long ua = ustrtol(us,NULL,0);
    if((!fa || fa == Current_Visual->Frame_Advance) && ua)
      Current_Visual->Frame_Advance = ua;
    else if(fa)
      Current_Visual->Frame_Advance = fa;
    else
      Current_Visual->Frame_Advance = 1;
    ReSizeViews();
  }
  return D_REDRAW;
}

MENU Menu_Status[] =
{
//  (text)                  (proc)          (child)     (flags)     (dp)
    { "3D ",                NULL,           NULL,       0,          NULL },
    { "&Absolute",          Status_Checker, NULL,       0,          NULL },
    { "&Relative",          Status_Checker, NULL,       0,          NULL },
    { "&Polar",             Status_Checker, NULL,       0,          NULL },
    { "2D (Projected)",     NULL,           NULL,       0,          NULL },
    { "&Absolute",          Status_Checker, NULL,       0,          NULL },
    { "&Relative",          Status_Checker, NULL,       0,          NULL },
    { "&Polar",             Status_Checker, NULL,       0,          NULL },
    { "",                   NULL,           NULL,       0,          NULL },
    { "At &Top",            Status_Mover,   NULL,       0,          NULL },
    { "&Hide",              Status_Hider,   NULL,       0,          NULL },
    { "&Colors",            Status_Color_Setter, NULL,  0,          NULL },
    { "Help",               Menu_Help_Proc, NULL,       0, (void*)"Status Display" },
    { 0 }
};

int Color_Sep_Checker();  // predefine

MENU Menu_Color_Sep[] =
{
//  (text)                  (proc)          (child)     (flags)     (dp)
    { "&Type",           Color_Sep_Checker, NULL,       D_SELECTED, NULL },
    { "&Layer",          Color_Sep_Checker, NULL,       0,          NULL },
    { "&Block",          Color_Sep_Checker, NULL,       0,          NULL },
    { 0 }
};

int Color_Seperation_Mode = 0;   // normal

int Color_Sep_Checker()
{ int m = active_menu - Menu_Color_Sep;
  int lm  = Color_Seperation_Mode;  // last menu item - skipping the seperators
  if(m == lm)
    return D_OK;
  Menu_Color_Sep[lm].flags = 0;  // untick
  active_menu->flags |= D_SELECTED; // tick
  Color_Seperation_Mode =  m;
  return D_REDRAW;
}

MENU Menu_View[] =
{
//  (text)          (proc)          (child)         (flags)     (dp)
    { "3D Split", Split3D,           NULL,          0,          NULL },
    { "Split &Horizontal", HSplitChecker, NULL,     0,          NULL },
    { "Split &Vertical",   VSplitChecker, NULL,     0,          NULL },
    { "Split Settings", Split_Sizer, NULL,          0,          NULL },
    { "Top    Left",  View_Editor,   NULL,          0,          NULL },
    { "Top    Right", View_Editor,   NULL,          D_DISABLED, NULL },
    { "Bottom Left",  View_Editor,   NULL,          D_DISABLED, NULL },
    { "Bottom Right", View_Editor,   NULL,          D_DISABLED, NULL },
    { "&Axis",      NULL,            Menu_Axis,     0,          NULL },
    { "&Grid",      NULL,            Menu_Grid,     0,          NULL },
    { "&Snap",      Checker,         NULL,          0,          NULL },
    { "Status Display", NULL,        Menu_Status,   0,          NULL },
    { "Frame Advance", Frame_Control, NULL,         0,          NULL },
    { "Render Options",NULL,         Menu_Render,   0,          NULL },  // see Render.cpp
    { "Color Seperation",NULL,       Menu_Color_Sep, 0,         NULL },
    { "Help&?",     Menu_Help_Proc,  NULL,          0, (void*)"View Menu" },
    { 0 }
};
/*  Actually located in view.h but copied here for reference to Menu_View changes
enum Menu_View_Children { SPLIT3D = 0, SPLIT_H, SPLIT_V, SPLIT_S, TL_VIEW, TR_VIEW, BL_VIEW, BR_VIEW,
                          AXIS_M, GRID_M, SNAP_O };
*/
int LinkChecker();

VisualClass* Menu_Visual = NULL;

int View_Setting_Load()
{
  if(!Menu_Visual)
    return(D_OK);  // nothing to save
  FILE *f;
  char path[MAXPATH*2];
  int i;

  ustrcpy(path, Save_Path);
  ustrcat(path,Menu_Visual->BitmapBackground.File_Name);
  char* tp = ustrchr(path,'.');
  if(tp)
    ustrcpy(tp,".vsf");
  if(billfile_select("Load View Settings File [.VSF]", path, "VSF"))
  { if(!ustrchr(path,'.'))
      ustrcat(path,".VSF");  // default save
    Menu_Visual->Load(path);
  }
  return D_EXIT;  // easier than trying to convince it to reset all the dialog strings
}

int View_Setting_Save()
{
  if(!Menu_Visual)
    return(D_OK);  // nothing to save
  FILE *f;
  char path[MAXPATH*2];
  int i;

  ustrcpy(path, Save_Path);
/*  ustrcat(path,Menu_Visual->BitmapBackground.File_Name);
  char* tp = ustrchr(path,'.');
  if(tp)
    ustrcpy(tp,".vsf"); */
  if(billfile_select("Save as View Settings File [.VSF]", path, "VSF"))
  { if(!ustrchr(path,'.'))
      ustrcat(path,".VSF");  // default save
    if(exists(path))
    { if(billalert(" WARNING", path, "File already exists", "&Ok", "&Cancel", RTN, ESC)!=1)
        return D_OK;
    }
    Menu_Visual->Save(path);
  }
  return D_OK;
}

int Call_View_Help()
{
  Get_Help("VIEW CONTROL");
  return D_OK;
}

char BitmapName[MAXPATH];
char BitmapStatusStr[33];
char Bxl[30],Byl[30],Bwl[30],Bhl[30],Bafl[30],Bref1l[30],Bref2l[30],Btfl[34];

void Set_Background_Strings()
{
  if(Menu_Visual->Background_Buffer)
    usprintf(BitmapStatusStr, "%8u x %8u",Menu_Visual->Background_Buffer->w,Menu_Visual->Background_Buffer->h);
  else
    usprintf(BitmapStatusStr,"No bitmap loaded");
  usprintf(Bxl, "%8.3f",fixtof(Menu_Visual->BitmapBackground.X));
  usprintf(Byl, "%8.3f",fixtof(Menu_Visual->BitmapBackground.Y));
  usprintf(Bwl, "%8.3f",fixtof(Menu_Visual->BitmapBackground.W));
  usprintf(Bhl, "%8.3f",fixtof(Menu_Visual->BitmapBackground.H));
  usprintf(Bafl, "%7u",Menu_Visual->BitmapBackground.Frame);
  usprintf(Bref1l,"%7u",Menu_Visual->BitmapBackground.Ref1);
  usprintf(Bref2l,"%7u",Menu_Visual->BitmapBackground.Ref2);
  usprintf(Btfl, "of%7u",Menu_Visual->BitmapBackground.Frames);
}

int Get_Background_Filename()
{
  if(billfile_select("Background[*.bmp/pcx/lbm/tga/avi]", BitmapName, "BMP;PCX;LBM;TGA;AVI"))
  { if(Menu_Visual->Load_Background(BitmapName))
      billalert(" Warning", "File could not be loaded", BitmapName,
                                   "Okay", NULL, 13, 27);
    else
      Set_Background_Strings();
    return D_REDRAW;
  }
  else
    return D_OK;
}

int Reset_Background_Pos()
{
  Menu_Visual->Reset_Bitmap_Pos();
  Set_Background_Strings();
  return D_REDRAW;
}

int Reset_Background_Size()
{
  Menu_Visual->Reset_Bitmap_Size();
  Set_Background_Strings();
  return D_REDRAW;
}

int Convert_Bg_Pal()
{
  // may in future specify how background colors are converted
  // eg. greyscale, redscale etc.
  // at the moment you'll just have to edit the background externally
  return D_OK;
}

int Set_Ref1()
{
  Menu_Visual->BitmapBackground.Ref1Camera = Menu_Visual->Camera;
  return D_OK;
}

int Set_Ref2()
{
  Menu_Visual->BitmapBackground.Ref2Camera = Menu_Visual->Camera;
  return D_OK;
}

int Disable_All_Joystick()
{
  // to ensure only one view has a joystick control
  for(int i = 0; i < 4; i++)
    Visual[i].Joystick_Enabled = false;
  Joystick_Init(2);  // up to two joysticks
  return D_OK;
}

#include "layset.h"  // for layer_editor
#include "blockset.h" // for block_editor

int View_Editor()
{
  char title[80] = "";
  int i;
  for(i = 0; i < 4; i++)
  { if(Menu_View[TL_VIEW+i].flags & D_DISABLED)
      continue;
    if(active_menu == &Menu_View[TL_VIEW+i])
      break;
  }
  if(i >= 4)
    return D_OK;
  if(i >= 2)
    ustrcpy(title,"BOTTOM ");
  else if(Menu_View[SPLIT_H].flags & D_SELECTED)
    ustrcpy(title,"TOP ");
  if(i & 1)
    ustrcat(title,"RIGHT ");
  else if(Menu_View[SPLIT_V].flags & D_SELECTED)
    ustrcat(title,"LEFT ");
  usprintf(title + ustrlen(title),"VIEW #%d MODE CONTROL",i+1);
  char sizestr[80];
  VisualClass* vis = Visual + i;
  Menu_Visual = vis;  // for the use of Save and Recall
  usprintf(sizestr,"X1:%d Y1:%d X2:%d Y2:%d",vis->X1,vis->Y1,vis->X2,vis->Y2);
  char sl[30], xl[30], yl[30], zl[30], rxl[30], ryl[30], rzl[30], fovl[30];
  if(vis->Scale > 1)
    usprintf(sl, "%d:1", vis->Scale);
  else if(vis->Scale < -1)
    usprintf(sl, "1:%d", -vis->Scale);
  else
    usprintf(sl, "1:1");
  usprintf(xl,  "%8.3f",fixtof(vis->Camera.Sc.X));
  usprintf(yl,  "%8.3f",fixtof(vis->Camera.Sc.Y));
  usprintf(zl,  "%8.3f",fixtof(vis->Camera.Sc.Z));
  usprintf(rxl, "%8.3f",fixtof(vis->Camera.Sc.Rx)*360.0/256.0);
  usprintf(ryl, "%8.3f",fixtof(vis->Camera.Sc.Ry)*360.0/256.0);
  usprintf(rzl, "%8.3f",fixtof(vis->Camera.Sc.Rz)*360.0/256.0);
  usprintf(fovl,"%8.3f",fixtof(vis->Camera.FOV));
  ustrncpy(BitmapName,vis->BitmapBackground.File_Name,sizeof(BitmapName));
  Set_Background_Strings();
  
  int res;

  DIALOG view_dlg[] =
{
   // (proc)            (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1)     (d2)          (dp)                             (dp2)                    (dp3)
   { d_billwin_proc,    0,   0,   440, 272, 0,   0,   0,    0,      0,       0,            (void*)title,                    NULL,                    NULL },
   { d_billbutton_proc, 8,   248, 48,  16,  0,   0,   13,   D_EXIT, 0,       0,            (void*)"OK",                     NULL,                    NULL },
   { d_billbutton_proc, 64,  248, 72,  16,  0,   0,   27,   D_EXIT, 0,       0,            (void*)"Cancel",                 NULL,                    NULL },
   { d_billtext_proc,   8,   20,  144, 8,   0,   0,   0,    0,      0,       0,            (void*)sizestr,                  NULL,                    NULL },
   { d_billradio_proc,  8,   32,  8,   8,   0,   0,   0,    0,      1,       0,            (void*)"No Link",                NULL,                    NULL },
   { d_billradio_proc,  108, 32,  8,   8,   0,   0,   0,    0,      1,       0,            (void*)"Link #1",                NULL,                    NULL },
   { d_billradio_proc,  208, 32,  8,   8,   0,   0,   0,    0,      1,       0,            (void*)"Link #2",                NULL,                    NULL },
   { d_billcheck_proc,  288, 36,  12,  12,  0,   0,   0,    0,      0,       0,            (void*)"&Link Background",       NULL,                    NULL },
   { d_billtext_proc,   8,   48,  100, 8,   0,   0,   0,    0,      0,       0,            (void*)"Camera",                 NULL,                    NULL },
   { d_billtext_proc,   8,   60,  100, 8,   0,   0,   0,    0,      0,       0,            (void*)"Position X:",            NULL,                    NULL },
   { Xedit_proc,        100, 56,  92,  16,  0,   0,   0,    0,      9,       0,            xl,                              NULL,                    NULL },
   { d_billtext_proc,   200, 60,  20,  8,   0,   0,   0,    0,      0,       0,            (void*)"Y:",                     NULL,                    NULL },
   { Xedit_proc,        224, 56,  92,  16,  0,   0,   0,    0,      9,       0,            yl,                              NULL,                    NULL },
   { d_billtext_proc,   320, 60,  20,  8,   0,   0,   0,    0,      0,       0,            (void*)"Z:",                     NULL,                    NULL },
   { Xedit_proc,        344, 56,  92,  16,  0,   0,   0,    0,      9,       0,            zl,                              NULL,                    NULL },
   { d_billtext_proc,   8,   80,  100, 8,   0,   0,   0,    0,      0,       0,            (void*)"Angle   RX:",            NULL,                    NULL },
   { Xedit_proc,        100, 76,  92,  16,  0,   0,   0,    0,      9,       0,            rxl,                             NULL,                    NULL },
   { d_billtext_proc,   200, 80,  30,  8,   0,   0,   0,    0,      0,       0,            (void*)"RY:",                    NULL,                    NULL },
   { Xedit_proc,        224, 76,  92,  16,  0,   0,   0,    0,      9,       0,            ryl,                             NULL,                    NULL },
   { d_billtext_proc,   320, 80,  30,  8,   0,   0,   0,    0,      0,       0,            (void*)"RZ:",                    NULL,                    NULL },
   { Xedit_proc,        344, 76,  92,  16,  0,   0,   0,    0,      9,       0,            rzl,                             NULL,                    NULL },
   { d_billtext_proc,   8,   102, 50,  8,   0,   0,   0,    0,      0,       0,            (void*)"Scale:",                 NULL,                    NULL },
   { Xedit_proc,        60,  96,  78,  16,  0,   0,   0,    0,      9,       0,            sl,                              NULL,                    NULL },
   { d_billcheck_proc,  140, 98,  12,  12,  0,   0,   'p',  0,      0,       0,            (void*)"&Perspective",           NULL,                    NULL },
   { d_billtext_proc,   320, 104, 30,  8,   0,   0,   0,    0,      0,       0,            (void*)"FOV:",                   NULL,                    NULL },
   { Xedit_proc,        352, 96,  80,  16,  0,   0,   0,    0,      9,       0,            fovl,                            NULL,                    NULL },
   { d_billtext_proc,   8,   120, 176, 8,   0,   0,   0,    0,      0,       0,            (void*)"Background Bitmap/AVI:", NULL,                    NULL },
   { Xedit_proc,        188, 120, 176, 16,  0,   0,   0,    0,      MAXPATH, 0,            BitmapName,                      NULL,                    NULL },
   { Xbutton_proc,      372, 120, 64,  16,  0,   0,   'b',  D_EXIT, 0,       0,            (void*)"&Browse",                Get_Background_Filename, NULL },
   { Xbutton_proc,      372, 152, 64,  16,  0,   0,   'p',  D_EXIT, 0,       0,            (void*)"&Palette",               Convert_Bg_Pal,          NULL },
   { d_billtext_proc,   8,   132, 72,  8,   0,   0,   0,    0,      0,       0,            (void*)BitmapStatusStr,          NULL,                    NULL },
   { d_billtext_proc,   8,   144, 112, 8,   0,   0,   0,    0,      0,       0,            (void*)"Projection",             NULL,                    NULL },
   { Xbutton_proc,      8,   152, 64,  16,  0,   0,   0,    D_EXIT, 0,       0,            (void*)"Position",               Reset_Background_Pos,    NULL },
   { d_billtext_proc,   80,  156, 16,  8,   0,   0,   0,    0,      0,       0,            (void*)"X:",                     NULL,                    NULL },
   { Xedit_proc,        100, 152, 80,  16,  0,   0,   0,    0,      9,       0,            Bxl,                             NULL,                    NULL },
   { d_billtext_proc,   184, 156, 20,  8,   0,   0,   0,    0,      0,       0,            (void*)"Y:",                     NULL,                    NULL },
   { Xedit_proc,        200, 152, 80,  16,  0,   0,   0,    0,      9,       0,            Byl,                             NULL,                    NULL },
   { Xbutton_proc,      8,   176, 64,  16,  0,   0,   0,    D_EXIT, 0,       0,            (void*)"Size",                   Reset_Background_Size,   NULL },
   { d_billtext_proc,   80,  180, 16,  8,   0,   0,   0,    0,      0,       0,            (void*)"W:",                     NULL,                    NULL },
   { Xedit_proc,        100, 176, 80,  16,  0,   0,   0,    0,      9,       0,            Bwl,                             NULL,                    NULL },
   { d_billtext_proc,   184, 180, 20,  8,   0,   0,   0,    0,      0,       0,            (void*)"H:",                     NULL,                    NULL },
   { Xedit_proc,        200, 176, 80,  16,  0,   0,   0,    0,      9,       0,            Bhl,                             NULL,                    NULL },
   { d_billcheck_proc,  288, 152, 14,  16,  0,   0,   'h',  0,      0,       0,            (void*)"&Hide",                  NULL,                    NULL },
   { d_billcheck_proc,  288, 176, 16,  16,  0,   0,   'f',  0,      0,       0,            (void*)"&Fixed Aspect",          NULL,                    NULL },
   { d_billtext_proc,   8,   204, 80,  8,   0,   0,   0,    0,      0,       0,            (void*)"Frame:",                 NULL,                    NULL },
   { Xedit_proc,        56,  200, 56,  16,  0,   0,   0,    0,      9,       0,            Bafl,                            NULL,                    NULL },
   { Xbutton_proc,      120, 200, 48,  16,  0,   0,   0,    D_EXIT, 0,       0,            (void*)"Ref#1:",                 Set_Ref1,                NULL },
   { Xedit_proc,        176, 200, 56,  16,  0,   0,   0,    0,      9,       0,            Bref1l,                          NULL,                    NULL },
   { Xbutton_proc,      240, 200, 48,  16,  0,   0,   0,    D_EXIT, 0,       0,            (void*)"Ref#2:",                 Set_Ref2,                NULL },
   { Xedit_proc,        296, 200, 56,  16,  0,   0,   0,    0,      9,       0,            Bref2l,                          NULL,                    NULL },
   { d_billtext_proc,   360, 204, 72,  8,   0,   0,   0,    0,      0,       0,            (void*)Btfl,                     NULL,                    NULL },
   { d_billtext_proc,   8,   228, 80,  12,  0,   0,   0,    0,      0,       0,            (void*)"Filters:",               NULL,                    NULL },
   { Xbutton_proc,      76,  224, 112, 16,  0,   0,   0,    D_EXIT, 0,       0,            (void*)"by Layer",               Layer_Editer,            NULL },
   { Xbutton_proc,      192, 224, 120, 16,  0,   0,   0,    0,      0,       0,            (void*)"by Block",               Block_Editer,            NULL },
   { Xbutton_proc,      316, 224, 120, 16,  0,   0,   0,    D_DISABLED, 0,   0,            (void*)"by Type",                NULL,                    NULL },
   { Xbutton_proc,      152, 248, 72,  16,  0,   0,   's',  D_EXIT, 0,       0,            (void*)"&Save",                  View_Setting_Save,       NULL },
   { Xbutton_proc,      232, 248, 80,  16,  0,   0,   'r',  D_EXIT, 0,       0,            (void*)"&Recall",                View_Setting_Load,       NULL },
   { Xbutton_proc,      384, 248, 48,  16,  0,   0,   '?',  D_EXIT, 0,       0,            (void*)"Help&?",                 Call_View_Help,          NULL },
   { d_billcheck_proc,  288, 20,  12,   12, 0,   0,   0,    0,      0,       0,            (void*)"&Joystick",              NULL,                    NULL },
   { d_billcheck_proc,  248, 98,  12,  12,  0,   0,   0,    0,      0,       0,            (void*)"Render",                 NULL,                    NULL },
   { NULL,              0,   0,   0,   0,   0,   0,   0,    0,      0,       0,            NULL,                            NULL,                    NULL }
};

    view_dlg[vis->Linked + 4].flags |= D_SELECTED;
    if(vis->Mode)
      view_dlg[23].flags = D_SELECTED;    // perspective
    if(!vis->Background_Visible)
      view_dlg[42].flags = D_SELECTED;    // hide
    if(vis->Background_Fixed_Aspect != 0)
      view_dlg[43].flags = D_SELECTED;    // fixed aspect
    if(vis->Link_Background)
      view_dlg[7].flags = D_SELECTED;   // link background
    if(vis->Joystick_Enabled)
      view_dlg[58].flags = D_SELECTED;
    else {
      for(int i = 0; i < 4; i++)
      {  if(Visual[i].Joystick_Enabled)
           view_dlg[58].flags = D_DISABLED;
      }
    }
    if(vis->Render)
      view_dlg[59].flags = D_SELECTED;

    // Do the dialog
    set_dialog_color(view_dlg, gui_fg_color, gui_bg_color);
    centre_dialog(view_dlg);
    res = popup_dialog(view_dlg, -1);  // -1 = no focus

    // Read data back
    if(res == 1)  // OK
    { vis->Linked = ((view_dlg[5].flags & D_SELECTED) != 0)
                + 2*((view_dlg[6].flags & D_SELECTED) != 0);
      if(view_dlg[58].flags & D_SELECTED)
        Disable_All_Joystick();         // and calibrate if necessary
      vis->Joystick_Enabled = ((view_dlg[58].flags & D_SELECTED)!=0);
      vis->Render = ((view_dlg[59].flags & D_SELECTED)!=0);
      vis->Link_Background = ((view_dlg[7].flags & D_SELECTED)!=0);
      vis->Scale = 0;
      char* fp = ustrchr(sl,':');
      vis->Scale = ustrtol(sl,NULL,0);
      if(vis->Scale <= 1)   // first number was zero, so find number after ':'
        vis->Scale = -ustrtol(fp+1,NULL,0);

      vis->Camera.Sc.X  = ftofix(ustrtod(xl,NULL));
      vis->Camera.Sc.Y  = ftofix(ustrtod(yl,NULL));
      vis->Camera.Sc.Z  = ftofix(ustrtod(zl,NULL));
      vis->Camera.Sc.Rx = ftofix(ustrtod(rxl,NULL)*256.0/360.0);
      vis->Camera.Sc.Ry = ftofix(ustrtod(ryl,NULL)*256.0/360.0);
      vis->Camera.Sc.Rz = ftofix(ustrtod(rzl,NULL)*256.0/360.0);
      vis->Camera.FOV = ftofix(ustrtod(fovl,NULL));
      vis->BitmapBackground.X = ftofix(ustrtod(Bxl,NULL));
      vis->BitmapBackground.Y = ftofix(ustrtod(Byl,NULL));
      vis->BitmapBackground.W = ftofix(ustrtod(Bwl,NULL));
      vis->BitmapBackground.H = ftofix(ustrtod(Bhl,NULL));
      vis->BitmapBackground.Frame = ustrtol(Bafl,NULL,10);
      vis->BitmapBackground.Ref1  = ustrtol(Bref1l,NULL,10);
      vis->BitmapBackground.Ref2  = ustrtol(Bref2l,NULL,10);
      vis->Mode = (view_dlg[23].flags == D_SELECTED);  // select dodgy perspective
      vis->Background_Visible = (view_dlg[42].flags != D_SELECTED);
      if((view_dlg[43].flags == D_SELECTED) && (vis->BitmapBackground.W != 0))
        vis->Background_Fixed_Aspect = fdiv(vis->BitmapBackground.H,vis->BitmapBackground.W);
      else
        vis->Background_Fixed_Aspect = 0;
    }
    if(res == 1 || res == 56)  // OK or RECALL
    { bool newbitmap = false;
      if(strcmp(vis->BitmapBackground.File_Name,BitmapName))  // background has changed
        newbitmap = true;
      if(res == 1)
      {  strncpy(vis->BitmapBackground.File_Name,BitmapName,
           sizeof(vis->BitmapBackground.File_Name));
      }
//      if(newbitmap)
    }
  return D_REDRAW;
}

#include "blocks.h"

void Draw_Views(BITMAP* bmp)
{
  VisualClass* focus = Current_Visual;
  int c;
  for(int i = 0; i < 5; i++)
  { VisualClass* vis = Visual + i;
    int v = i;
    if(i == 4) // do Current Visual last so that all the X,Y,W,Hs are correct
    { vis = focus;
      v = c;
    }
    else if(focus == vis)
    { c = i;
      continue;  // do this last
    }
    if(!(Menu_View[TL_VIEW+v].flags & D_DISABLED))
    { Current_Visual = vis;  // for the benefit of Update()
      CurrentRealToFlatConversion++; // found in Vertex.cpp
      vis->Draw(bmp,(Current_Visual == focus),&DrawingPlane); // draw the frame and background
      // only need to update target list if not the same as last draw
      // and only then if object limits are used for Indicate - they're not anymore
      // however, needs updating after a mod - so here for the moment
      TargetList->Update();   // update the visual link parameters for this visual
      DrawVisu(bmp,Base_Block);   // see findvisu.cpp - does the actual drawing of objects
      TargetList->Indicate_Selected(bmp); // uses visual link parameters
    }
  }
  Current_Visual = focus;
  set_clip(bmp,0,0,bmp->w-1,bmp->h-1);
  Last_Disp_Mode = -1;  // force refresh of cursor display
  RestoreDepth();  // ensures that FlatToReal will work
}

void Refresh_Views(BITMAP* bmp, BITMAP* screen_bmp)
{
  VisualClass* focus = Current_Visual;
  int c;
  for(int i = 0; i < 5; i++)
  { VisualClass* vis = Visual + i;
    int v = i;
    if(i == 4) // do Current Visual last so that all the X,Y,W,Hs are correct
    { vis = focus;
      v = c;
    }
    else if(focus == vis)
    { c = i;
      continue;  // do this last
    }
    if(!(Menu_View[TL_VIEW+v].flags & D_DISABLED) && vis->Refresh)
    { rectfill(bmp,vis->X1,vis->Y1,vis->X2,vis->Y2,makecol(127,127,127));
      Current_Visual = vis;  // for the benefit of Update()
      CurrentRealToFlatConversion++; // found in Vertex.cpp
      vis->Draw(bmp,(Current_Visual == focus),&DrawingPlane); // draw the frame and background
      TargetList->Update();   // update the visual link parameters for this visual
      DrawVisu(bmp,Base_Block);   // see findvisu.cpp - does the actual drawing of objects
      TargetList->Indicate_Selected(bmp); // uses visual link parameters
      blit(bmp,screen_bmp,vis->X1,vis->Y1,vis->X1,vis->Y1,vis->X2-vis->X1,vis->Y2-vis->Y1);
    }
  }
  Current_Visual = focus;
  set_clip(bmp,0,0,bmp->w-1,bmp->h-1);
  Last_Disp_Mode = -1;  // force refresh of cursor display
}

int Find_View(int x,int y)
{
  for(int i = 0; i < 4; i++)
  { if(!(Menu_View[TL_VIEW+i].flags & D_DISABLED))
    { if(x > Visual[i].X1 && x < Visual[i].X2 && y > Visual[i].Y1 && y < Visual[i].Y2)
        return(i);
    }
  }
  return(-1);
}

int Set_View(int x,int y, BITMAP* bmp)
{
   int v = Find_View(x,y);
   if(v >= 0)  // valid view found
   { VisualClass* vis = Visual + v;
     if(vis != Current_Visual) {
       Set_Current_Visual(vis);
       return 1;
     }
     return 0;
   }
   return -1;
}

void Set_DrawingPlane(int x, int y)
{
  // Get the current point and use it to set depth in other views
//  VTX vtx;
  Current_Visual->FlatToReal(&DrawingPlane,SnapX(x),SnapY(y));
  for(int i = 0; i < 4; i++)
  { if(!(Menu_View[TL_VIEW+i].flags & D_DISABLED) && (&Visual[i] != Current_Visual))
      Visual[i].SetDepth(&DrawingPlane);
  }
}

void RestoreDepth()
{
  for(int i = 0; i < 4; i++)
  { if(!(Menu_View[TL_VIEW+i].flags & D_DISABLED))
      Visual[i].SetDepth(&DrawingPlane);
  }
}

void Set_Current_Visual(VisualClass* vis, BITMAP* bmp)
{ if(bmp)
    Current_Visual->Rect(bmp,BLACK);
  Current_Visual = vis;
  if(bmp)
    Current_Visual->Rect(bmp,WHITE);
  // set the X,Y,W,H of target list for this view
  if(TargetList)
    TargetList->Update();
  // grid offsets need to be updated for new object drag if snap set
  Current_Visual->Set_Origin();  // sp that Xg and Yg are up to date
  Grid_Offset_X = &(Current_Visual->Xg);
  Grid_Offset_Y = &(Current_Visual->Yg);
  if(TargetList)
    TargetList->Set_Clip(Current_Visual->X1,Current_Visual->Y1,
                          Current_Visual->X2,Current_Visual->Y2);
}

void Update_Views()
{ if(Current_Visual->Linked)
  { // for views that are also linked, match to current visual
    for(int i = 0; i < 4; i++)
    { if((Visual+i != Current_Visual) && Visual[i].Linked == Current_Visual->Linked)
        Visual[i].Set(Current_Visual);
    }
  }
}

bool Animate_Views()    // called by MSG_IDLE in edit3d.cpp
{
  return(Current_Visual->Animate());  // at the moment only animate current visual
         // && Current_Visual->Linked); - it would be nice if unlinked views could
         // be redrawn independantly
}

#include "space.h"
VTX* StatusVTX = NULL;
VTX* StatusVTX1 = NULL;
VTX* LastStatusVTX = NULL;
char* StatusMsg = NULL;
char* LastStatusMsg = NULL;

void Position_Status(BITMAP* bmp, int x, int y, int xo, int yo)
{
  float dx = (float)(x-xo);
  float dy = (float)(y-yo);
  VTX pos, old;
  float fx, fy, fz;
  if(!StatusVTX)
    Current_Visual->FlatToReal(&pos,x,y);
  else
    pos = *StatusVTX;
  fx = fixtof(pos.X);
  fy = fixtof(pos.Y);
  fz = fixtof(pos.Z);
  float ox, oy, oz;
  if(!StatusVTX1)
    Current_Visual->FlatToReal(&old,xo,yo);
  else
    old = *StatusVTX1;
  ox = fixtof(old.X);
  oy = fixtof(old.Y);
  oz = fixtof(old.Z);
  float dist, angle;
  int c = Status_Colors[Status_Disp_Mode*2];
  text_mode(Status_Colors[Status_Disp_Mode*2+1]);
  float ax,ay;
  VTX origin = { 0, 0, 0};
  if(Status_Disp_Mode != Last_Disp_Mode || Status_Pos_Y != Last_Disp_Y ||
     StatusVTX != LastStatusVTX || StatusMsg != LastStatusMsg)
  { Last_Disp_Mode = Status_Disp_Mode;
     // restore blank display
    // clear old status using background color for status mode
    int cx;
    if(LastStatusMsg)
      cx = 0;
    else
      cx = Status_Pos_X[0];
    LastStatusVTX = StatusVTX;
    LastStatusMsg = StatusMsg;
    if(Last_Disp_Y > 0)
      rectfill(bmp,cx,Last_Disp_Y,SCREEN_W,Last_Disp_Y+text_height(font),
        Status_Colors[Status_Disp_Mode*2+1]);
    Last_Disp_Y = Status_Pos_Y;
    if(!StatusVTX && !StatusMsg) {
      textout(bmp,font,"S:",Status_Pos_X[0],Status_Pos_Y,Axis_Colors[0]);
      textprintf(bmp,font,Status_Pos_X[1],Status_Pos_Y,c,"%10.3f",
                                           fixtof(Current_Visual->Camera.Sc.Scale));
    }
    // possibly load statusmsg with name of baseblock
    if(StatusMsg) {
      textout(bmp,font,StatusMsg,0,Status_Pos_Y,Axis_Colors[0]);
    }
    switch(Status_Disp_Mode)
    { case 0:
        textout(bmp,font,"X:",Status_Pos_X[2],Status_Pos_Y,Axis_Colors[0]);
        textout(bmp,font,"Y:",Status_Pos_X[4],Status_Pos_Y,Axis_Colors[1]);
        textout(bmp,font,"Z:",Status_Pos_X[6],Status_Pos_Y,Axis_Colors[2]);
        break;
      case 1:
        textout(bmp,font,"dX:",Status_Pos_X[2],Status_Pos_Y,Axis_Colors[0]);
        textout(bmp,font,"dY:",Status_Pos_X[4],Status_Pos_Y,Axis_Colors[1]);
        textout(bmp,font,"dZ:",Status_Pos_X[6],Status_Pos_Y,Axis_Colors[2]);
        break;
    }
  }
  switch(Status_Disp_Mode)
  { case 0:  // absolute
      textprintf(bmp,font,Status_Pos_X[3],Status_Pos_Y,c,"%+10.3f",fx);
      textprintf(bmp,font,Status_Pos_X[5],Status_Pos_Y,c,"%+10.3f",fy);
      textprintf(bmp,font,Status_Pos_X[7],Status_Pos_Y,c,"%+10.3f",fz);
      break;
    case 1:  // relative
      textprintf(bmp,font,Status_Pos_X[3],Status_Pos_Y,c,"%+10.3f",fx - ox);
      textprintf(bmp,font,Status_Pos_X[5],Status_Pos_Y,c,"%+10.3f",fy - oy);
      textprintf(bmp,font,Status_Pos_X[7],Status_Pos_Y,c,"%+10.3f",fz - oz);
      break;
    case 2:  // relative polar - well distance only
      dist = vector_length_f(fx - ox, fy - oy, fz - oz);
      textprintf(bmp,font,Status_Pos_X[2],Status_Pos_Y,c,
                      "dist:%+10.3f", dist);
      break;
    case 3: // absolute projected
      int ox,oy,oz;
      // find origin projected onto this view
      Current_Visual->RealToFlat(&origin,ox,oy,oz);
      ax = x - ox;
      ay = y - oy;
      textprintf(bmp,font,Status_Pos_X[2],Status_Pos_Y,c,
                      "H:%+10.3f V:%+10.3f",
                      ax*fixtof(Current_Visual->Camera.Sc.Scale),
                      ay*fixtof(Current_Visual->Camera.Sc.Scale));
      break;
    case 4: // relative projected
      textprintf(bmp,font,Status_Pos_X[2],Status_Pos_Y,c,
                      "dH:%+10.3f dV:%+10.3f",
                      dx*fixtof(Current_Visual->Camera.Sc.Scale),
                      dy*fixtof(Current_Visual->Camera.Sc.Scale));
      break;
    case 5: // polar projected
      dist = vector_length_f(dx,dy,0)*fixtof(Current_Visual->Camera.Sc.Scale);
      angle = atan2(dx,dy)/PI*360.0;
      if(dist > 10)  // just for diagnotics
        dist = 10;
      textprintf(bmp,font,Status_Pos_X[2],Status_Pos_Y,c,
                      "dist:%+10.3f angle:%7.3", dist, angle);
      break;
  }
}

int Previous_Frame()
{ if(Current_Visual->Previous_Frame())
    return D_OK;
  else
  { if(Current_Visual->Linked && Current_Visual->Link_Background)
    { // for views that are also linked, match to current visual
      for(int i = 0; i < 4; i++)
      { if((Visual+i != Current_Visual) &&
           (Visual[i].Linked == Current_Visual->Linked) &&
            Visual[i].Link_Background)
        Visual[i].Previous_Frame();
      }
    }
    return D_REDRAW;
  }
}

int Next_Frame()
{ if(Current_Visual->Next_Frame())
    return D_OK;
  else
  { if(Current_Visual->Linked && Current_Visual->Link_Background)
    { // for views that are also linked, match to current visual
      for(int i = 0; i < 4; i++)
      { if((Visual+i != Current_Visual) &&
           (Visual[i].Linked == Current_Visual->Linked) &&
            Visual[i].Link_Background)
        Visual[i].Next_Frame();
      }
    }
    return D_REDRAW;
  }
}

// first used for logpos.cpp
int Get_Real(int x, int y, VTX* pos)
{
  int i = Find_View(x,y);
  if(i >= 0)
    Visual[i].FlatToReal(pos,x,y);
  return i;
}

void Get_Frame_Number(int* fc)    // fc must be an array of at least four
{
  for(int v = 0; v < 4; v++)
  { if(Visual[v].BitmapBackground.Frames < 1)
      fc[v] = -1;
    else
      fc[v] = Visual[v].BitmapBackground.Frame;
  }
}


