/*  Source file for the Edit3d program by Robert Parker using the Allegro
    and Bgui2 - see credits else where - also see BasicGUI source code.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "a3dload.h"

// see a3d_fmt.txt for details of the *.A3D format usedbloew

PACKFILE* pf = NULL;
unsigned Version = THIS_VERSION;
#define MAXSTRING 256
unsigned long UnUsedBytes = 0;

int RByte(BYTE* b)
{ return(pack_fread(b,1,pf) != 1);
}

int RUnsigned(unsigned* u)
{ *u = 0;
  return(pack_fread(u,sizeof(unsigned),pf) != sizeof(unsigned));
}

int RInt(int* i)
{ *i = 0;
  return(pack_fread(i,sizeof(int),pf) != sizeof(int));
}

int RFixed(fixed* fx)
{ *fx = 0;
  return(pack_fread(fx,sizeof(fixed),pf) != sizeof(fixed));
}

int RString(char* str, int maxstring = MAXSTRING)
{ int size;
  if(pack_fread(&size,sizeof(int),pf) != sizeof(int))
    return(2);
  char* buf = new char[size+2];
  if(!buf)
    return(3);
  buf[size] = 0;  // not sure how unicode is termiated
  buf[size+1] = 0;  // this should do it okay
  if(pack_fread(buf,size,pf) != size) {
    delete buf;
    return(1);
  }
  char* s = uconvert(buf,U_ASCII,str,U_CURRENT,maxstring);
  if(s != str)
    ustrncpy(str,buf,maxstring);
  delete buf;
  return(0);
}

#include "vertex.h"
#include "line.h"
#include "arc.h"
#include "spline.h"
#include "face.h"
#include "insert.h"
#include "findvisu.h"
#include "vislst3d.h"

BYTE SizeOfA3DObject[OBJECT_TYPES] = {
  3*sizeof(fixed),           // VERTEX
  (4 + 1 + 4)*sizeof(int),   // FACE
  4 + 6*sizeof(fixed),       // INSERT
  2*sizeof(int) + 2,         // LINE
  2*sizeof(int) + 2 + 2*sizeof(int) + sizeof(unsigned) + 6*sizeof(fixed) }; // SPLINE
  

int RColor(int* color)
{
  BYTE red,green,blue;
  if(RByte(&red) || RByte(&green) || RByte(&blue))
    return(1);
  *color = makecol(red,green,blue);
  return(0);
}

int LineBaseLoad(LineClass* line, BlockClass* block, LayerClass* layer, int dp) {
  unsigned p1, p2, l1, pl1;
  if(RUnsigned(&p1) || RUnsigned(&p2))
    return(1);
  p1 += dp;  // add offset to existing vertex
  p2 += dp;
  // even if they are -1, no point worrying about it - not fatal
  line->P1 = FindMatchingVertex(p1,block,layer);
  line->P2 = FindMatchingVertex(p2,block,layer);
  if(RByte(&line->Type) || RByte(&line->Width))
    return(3);
  if(RUnsigned(&l1) || RUnsigned(&pl1))
    return(4);
  if(l1 != -1) {  // look for joining line
    line->L1 = FindMatchingLine(l1,block,layer);
    if(line->L1)
      (line->L1)->L2 = line;
    // else no worry
  }
  if(pl1 != -1) { // look for parrallel line or another joining line
    line->PL1 = FindMatchingLine(pl1,block,layer);
    if(line->PL1)
      (line->PL1)->PL2 = line;
    // else - no worry
  }
  return(0);
}

#define MAX_LAYERS 64    // really just the limit for each added file/library

int Load_A3D()
{
  char namestr[MAXSTRING];  // temporary storage of incoming names
  // system is accumlative, so record existing in each layer of null block
  unsigned existing_layers = Layers;
  unsigned existing_vertex[MAX_LAYERS]; // could have used "new" but then messy
  existing_vertex[0] = CountMatchingObject(&VertexControl,NULL,NULL);
  LinkClass* layerlink = LayerList.FirstLink();
  int l = 1;
  while(layerlink && l < Layers+1 && l < MAX_LAYERS)
  { LayerClass* layer = (LayerClass*)(layerlink->Object);
    existing_vertex[l] = CountMatchingObject(&VertexControl,NULL,layer);
    layerlink = layerlink->NextLink();
    l++;
  }

  if(RUnsigned(&Version))
    return(1);
  if(RUnsigned(&Textures))
    return(2);
  for(int l = 0; l < Textures; l++)
  { TextureClass* texture = new TextureClass();
    TextureList.Add(texture);
    if(RString(namestr))
      return(3);
    texture->Name = new char[ustrsizez(namestr)];
    ustrcpy(texture->Name,namestr);
  }
  unsigned layers;
  if(RUnsigned(&layers))
    return(4);
  unsigned layer_ref[MAX_LAYERS];  // could have used 'new' but too messy
  if(!layer_ref)
    return(4);
  for(int l = 0; l < layers; l++)
  { if(RString(namestr))
      return(5);
    if(l >= MAX_LAYERS)
      continue;
    // look to see if there is a match with an existing layer
    layerlink = LayerList.FirstLink();
    int ol = 0;
    LayerClass* layer;
    while(layerlink)
    { layer = (LayerClass*)(layerlink->Object);
      if(stricmp(layer->Name,namestr) == 0)
        break;
      layerlink = layerlink->NextLink();
      ol++;
    }
    if(layerlink) {   // new layer
      layer = new LayerClass();
      LayerList.Add(layer);
      LayerList.Add(layer);
      layer->Name = new char[ustrsizez(namestr)];
      ustrcpy(layer->Name,namestr);
      layer_ref[l] = Layers - 1;
    }
    else
      layer_ref[l] = ol;   // reference to an existing layer
    if(RColor(&layer->Color))
      return(6);
    BYTE lt = 0;
    if(RByte(&lt))
      return(7);
    layer->LineType = (LINETYPE)lt;
  }
  unsigned blocks = 0;
  if(RUnsigned(&blocks))
    return(8);
  // create all blocks so they can be referred to
  for(int b = 0; b < blocks; b++)
    BlockList.Add(new BlockClass());

  LinkClass* blocklink = BlockList.FirstLink();
  while(TRUE)
  { BlockClass* block = NULL;  // the very last block is the base block = NULL
    if(blocklink)
    { block = (BlockClass*)(blocklink->Object);
      if(RString(namestr))
        return(9);
      block->Name = new char[ustrsizez(namestr)];
      ustrcpy(block->Name,namestr);
      if(RColor(&block->Color))
        return(10);
      if(RFixed(&block->Base.X) || RFixed(&block->Base.Y) || RFixed(&block->Base.Z))
        return(11);
    }
    unsigned layr = 0;
    do
    { LayerClass* layer = NULL;  // the very last layer is the base layer = NULL
      if(layr < Layers &&       // not the base layer, so find it
         layr < MAX_LAYERS)      // for simplicity
      { LinkClass* layerlink = LayerList.Find(layer_ref[layr]);
        if(layerlink)
          layer = (LayerClass*)(layerlink->Object);
      }
      int dp = 0;
      if(block == NULL &&
        ((layer == NULL) || (layer_ref[layr] < existing_layers)) ) {
        if(layer == NULL)
          dp = existing_vertex[0];
        else
          dp = existing_vertex[layer_ref[layr]];
      }
      while(TRUE)
      { BYTE t;
        if(RByte(&t))
          return(12);
        if(t == 0)
          break;
        t--;   // was base 1, not base 0
        BYTE sizeoftype;
        if(RByte(&sizeoftype))
          return(13);
        unsigned number;
        if(RUnsigned(&number))
          return(14);
        unsigned expected_size;
        if(t < OBJECT_TYPES)
          expected_size = SizeOfA3DObject[t];
        else
          expected_size = 0;   // unknown type
        OBJECT_CONTROL_TYPES type = (OBJECT_CONTROL_TYPES)t;
        for(unsigned i = 0; i < number; i++)
        {
          unsigned p1, p2, p3, p4;
          int err;
          VertexClass* v;
          LineClass* line;
          ArcClass* arc;
          SplineClass* spline;
          unsigned flags;
          FaceClass* face;
          unsigned texturenum;
          InsertClass* insert;
          unsigned blocknum;
          LinkClass* blocklink;
          switch(type)
          { case VERTEX_CONTROL:  //  -- must come before Line, Face or Insert etc.
              // should really check, for Base level that there is no matching vertex
              // can always do this as a filter later I suppose
              v = new VertexClass();
              if(!v)
                break;
              Target3DList->Add(v,&VertexControl);
              if(RFixed(&v->Pos.X))
                return(20);
              if(RFixed(&v->Pos.Y))
                return(21);
              if(RFixed(&v->Pos.Z))
                return(22);
              v->Layer = layer;
              v->Block = block;
              break;
            case FACE_CONTROL:
              face = new FaceClass();
              Target3DList->Add(face,&FaceControl);
              if(RUnsigned(&p1) || RUnsigned(&p2) || RUnsigned(&p3) || RUnsigned(&p4))
                return(31);
              p1 += dp; // add offset to existing vertex
              p2 += dp;
              p3 += dp;
              p4 += dp;
              face->P1 = FindMatchingVertex(p1,block,layer);
              face->P2 = FindMatchingVertex(p2,block,layer);
              face->P3 = FindMatchingVertex(p3,block,layer);
              if(p4 == p3)
                face->P4 = NULL;
              else
                face->P4 = FindMatchingVertex(p4,block,layer);
              if(RUnsigned(&texturenum))
                return(32);
              if(texturenum != 0xFFFF)
              { LinkClass* l =TextureList.Find(texturenum - 1);
                if(l)
                  face->Texture = (TextureClass*)(l->Object);
              }
              if(RColor(&face->C1))
                return(33);
              if(RColor(&face->C2))
                return(34);
              if(RColor(&face->C3))
                return(35);
              if(RColor(&face->C4))
                return(36);
              if(RUnsigned(&face->Flags))
                return(37);
              break;
            case INSERT_CONTROL:
              insert = new InsertClass();
              Target3DList->Add(insert,&InsertControl);
              if(RUnsigned(&blocknum))
                return(40);
              blocklink = BlockList.Find(blocknum);
              if(blocklink)
                insert->InsertBlock = (BlockClass*)(blocklink->Object);
              if(RUnsigned(&p1))
                return(41);
              p1 += dp; // add offset to existing vertex
              insert->P1 = FindMatchingVertex(p1,block,layer);
              if(RUnsigned(&p2))
                return(42);
              insert->P2 = FindMatchingVertex(p2,block,layer);
              if(RFixed(&insert->Rx) || RFixed(&insert->Ry) || RFixed(&insert->Rz))
                return(43);
              if(RByte(&insert->RotOrder))
                return(44);
              if(RFixed(&insert->Sx) || RFixed(&insert->Sy) || RFixed(&insert->Sz))
                return(45);
              break;
            case LINE_CONTROL:
              line = new LineClass();
              err = LineBaseLoad(line,block,layer,dp);
              if(err)
                return err+50;
              Target3DList->Add(line,&LineControl);
              break;
            case ARC_CONTROL:
              arc = new ArcClass();
              err = LineBaseLoad((LineClass*) arc,block,layer,dp);
              if(err)
                return err+60;
              Target3DList->Add(arc,&ArcControl);
              if(RUnsigned(&p3))
                return(67);
              arc->P3 = FindMatchingVertex(p3,block,layer);
              if(RUnsigned(&p4))
                return(68);
              arc->P4 = FindMatchingVertex(p4,block,layer);
              if(RByte(&arc->Flags))
                return(69);
              break;
            case SPLINE_CONTROL:
              spline = new SplineClass();
              err = LineBaseLoad((LineClass*)spline,block,layer,dp);
              if(err)
                return err+70;
              Target3DList->Add(spline,&SplineControl);
              if(RUnsigned(&flags))
                return 78;
              flags |= SPLINE_V1_CALC | SPLINE_V2_CALC;
              if(!(flags & SPLINE_V1_LOCK))
                flags ^= SPLINE_V1_CALC;  // let it know it needs to be calc'd
              if(!(flags & SPLINE_V2_LOCK))
                flags ^= SPLINE_V2_CALC;  // let it know it needs to be calc'd
              spline->Flags = flags;
              if(RFixed(&spline->V1.X) ||
                 RFixed(&spline->V1.Y) ||
                 RFixed(&spline->V1.Z) ||
                 RFixed(&spline->V2.X) ||
                 RFixed(&spline->V2.Y) ||
                 RFixed(&spline->V2.Z))
                 return(79);
              break;
          }
          // use up any left over bytes resulting from future versions
          for(int s = expected_size; s < sizeoftype; s++) {
            BYTE t;
            RByte(&t);
            UnUsedBytes++;
          }
        }
      }
      layr++;
    } while(layr < Layers);
    if(!block)
      break;
    blocklink = blocklink->NextLink();
  }
  return(0);
}

int LoadA3D(char* path)
{
  char* errorstr = "ALLEGRO PACKED 3D LOAD ERROR";
  pf = pack_fopen(path,"rp");
  if(!pf)
  { billalert(errorstr,path,"Couldn't open file for reading","&Ok",NULL,'o',0);
    return(1);
  }
  UnUsedBytes = 0;
  int errorcode = Load_A3D();
  if(errorcode)
  { char errortype[32];
    sprintf(errortype,"Code:%3d",errorcode);
    billalert(errorstr,errortype,"in Load_A3D","&Ok",NULL,'o',0);
    pack_fclose(pf);
    return(1);
  }
  pack_fclose(pf);
  if(UnUsedBytes > 0)
  { char msgstr[80];
    sprintf(msgstr,"    %2d    %2d     %10lu.",
      THIS_VERSION, Version, UnUsedBytes);
    billalert("FILE LOAD WARNING","This Version    File Version   Bytes Unused",
                   msgstr,"&Ok",NULL,'o',0);
  }
  return(0);
}

