/*  Source file for the BasicGUI program by Robert Parker using the Allegro 
    and Bgui2 - see credits else where.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include "visulist.h"

LinkClass* VisualLinkMaker(ObjectClass* object)
{ return((LinkClass*)new VisualLinkClass(object));
}

VisualLinkClass::VisualLinkClass(ObjectClass* object) : LinkClass(object)
{
  Control = NULL;   // must be set after creation
  X = Y = W = H = 0;
  PosLocked = false;
  SizeLocked = false;
  Visible = TRUE;
}

VisualLinkClass::~VisualLinkClass()
{ if(Link_Notify)
    Link_Notify(this);   // "I'm leaving; don't call me here again!"
/*  - this is done in LinkClass
  if(Previous)
    Previous->Next = Next;
  if(Next)
    Next->Previous = Previous;
*/
  if(Object->References)
    Object->References--;  // unlike LinkClass, take responibility for removing it
  if(!Object->References) {
    Control->Remove(Object);
    Object = NULL;  // let's LinkClass no not to do anything more
  }
}

// virtual functions
bool VisualLinkClass::Inside(int x, int y)
{ return(Visible && x >= X && x < X + W && y >= Y && y < Y + H &&
                    Control->Inside(Object,x,y));
}

bool VisualLinkClass::Include(int x1, int y1, int x2, int y2)
{ return(Visible &&
         x1 <= X && x2 >= X+W-1 &&
         y1 <= Y && y2 >= Y+H-1 &&
         Control->Include(Object,x1,y1,x2,y2));
}

void VisualLinkClass::Indicate(BITMAP* bmp, int fg, int bg)  // e.g. to show selection
{
  Control->Indicate(bmp, Object, X, Y, X+W, Y+H, fg, bg);
}

/* xor-draw rectangles around selected objects */
void VisualLinkClass::Xor_Indicate()
{
  int i;
  scare_mouse();
  xor_mode(TRUE);
  Indicate(screen);
  xor_mode(FALSE);
  unscare_mouse();
}

/* resize a single object by dragging the mouse */
int VisualLinkClass::Drag_Resize(int ox, int oy, int mode)
{
    int x, y, dx, dy, sx, sy, back_grid = Grid;


    // Resize the object
    sx = sy = 0;
    if(ox <= X+3)
      sx = -1;
    else if(ox >= X+W-3)
      sx = 1;

    if(oy <= Y+3)
      sy = -1;
    else if(oy >= Y+H-3)
      sy = 1;

    if((!(mode & 1)) ^ ((key_shifts & KB_ALT_FLAG) != 0))
      Grid = 0;   // turns off any snap

    x = SnapX(ox);
    y = SnapY(oy);

    Xor_Indicate();
    while(mouse_b)
    {
      Grid = ((!(mode & 1)) ^ ((key_shifts & KB_ALT_FLAG) != 0)) ? 0 : back_grid;

      if(SnapX(mouse_x) != x || SnapY(mouse_y) != y)
      {
        x = SnapX( mouse_x );
        y = SnapY( mouse_y );

        if(sx<0)
          dx = X - SnapX(X + x-ox);
        else if(sx>0)
          dx = SnapX(X+W + x-ox) - (X+W);
        else
          dx = 0;

        if(sy<0)
          dy = Y - SnapY(Y + y-oy);
        else if(sy>0)
          dy = SnapY(Y+H + y-oy) - (Y+H);
        else
          dy = 0;

        Xor_Indicate();

        if(W+dx < 8)
          dx = 0;
        if(H+dy < 8)
          dy = 0;
        if(sx<0)
          X -= dx;
        if(sy<0)
          Y -= dy;
        W += dx;
        H += dy;

        Xor_Indicate();

        ox = x;
        oy = y;
      }
      broadcast_dialog_message(MSG_IDLE, 0);
    }
    Xor_Indicate();
    Grid = back_grid;
    Control->RetainZ = true;
    Control->SetLimits(Object,X,Y,W,H);
    Control->RetainZ = false;
    return D_REDRAW;
}

void VisualLinkClass::Update()
{
  Control->GetLimits(Object,X,Y,W,H);  // used to have Visible =
  // but this conflicted with it's use in edit.cpp
  LinkClass::Update();
}

// default draw mainly for debugging
void VisualLinkClass::Draw(BITMAP* bmp)
{
  if(Visible)
    Control->Draw(bmp,Object);
}

VisualListClass::VisualListClass() : ListClass()
{
  LinkMaker = VisualLinkMaker;
  ObjectSnap = FALSE;
  FindOnly = NULL;
}

#define BEGIN_LIST VisualLinkClass* l = (VisualLinkClass*)FirstLink(); while(l && l != Undone) {
#define END_LIST   l = (VisualLinkClass*)l->NextLink(); }

VisualLinkClass* VisualListClass::Add(ObjectClass* object, ObjectControlClass* control,
                                                   LinkClass* p)
{
  VisualLinkClass* vl = (VisualLinkClass*)LinkMaker(object);
  if(vl) {
    vl->Control = control;
    Insert(vl,p);
  }
  return(vl);
}

/* CreateLink solves the problem created by Visual3DListClass */
VisualLinkClass* VisualListClass::CreateLink(ObjectClass* obj, ObjectControlClass* control)
{
  VisualLinkClass* vl = (VisualLinkClass*)LinkMaker(obj);
  if(vl) {
    vl->Control = control;
    Insert(vl);
  }
  return vl;
}


VisualLinkClass* VisualListClass::Add(ObjectControlClass* control,
                                           int x, int y, int w, int h)
{
  ObjectClass* obj = control->Create();
  if(!obj)
    return(NULL);
  VisualLinkClass* vl = CreateLink(obj,control);
  if(!vl)
    return(NULL);
  vl->X = x;
  if(w < 0) {
    vl->X += w;
    vl->W = -w;
  } else {
    vl->W = w;
  }
  vl->Y = y;
  if(h < 0) {
    vl->Y += h;
    vl->H = -h;
  } else {
    vl->H = h;
  }
  control->SetLimits(obj,x,y,w,h);
  return(vl);
}

VisualLinkClass* VisualListClass::Copy(VisualLinkClass* link, LinkClass* p,
                     bool copy_object)
{ // unlike base class, this function does create a seperate object
  // referred object has an owner other than the clipboard then clone it
  ObjectClass* obj = link->Object;
  if(copy_object && link->Object && link->Object->References > 1) {
    obj = link->Control->Create();
    if(obj) {
      memcpy(obj,link->Object,link->Control->Size());
    }
  }
  VisualLinkClass* l = Add(obj,link->Control,p);
  // now copy the factors
  if(l)
  { l->X = link->X;
    l->Y = link->Y;
    l->W = link->W;
    l->H = link->H;
    l->Visible = link->Visible;
  }
  return(l);
}

/* find the object under the mouse */
VisualLinkClass* VisualListClass::Find_Mouse_Object()
{
  int mx, my;
  mx = mouse_x;
  my = mouse_y;
  if(!ObjectSnap)
    return(Find_Object(mx,my));
  else
  { int closest_distance = MAXINT;
    VisualLinkClass* closest = NULL;
    int cx1 = ClipX1;
    int cx2 = ClipX2;
    int cy1 = ClipY1;
    int cy2 = ClipY2;
    int tx = SCREEN_W/80;
    int ty = SCREEN_H/80;
    if(Grid)  // if a Grid is selected, then limit to +/- a grid spacing
    { tx = Grid_W[Grid];
      ty = Grid_H[Grid];
    }
    if(cx1 < mx - tx)
      cx1 = mx - tx;
    if(cx2 > mx + tx)
      cx2 = mx + tx;
    if(cy1 < my - ty)
      cy1 = my - ty;
    if(cy2 > my + ty)
      cy2 = my + ty;
    // FindOnly is a bit redundant as Inside should call Current_Visual->Include(type)
    BEGIN_LIST
      if((!FindOnly || l->Control == FindOnly) &&
         l->Include(cx1,cy1,cx2,cy2)) // check it's within reason
      { int d = l->Control->Distance(l->Object,mx,my);
        if(d < closest_distance)
        { closest_distance = d;
          closest = l;
        }
      }
    END_LIST
    return(closest);
  }
}

VisualLinkClass* VisualListClass::Find_Object(int x, int y)
{ // FindOnly is a bit redundant as Inside should call Current_Visual->Include(type)
  BEGIN_LIST
    if((!FindOnly || l->Control == FindOnly) &&
      l->Inside(x,y))
      return(l);
  END_LIST
  return NULL;
}

void VisualListClass::Set_Clip(int x1, int y1, int x2, int y2)
{ ClipX1 = x1;
  ClipY1 = y1;
  ClipX2 = x2;
  ClipY2 = y2;
}

/* let the user drag a selection rectangle with the mouse and select/unselect
 * objects accordingly (CTRL to add objects to an existing selection, SHIFT
 * to remove them).
 */
int VisualListClass::Drag_Select(void)
{
  int x1, y1, x2, y2, i;
  x1 = mouse_x;
  y1 = mouse_y;
  Drag_Rectangle(&x1, &y1, &x2, &y2, 0);

  // need shift or control key to add to selection
  if(!(key_shifts & KB_CTRL_FLAG || key_shifts & KB_SHIFT_FLAG))
    Select_None();
    
  BEGIN_LIST
    if(l->Include(x1,y1,x2,y2))
    { l->Control->RetainZ = true;  // use existing data in z plane
      l->Selected = !(key_shifts & KB_SHIFT_FLAG);  // shift to remove
      l->Control->RetainZ = false;
    }
  END_LIST

  return D_REDRAW;
}




/* xor-draw rectangles around selected objects */
void VisualListClass::Xor_Indicate_Selected(void)
{
  int i;
  scare_mouse();
  xor_mode(TRUE);
  Indicate_Selected(screen);
  xor_mode(FALSE);
  unscare_mouse();
}



/* move selected objects by draging the mouse. td is the object that was
 * clicked, and thus will be snaped to the grid, the others keeping their
 * position relative to it.
 */
int VisualListClass::Drag_Move(VisualLinkClass* t,int ox, int oy, int mode)
{
  int x, y, i, dx, dy, back_grid = Grid;

  if((!(mode & 1)) ^ ((key_shifts & KB_ALT_FLAG) != 0))
    Grid = 0;   // turns off any snap
    
  x = SnapY(ox);
  y = SnapY(oy);

  // Move selection rectangles
  set_mouse_sprite(NULL);  // standard arrow
  scare_mouse();
  xor_mode(TRUE);
  hline(screen, ClipX1, oy, ClipX2, XOR_COLOR);
  vline(screen, ox, ClipY1,ClipY2, XOR_COLOR);
  xor_mode(FALSE);
  Xor_Indicate_Selected();
  while(mouse_b)
  {
    Grid = ((!(mode & 1)) ^ ((key_shifts & KB_ALT_FLAG) != 0)) ? 0 : back_grid;

    if(SnapX(mouse_x) != x || SnapY(mouse_y) != y)
    {
      x = SnapX(mouse_x);
      y = SnapY(mouse_y);
      dx = SnapX(t->X + x-ox) - t->X;
      dy = SnapY(t->Y + y-oy) - t->Y;

      Xor_Indicate_Selected();
      xor_mode(TRUE);
      hline(screen, ClipX1, oy, ClipX2, XOR_COLOR);
      vline(screen, ox, ClipY1, ClipY2, XOR_COLOR);
      xor_mode(FALSE);
      BEGIN_LIST
        if(l->Selected)
        {
          l->X += dx;
          l->Y += dy;
        }
      END_LIST
      Xor_Indicate_Selected();
      xor_mode(TRUE);
      hline(screen, ClipX1, y, ClipX2, XOR_COLOR);
      vline(screen, x, ClipY1, ClipY2, XOR_COLOR);
      xor_mode(FALSE);

      ox = x;
      oy = y;
    }
    broadcast_dialog_message(MSG_IDLE, 0);
  }
  xor_mode(TRUE);
  hline(screen, ClipX1, oy, ClipX2, XOR_COLOR);
  vline(screen, ox, ClipY1, ClipY2, XOR_COLOR);
  xor_mode(FALSE);

  Xor_Indicate_Selected();
  Grid = back_grid;
  BEGIN_LIST
   if(l->Selected)
   { l->Control->RetainZ = true;  // use existing data in z plane
     l->Control->SetLimits(l->Object,l->X,l->Y,l->W,l->H);
     l->Control->RetainZ = false;
   }
  END_LIST
  return D_REDRAW;
}



/* snap the objects of the selection to the grid */
int VisualListClass::Snap_Selected(void)
{
  BEGIN_LIST
    if(l->Selected)
    { l->X = SnapX(l->X);
      l->Y = SnapY(l->Y);
      l->W = SnapX(l->W);
      l->H = SnapY(l->H);
    }
  END_LIST
  return D_REDRAW;
}

void VisualListClass::Indicate_Selected(BITMAP* bmp, int fg, int bg)
{
  BEGIN_LIST
    if(l->Selected)
      l->Indicate(bmp, fg, bg);
  END_LIST
}

void VisualListClass::Draw_All(BITMAP* bmp)
{
  BEGIN_LIST
    l->Draw(bmp);
  END_LIST
}

long VisualListClass::Load(FILE* f)
{
  long size = 0;
  int type;
  while(!feof(f))
  { if(fread(&type,sizeof(int),1,f) != 1)
      return(0);
    // use type to choose ObjectControlClass; e.g. 0 = Dummy
    ObjectControlClass* control = ObjectControlClasses[type];
    ObjectClass* obj = control->Create();
    fread(obj,control->Size(),1,f);
    Add(obj,control);
    size += control->Size() + sizeof(int);
  }
  return(size);
}

void VisualListClass::Save(PACKFILE* pf)
{
  BEGIN_LIST
    pack_fwrite(&l->Control->Type,sizeof(int),pf);
    pack_fwrite(l->Object,l->Control->Size(),pf);
  END_LIST
}

int VisualListClass::Find_OfType(ObjectClass* object,
                             ObjectControlClass* control1,
                             ObjectControlClass* control2,
                             ObjectControlClass* control3)
{
  int i = 0;
  BEGIN_LIST
    if(l->Object == object)
    { if(l->Control == control1 ||
         l->Control == control2 ||
         l->Control == control3)
        return(i);
      else
        return(-2);
    }
    i++;
  END_LIST
  return(-1);
}

ObjectClass* VisualListClass::Find_OfType(int o,
                           ObjectControlClass* control1,
                           ObjectControlClass* control2,
                           ObjectControlClass* control3)
{
  int i = 0;
  BEGIN_LIST
    if(i == o)
    { if(l->Control == control1 ||
         l->Control == control2 ||
         l->Control == control3)
        return(l->Object);
      else
        return(NULL);
    }
    i++;
  END_LIST
  return(NULL);
}

bool VisualListClass::DependantOn(ObjectClass* object)
{
  int i = 0;
  BEGIN_LIST
    if(l->Control->DependantOn(l->Object,object))
      return i;
    i++;
  END_LIST
  return -1;
}

void VisualListClass::Delete_Selected()
{ // same as LinkListClass version except it checks for dependants
  // and it calls the correct destructor
  BEGIN_LIST
    VisualLinkClass* nl =(VisualLinkClass*)l->NextLink(); // shouldn't really use it after it's deleted so save next link
    if(l->Selected && !DependantOn(l->Object)) {
      Destroy(l);
      delete l;
    }
    l = nl;
  }
}

/*
void VisualListClass::Clear()
{
  BEGIN_LIST
    l->Delete()
      l->Indicate(bmp, fg, bg);
  END_LIST
}
*/

VisualListClass* TargetList;   // current edit target - set by user
VisualLinkClass* target;   // used by macros
unsigned Targets;

