/*  Source file for the Edit3d program by Robert Parker using the Allegro
    and Bgui2 - see credits else where - also see BasicGUI source code.
    Copyright (C) 2001-2004  Robert Parker

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* CAD system by Robert Parker
 * based on basicgui
 *
 * edit3d.c : The core functions of the editor
 *
 */

#include "edit3d.h"
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include "obj3ctrl.h"
#include "edit.h"
#include "layset.h"
#include "blockset.h"
#include "textuset.h"
#include "open3d.h"
#include "view.h"
#include "vislst3d.h"
#include <math.h>
#include "logpos.h"
#include "joy.h"    // for joystick calibrate in options menu

#include "line.h"
#include "arc.h"
#include "spline.h"
#include "face.h"
#include "facedlg.h"
#include "solid.h"
#include "insert.h"

void Demo3D(BITMAP*); // remove when 3d fixed

int RunDemo()
{
  Demo3D(ScreenBuffer);
  return D_REDRAW;
}

VisualList3DClass TemporaryList;     // clipboard for cut and paste

//PROC *proc_list = NULL;
//PROC *new_object = NULL;
DIALOG_PLAYER *Editor_Player = NULL;
DATAFILE *FontFile = NULL;

/* gui settings of the edited dialog ( != those of the editor ) */
FONT* Back_font;
int Back_baseline, Back_mouse_focus;

// mouse display
int XRef, YRef, Xold, Yold;
VisualLinkClass* Mouse_Target = NULL;
VisualLinkClass* Something_Selected = NULL;
VisualLinkClass* Something_Found = NULL;

void Init_Editor();
void Deinit_Editor();
void Set_Editor_Gui_State();
void Set_Back_Gui_State();

/* menu callbacks */
int About();
int Action_Checker();
int Auto_Vertex_Checker();
int Snap_Vertex_Checker();
int Find_All_Checker();
bool FindAll = false; // determines which vertex are checked
int Font_Checker();
int Snaper();
int Gui_Setter();
int Gui_Color_Setter();
int Color_Selected();
int Source_Viewer();
int Shift_Up();
int Shift_Down();

/* dialog procs */
int Workspace_proc(int, DIALOG*, int);
int Drag_New_Object(void);

/* strings functions */
char *Get_Copy(char*);

int Screen_Change()
{
  Screen_Mode->ChangeMode(Screen_Mode->W,Screen_Mode->H,Screen_Mode->BPP,TRUE);
  bill_init_radio_buttons();
  return D_REDRAW;
}

/******************************************/
/********** The menus & dialogs ***********/
/******************************************/

// Checker is defined in View.cpp

MENU Menu_Selection[] =
{
//  (text)                  (proc)              (child)     (flags)     (dp)
    { "Select &None",       Select_None,        NULL,       0,          NULL },
    { "Select &All\tCtrl-A",Select_All,         NULL,       0,          NULL },
    { "&Invert\tCtrl-I",    Invert_Selection,   NULL,       0,          NULL },
    { "",                   NULL,               NULL,       0,          NULL },
    { "C&ut\tCtrl-T",       Cutter,             NULL,       0,          NULL },
    { "C&opy\tCtrl-C",      Copier,             NULL,       0,          NULL },
    { "Pa&ste\tCtrl-P",     Paster,             NULL,       0,          NULL },
    { "",                   NULL,               NULL,       0,          NULL },
    { "&Edit List",         Selection_Editer,   NULL,       0,          NULL },
    { "Set &Colors",        Color_Selected,     NULL,       0,          NULL },
    { "&Properties",        Properties,         NULL,       0,          NULL },
    { "&Snap Now!",         Snaper,             NULL,       0,          NULL },
    { "&Layers",            Layer_Editer,       NULL,       0,          NULL },
    { "&Blocks",            Block_Editer,       NULL,       0,          NULL },
    { "&Textures",          Texture_Editer,     NULL,       0,          NULL },
    { "Help&?",             Menu_Help_Proc,     NULL,       0,  (void*)"SELECTION MENU" },
    { 0 }
};

MENU Menu_Action[] =
{
//  (text)        (proc)              (child)     (flags)     (dp)
    { "&Add",        0,                  NULL,       0,          NULL },
    { "&Zoom",       Action_Checker,     NULL,       0,          NULL },
    { "&Edit",       Action_Checker,     NULL,       0,          NULL },
    { "&Paste",      Action_Checker,     NULL,       D_DISABLED, NULL },
    { "&Background", Action_Checker,     NULL,       D_DISABLED, NULL },
    { "",            NULL,               NULL,       0,          NULL },
    { "Auto&Vertex", Auto_Vertex_Checker,NULL,       D_SELECTED, NULL },
    { "&SnapVertex", Snap_Vertex_Checker,NULL,       0,          NULL },
    { "&FindAll",    Find_All_Checker,   NULL,       0,          NULL },
    { "Help&?",      Menu_Help_Proc,     NULL,       0, (void*)"ACTION MENU" },
    { 0 }
};

#define MENU_ACTION_ADD        (Menu_Action[0].flags & D_SELECTED)
#define MENU_ACTION_ZOOM       (Menu_Action[1].flags & D_SELECTED)
#define MENU_ACTION_EDIT       (Menu_Action[2].flags & D_SELECTED)
#define MENU_ACTION_PASTE      (Menu_Action[3].flags & D_SELECTED)
#define MENU_ACTION_BACKGROUND (Menu_Action[4].flags & D_SELECTED)
#define MENU_ACTION_AUTOVERTEX (Menu_Action[6].flags & D_SELECTED)


MENU Menu_Options[] =
{
//  (text)                 (proc)         (child)  (flags)  (dp)
    { "&Screen Mode",      Screen_Change, NULL,    0,       NULL },
    { "Gui &Settings",     Gui_Setter,    NULL,    0,       NULL },
    { "&Ega Colors",       Ega_Color_Setter, NULL, 0,       NULL },
    { "Use &Default Font", Font_Checker,  NULL,    0,       NULL },
    { "Calibrate Joystick",Joystick_Calibrate, NULL, 0,     NULL },
    { "Arc Options",       NULL,          NULL,    D_DISABLED, NULL },
    { "Spline Options",    NULL,          NULL,    D_DISABLED, NULL },
    { "",                  NULL,          NULL,    0,       NULL },
//    { "&View Source",      source_viewer, NULL,    0,       NULL },
//    { "",                  NULL,          NULL,    0,       NULL },
    { "Help&?",            Menu_Help_Proc, NULL,   0, (void*)"OPTIONS MENU"},
    { 0 }
};

MENU Menu_Help[] =
{
//  (text)          (proc)          (child)         (flags)     (dp)
    { "&Help",       Menu_Help_Proc,     NULL,       0, (void*)"HELP CONTENTS" },
    { "&About...",        About,         NULL,    0,       NULL },
    { "&Credits",    Menu_Help_Proc,     NULL,       0, (void*)"CREDITS" },
    { 0 }
};

MENU Menu[] =
{
//  (text)          (proc)          (child)         (flags)     (dp)
    { "&File",      NULL,           Menu_File,      0,          NULL },   // see open3d.cpp
    { "&Selection", NULL,           Menu_Selection, 0,          NULL },
    { "&View",      NULL,           Menu_View,      0,          NULL },  // see view.cpp
    { "&Action",    NULL,           Menu_Action,    0,          NULL },
    { "&Options",   NULL,           Menu_Options,   0,          NULL },
    { "Demo",       RunDemo,        NULL,           0,          NULL },
    { "Help&?",     NULL,           Menu_Help,      0,          NULL },
    { 0 }
};

MENU Menu_right_click_norm[] =
{
//  (text)              (proc)          (child)     (flags)     (dp)
    { "&Add",           NULL,           NULL,       0,          NULL },
    { "",               NULL,           NULL,       0,          NULL },
    { "C&ut",           Cutter,         NULL,       0,          NULL },  // edit.cpp
    { "C&opy",          Copier,         NULL,       0,          NULL },
    { "Pa&ste",         Paster,         NULL,       0,          NULL },
    { "",               NULL,           NULL,       0,          NULL },
    { "&Properties",    Properties,     NULL,       0,          NULL },
    { 0 }
};

MENU Menu_right_click_edit[] =
{
//  (text)              (proc)          (child)     (flags)     (dp)
    { "C&ut",           Cutter,         NULL,       0,          NULL },
    { "C&opy",          Copier,         NULL,       0,          NULL },
    { "Pa&ste",         Paster,         NULL,       0,          NULL },
    { "",               NULL,           NULL,       0,          NULL },
    { "Change &Faces",  Faces_Editor,   NULL,       0,          NULL },  // facedlg.cpp
    { "&Properties",    Properties,     NULL,       0,          NULL },
    { 0 }
};

MENU* Menu_right_click = Menu_right_click_norm;

DIALOG editor[] =
{
   /* (proc)          (x) (y) (w) (h) (fg) (bg) (key)   (flags) (d1)      (d2) (dp)              (dp2) (dp3) */
   { Workspace_proc,  0,  0,  0,  0,  0,   0,   0,      0,      0,        0,   NULL,             NULL, NULL },
   { d_menu_proc,     0,  0,  0,  0,  0,   0,   0,      0,      0,        0,   Menu,             NULL, NULL },
   { d_icon_proc,     0,  16, 16, 16, 0,   0,   0,      D_HIDDEN, 0,      0,   0,                NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('q'), 0,      0,        0,   Closer,           NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('a'), 0,      0,        0,   Select_All,       NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('i'), 0,      0,        0,   Invert_Selection, NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('t'), 0,      0,        0,   Cutter,           NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('c'), 0,      0,        0,   Copier,           NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('v'), 0,      0,        0,   Paster,           NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('n'), 0,      0,        0,   Newer,            NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('o'), 0,      0,        0,   Opener,           NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   C('s'), 0,      0,        0,   Saver,            NULL, NULL },
//   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   '<',    0,      0,        0,   Previous_Frame,   NULL, NULL },
//   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   '>',    0,      0,        0,   Next_Frame,       NULL, NULL },
// see MSG_XCHAR below
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   0,      0,    KEY_ESC,    0,   Closer,         NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   0,      0,    KEY_INSERT, 0,   Inserter,         NULL, NULL },
   { d_keyboard_proc, 0,  0,  0,  0,  0,   0,   0,      0,      KEY_DEL,  0,   Deleter,          NULL, NULL },
   { NULL,            0,  0,  0,  0,  0,   0,   0,      0,      0,        0,   NULL,             NULL, NULL }
};

/******************************************/
/********* Some dialog procedures *********/
/******************************************/

/* workspace_proc */
int Workspace_proc(int msg, DIALOG *d, int c)
{
  int ret = D_O_K;
  int ox, oy;
  int x1, x2, y1, y2; // common usage
  long xl1,xl2,yl1,yl2;
//  BITMAP *back_screen;  // see remarks in case REDRAW
  VisualLinkClass* t;
  int new_view;
  int k;
  switch(msg)
  {
    case MSG_START:
      // update dialog for screen size
      d->w = SCREEN_W;   //Screen_Mode->W;
      d->h = SCREEN_H;   //Screen_Mode->H;
      d->dp = (void*)ScreenBuffer;
      // get the mouse to use correct colours
      set_mouse_sprite(NULL);
      break;

    case MSG_END:
      // Screen_Mode removes BackBuffer
      break;

    case MSG_CLICK:

      ox = mouse_x;
      oy = mouse_y;
      // highlight the current visual. Set_View is in View.cpp
      new_view = Set_View(ox,oy,ScreenBuffer);  // this will update Target3DList screen parameters

      // which mouse actions require update of views
      if(mouse_b & 1 ||                                          // left button
         ((MENU_ACTION_ZOOM ||               // or Zoom & right button
           MENU_ACTION_BACKGROUND) && mouse_b & 2)) // or Background & right button
      { int mchoice = 0;
        if(ox > Visual[0].X2 && ox < Visual[1].X1)  // between a horizontal split
          mchoice |= 1;
        if(oy > Visual[0].Y2 && oy < Visual[2].Y1)  // between a vertical split
          mchoice |= 2;
        if(mchoice)             // between visual splits
        { x1 = Visual[0].X1;
          y1 = Visual[0].Y1;
          Drag_Rectangle(&x1,&y1,&x2,&y2,0);    // see DragRect.cpp
          if(x2 != ox && y2 != oy)
          { if(mchoice & 1)
              VSplit = x2;
            if(mchoice & 2)
              HSplit = y2;
            ReSizeViews();    // View.cpp
            ret |= D_REDRAW;
          }
        }
        else if(MENU_ACTION_ZOOM)  // Zoom action mode
        { int m = mouse_b;
          int dm;
          if((m & 3) == 1)   // going to zoom in
          { // Current_Visual defined in Visual.cpp
            int w = Current_Visual->X2 - Current_Visual->X1;
            int h = Current_Visual->Y2 - Current_Visual->Y1;
            Current_Visual->ScaleIn(w,h);
            x1 = mouse_x - w/2;
            x2 = x1 + w;
            y1 = mouse_y - h/2;
            y2 = y1 + h;
            dm = 2 | 4;
          }
          else              // zooming out
          { x1 = Current_Visual->X1, y1 = Current_Visual->Y1;
            x2 = Current_Visual->X2, y2 = Current_Visual->Y2;
            dm = 2 | 4 | 8;
          }
          if(!Drag_Rectangle(&x1,&y1,&x2,&y2,dm))  // returns xy as 1=original 2=new
          { if(m & 2)
              Current_Visual->Out(x2,y2);
            else
              Current_Visual->In(x2,y2);
          }
          else
          { if(m & 2)
              Current_Visual->Pan(x2-x1,y2-y1);
            else
              Current_Visual->In(x2,y2);
          }
          RestoreDepth();  // camera position to drawing plane depth
          Update_Views();
          ret |= D_REDRAW;
        }
        else if(new_view)  // first press of all following actions only changes reference
        { // if view is orthagonal then save reference position in view
          CrossHair_Limits(x1,y1,x2,y2);  // view.cpp
          Drag_CrossHairs(ox,oy, &x1, &y1, &x2, &y2,            // see DrawRect.cpp
           ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
          Set_DrawingPlane(x2,y2);   // see view.cpp
          ret |= D_REDRAW;
        }
        else if(MENU_ACTION_ADD &&  // Add action mode
                 New_Object) {      // a New_Object has been created - see Object.Cpp
          if((New_Object == &InsertControl) && (Action_Block == NULL)) {
            billalert("WARNING","First select block","Goto Selection Menu","&Ok", NULL, 'o', 0);
          } else {
            ret |= Drag_New_Object();       // create a new object
          }
        }
        else if(MENU_ACTION_PASTE)  // paste mode
          ret |= Mouse_Paste();
        else if(Current_Visual->Background_Buffer &&  // visual has a background
             (MENU_ACTION_BACKGROUND)) // Background mode so background resizer
        { Current_Visual->Get_Background_Corners(xl1,yl1,xl2,yl2);
          int m = mouse_b;
          int dm;
          int t; // temporary
          // check if outside limits
          x1 = IntLimit(xl1);  // see visual.cpp
          y1 = IntLimit(yl1);
          x2 = IntLimit(xl2);
          y2 = IntLimit(yl2);
          /*
          if(xl1 < Current_Visual->X1)
            x1 = Current_Visual->X1;
          else if(xl1 > Current_Visual->X2)
            x1 = mouse_x;
          else
            x1 = xl1;
          if(xl2 > Current_Visual->X2)
            x2 = Current_Visual->X2;
          else if(xl2 < Current_Visual->X1)
            x2 = mouse_x;
          else
            x2 = xl2;
          if(yl1 < Current_Visual->Y1)
            y1 = Current_Visual->Y1;
          else if(yl1 > Current_Visual->Y2)
            y1 = mouse_y;
          else
            y1 = yl1;
          if(yl2 > Current_Visual->Y2)
            y2 = Current_Visual->Y2;
          else if(yl2 < Current_Visual->Y1)
            y2 = mouse_y;
          else
            y2 = yl2;
          */
          if((m & 3) == 1)   // resize bitmap
          { if(mouse_x < (xl1+xl2)/2)
            { t = x1;
              x1 = x2;
              x2 = t;
            }
            if(mouse_y < (yl1+yl2)/2)
            { t = y1;
              y1 = y2;
              y2 = t;
            }
            dm = 0;  // parameters are sorted so that x2 and y2 are > x1 and y1
          }
          else
            dm = 2 | 8;  // drag whole size plus sorting plus show mouse
          if(Drag_Rectangle(&x1,&y1,&x2,&y2,dm))  // returns xy as 1=original 2=new
          {
            // only change the corners that were within reason
            if(xl1 > MININT)
              xl1 = x1;
            if(xl2 < MAXINT)
              xl2 = x2;
            if(yl1 > MININT)
              yl1 = y1;
            if(yl2 < MAXINT)
              yl2 = y2;
            Current_Visual->Set_Background_Corners(xl1,yl1,xl2,yl2);
          }
          Update_Views();
          ret |= D_REDRAW;
        }
        // default MENU_ACTION_EDIT mode
        else if( (t = Target3DList->Find_Mouse_Object()) && !(key_shifts & KB_ALT_FLAG))
        { // object under mouse and Alt key not pressed
          // Target3DList is defined in Visulist.cpp
          // spline editing is special as vectors are independant of vertex (maybe shouldn't be)
          if(Something_Found->Control == &SplineControl) {
            int x = mouse_x;
            int y = mouse_y;
            int x1,y1,x2,y2;
            CrossHair_Limits(x1,y1,x2,y2);
            Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
            SplineControl.MoveVector(Something_Found->Object,x2,y2);
          }
        //  if(!t->Selected)   // not selected
        //    Target3DList->SelectOnly(t);
          // If the user release the mouse button without moving, just select the object
          while(!(ox-mouse_x || oy-mouse_y))
          { if(!mouse_b)     // mouse released without movement
            { if(!(key_shifts & KB_CTRL_FLAG) && !(key_shifts & KB_SHIFT_FLAG))
                Target3DList->SelectOnly(t);
              else           // add or (with shift) subtract from selection
                t->Selected = !(key_shifts & KB_SHIFT_FLAG);
              return D_REDRAW;
            }
          }

          // decide if this is a move or a resize
// Clip should already have been set
//          Target3DList->Set_Clip(Current_Visual->X1,Current_Visual->Y1,
//                               Current_Visual->X2,Current_Visual->Y2);

          if( t->SizeLocked ||
              ( mouse_x >= t->X+3 && mouse_x < t->X + t->W - 3 &&
                mouse_y >= t->Y+3 && mouse_y < t->Y + t->H - 3 ) )
          { if( !t->PosLocked)
              ret |= Target3DList->Drag_Move(t,ox,oy,
                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0));
          }
          else
            ret |= t->Drag_Resize(ox,oy,
                  ((Menu_View[SNAP_O].flags & D_SELECTED) != 0));
        }
        else  // no object under mouse or Alt key pressed
        { while(!(ox-mouse_x || oy-mouse_y))  // no mouse move
          { if(!mouse_b)              // mouse released with no movement
            { Target3DList->Select_None();   // so remove all selection
              return D_REDRAW;
            }
          }
          if(!(key_shifts & KB_CTRL_FLAG) && !(key_shifts & KB_SHIFT_FLAG))
          {    // find out if at least one target was already selected
            BEGIN_3D_LIST
              if(target->Selected)
                return(Target3DList->Drag_Move(target,ox,oy,
                                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0)));
              // if that's so, move selected targets
            END_3D_LIST
          }
          // if shift or cntrl key pressed then add or (with shift subtract) from selection
          ret |= Target3DList->Drag_Select();
        }
      }
      else if(mouse_b & 2)
      {
        t = Target3DList->Find_Mouse_Object();
    /*    if(New_Object)
        {
          New_Object = NULL;
          while(mouse_b)
            broadcast_dialog_message(MSG_IDLE, 0);
        }
        else
        {  */
          set_mouse_sprite(NULL);
          if(t)
          {
            if(!t->Selected)
              Target3DList->SelectOnly(t);
          }
          do_billmenu(Menu_right_click, mouse_x, mouse_y);
          ret |= D_REDRAW;
     //   }
      }
      break;

    case MSG_DCLICK:

      set_mouse_sprite(NULL);
      while(mouse_b)
        broadcast_dialog_message(MSG_IDLE, 0);
      ret |= Properties();
      break;

    case MSG_LOSTMOUSE:

      if(mouse_sprite != _mouse_pointer)
        set_mouse_sprite(_mouse_pointer);
      break;

    case MSG_DRAW:
      // update menu states
      if((Current_Visual->Background_Buffer != NULL) ^
         (MENU_ACTION_BACKGROUND == 0))
        Menu_Action[4].flags ^= D_DISABLED;

      // update dialog just incase they've changed
      d->w = SCREEN_W;  // Screen_Mode->W;
      d->h = SCREEN_H;  // Screen_Mode->H;
      d->dp = (void*)ScreenBuffer;
      
      clear_to_color(ScreenBuffer, makecol(127, 127, 127));

      Target3DList->Update();  // needed to get all those X,Y,Z,W,H,D's sorted


//    Slight of hand in original dlg program to force drawing to bitmap
//    nolonger needed for my purposes but I'll keep it here as a reminder
//      back_screen = screen;
//      screen = ScreenBuffer;  // just in case some targets draw to screen
//      Set_Back_Gui_State();
      Draw_Views(ScreenBuffer);
//      Set_Editor_Gui_State();
//      screen = back_screen;
      scare_mouse();
      blit(ScreenBuffer, screen, 0, 0, 0, 0, ScreenBuffer->w, ScreenBuffer->h);
      unscare_mouse();
      break;
    case MSG_XCHAR:
      k = c & 0xFF;
      if(k >= 'A' && k <= 'Z')
      { LogMousePos(k - 'A');
        return D_USED_CHAR;
      }
      else if( k >= 'a' && k <= 'z')
      { LogMousePos(k - 'a');
        return D_USED_CHAR;
      }
      switch(k)
      {
//        case 27:  // ESC
//         return(Closer());
        case '<':
        case ',':
          LogSave();
          Previous_Frame();
          return D_USED_CHAR | D_REDRAW;
        case '>':
        case '.':
          LogSave();
          Next_Frame();
          return D_USED_CHAR | D_REDRAW;
      }
      break;
    case MSG_IDLE:
      if(Animate_Views())  // change detected in some time process in a visual
      { scare_mouse();
        Refresh_Views(ScreenBuffer,screen);   // only draws those out of date
        unscare_mouse();
      }
      break;
    default:
      break;
  }

  return ret;
}

LineClass* FindLine(VertexClass* p1, VertexClass* p2)
{ LineClass* lc;
  BEGIN_3D_LIST
  if(target->Control == &LineControl)
  { lc = (LineClass*)target->Object;
      if(lc->P1 == p1 && lc->P2 == p2)
        return(lc);
  }
  END_3D_LIST
  return(NULL);
}

VertexClass* LastLineVertex = NULL;

double Included_Angle(int x1, int y1, int x2, int y2, int x3, int y3)
{
  double a1 = atan2((double)y2-y1,(double)x2-x1);
  double a2 = atan2((double)y2-y1,(double)x3-x1);
  return(a1 - a2);
}

bool Right_Quadrant(int x1, int y1, int x2, int y2, int x3, int y3, double tolerance)
{
  if(x1 == x2 && y1 == y2)  // same spot
    return true;
  double ia = Included_Angle(x1,y1,x2,y2,x3,y3);
  return(ia <= tolerance && ia >= -tolerance);
}

VisualLinkClass* Last_Line = NULL;

int Drag_New_Object(void)
{
    int x, y, x1, y1, x2, y2;
    VertexClass* p1 = NULL;
    x = mouse_x;
    y = mouse_y;
    x1 = Current_Visual->X1;
    x2 = Current_Visual->X2;
    y1 = Current_Visual->Y1;
    y2 = Current_Visual->Y2;
    if(New_Object != &VertexControl &&
       Target3DList->ObjectSnap)          // see VisuList.cpp
    { if(FindAll)   // use 2D find instead of 3D which is limited to the drawing plane
        target = ((VisualListClass*)Target3DList)->Find_Mouse_Object();
      else
        target = Target3DList->Find_Mouse_Object();
      if(target)
      { p1 = (VertexClass*)target->Object;
        // could display onscreen parameters of captured object??
        x = target->X+target->W/2; // should be near enough
        y = target->Y+target->H/2;
      }
    } else if(Something_Found) {    // cursor is currently highlighting a vertex
        x = Something_Found->X+Something_Found->W/2; // should be near enough
        y = Something_Found->Y+Something_Found->H/2;
    }
    XRef = x;  // for on screen display of relative mouse position
    YRef = y;
    CrossHair_Limits(x1,y1,x2,y2);
    Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
// was Drag_Rectangle
    if(New_Object)   // an ObjectControl has been selected
    { // create a new object
      // first ensure that only vertex's are looked for?
      long temptypes = Current_Visual->IncludeTypes;
      Current_Visual->IncludeTypes = 1;  // that is bit 1 for Vertex
      if(New_Object == &VertexControl)
      { // should look to see if an existing vertex exists at this point?
        target = Target3DList->Find_Object(x2,y2);
        if(!target)
        { target = Target3DList->Add(New_Object,x2,y2,1,1);
          if(target)
            target->SizeLocked = true;
        }
      }
      else if(New_Object == &LineControl ||
              New_Object == &ArcControl ||
              New_Object == &SplineControl ||
              New_Object == &InsertControl )
      { // should only find points of the current level and block
        if(!p1)
        { target = Target3DList->Find_Object(x1,y1);
          if(!target)
          { if(MENU_ACTION_AUTOVERTEX )  // autovertex
            { target = Target3DList->Add(&VertexControl,x1,y1,1,1);
              if(target)
              { target->SizeLocked = true;
                p1 = (VertexClass*)(target->Object);
              }
            }
            else // if no start point found then use last point
              p1 = LastLineVertex;
          }
          else
            p1 = (VertexClass*)(target->Object);
        }
        if(Target3DList->ObjectSnap)
          target = Target3DList->Find_Mouse_Object();
        else
        { target = Target3DList->Find_Object(x2,y2);
          // make sure this is a sensible selection - ie. at least in the right direction
          // may not be necessary with grid select limits
          if(target && !Right_Quadrant(x1,y1,x2,y2,target->X,target->Y,PI/8))
            target = NULL;
        }
        VertexClass* p2= NULL;
        if(target)
        { p2 = (VertexClass*)(target->Object);
          // for arc second point can be the same as the first
          if((p2 == p1 && New_Object != &ArcControl && New_Object != &InsertControl) ||
             (p2->Layer != p1->Layer))
          { // if(target)  // an object was created so target it
            Target3DList->SelectOnly(target);
            // restore include types for normal usage
            Current_Visual->IncludeTypes = temptypes;
            return D_REDRAW;
            // should p1 be deleted?
          }
        }
        if(!p2)
        { if(MENU_ACTION_AUTOVERTEX )  // autovertex
          { target = Target3DList->Add(&VertexControl,x2,y2,1,1);
            if(target)
            { target->SizeLocked = true;
              p2 = (VertexClass*)(target->Object);
            }
          }
        }
        LastLineVertex = p2;
        if(p1 && p2 &&
           (p1!=p2 ||
            New_Object == &ArcControl ||       // p1==p2 means a circle
            New_Object == &InsertControl ) &&  // Inserts don't always have two points
            p1->Layer == p2->Layer &&
            (New_Object == &InsertControl || !FindLine(p1,p2)))
        { target = Target3DList->Add(New_Object,x1,y1,x2-x1,y2-y1); // no need to worry about limits
          // they will be reset with the object update at the next view draw
          if(target) {
           if(New_Object == &InsertControl)
           { InsertClass* insert = (InsertClass*)(target->Object);
             insert->P1 = p1;
             if(p2 != p1)
               insert->P2 = p2;
             else
               insert->P2 = NULL;
             // could request insert type here
           } else {
            LineClass* line = (LineClass*)(target->Object);
            line->P1 = p1;
            p1->References++;
            line->P2 = p2;
            p2->References++;
            // for spline link previous reference
            if(Last_Line &&          // Last_Line returns to null if not continuous
                (Last_Line->Control == &LineControl  ||  // just checking
                 Last_Line->Control == &ArcControl  ||
                 Last_Line->Control == &SplineControl) &&
                ((LineClass*)(Last_Line->Object))->P2 == p1)  // new line starts at end of last
                {
                LineClass* lastline = (LineClass*)(Last_Line->Object);
                if(New_Object == &SplineControl) {
                  line->L1 = lastline;
                }
                lastline->L2 = line;
                if(Last_Line->Control == &SplineControl) {
                  ((SplineClass*)(Last_Line->Object))->Flags &= (SPLINE_V2_CALC ^ 0xFFFF); // recalc if not locked
// the following made redundant by Class variable in LineClass
//                  if(New_Object == &SplineControl) {
//                    ((SplineClass*)line)->Flags |= SPLINE_L1_SPLINE;
//                    ((SplineClass*)(Last_Line->Object))->Flags |= SPLINE_L2_SPLINE;
//                  }
                }
            }
            Last_Line = target;
            // but there's more points for arcs
            if(New_Object == &ArcControl) {
              ((ArcClass*)line)->P3 = NULL;  // temporary
              target->Indicate(screen);
              int xc = x1;
              int yc = y1;
              x = x2;
              y = y2;
              CrossHair_Limits(x1,y1,x2,y2);
              Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                   ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
              // needs to be on release of mouse, so simples way is to repeat
              CrossHair_Limits(x1,y1,x2,y2);
              Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                   ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
              if(Target3DList->ObjectSnap)
                target = Target3DList->Find_Mouse_Object();
              else
                target = Target3DList->Find_Object(x2,y2);
              VertexClass* p3= NULL;
              if(target)
                LastLineVertex = p3 = (VertexClass*)(target->Object);
              else
              { if(MENU_ACTION_AUTOVERTEX )  // autovertex
                { target = Target3DList->Add(&VertexControl,x2,y2,1,1);
                  if(target)
                  { target->SizeLocked = true;
                    p3 = (VertexClass*)(target->Object);
                  }
                }
              }
              if(p3 != p1 && p3 != p2 && p3->Layer == p2->Layer)
              { ((ArcClass*)line)->P3 = p3;
                p3->References++;
                if(p1 == p2)   // intended to be a circle
                  ((ArcClass*)line)->Flags |= ARC_FULL_CIRCLE;
                // calc x2 & y2 for 90deg angle at radius
                int dx = x2 - xc;
                int dy = y2 - yc;
                x2 -= dy;
                y2 += dx;
                // this will the circle a plane in the 3D
                // since it is generated, very little chance of repeat
                target = Target3DList->Add(&VertexControl,x2,y2,1,1);
                if(target)
                { target->SizeLocked = true;
                  VertexClass* p4 = (VertexClass*)(target->Object);
                  ((ArcClass*)line)->P4 = p4;
                }
              }
            }
           }
          }
        }
      }
      else if(New_Object == &FaceControl)
      { // should only find points of the current level and block
        if(!p1)
        { target = Target3DList->Find_Object(x1,y1);
          if(!target)
          { if(MENU_ACTION_AUTOVERTEX )  // autovertex
            { target = Target3DList->Add(&VertexControl,x1,y1,1,1);
              if(target)
              { target->SizeLocked = true;
                p1 = (VertexClass*)(target->Object);
              }
            }
            else // if no start point found then use last point
              p1 = LastLineVertex;
          }
          else
            p1 = (VertexClass*)(target->Object);
        }
        if(Target3DList->ObjectSnap)
          target = Target3DList->Find_Mouse_Object();
        else
          target = Target3DList->Find_Object(x2,y2);
        VertexClass* p2= NULL;
        if(target)
          LastLineVertex = p2 = (VertexClass*)(target->Object);
        else
        { if(MENU_ACTION_AUTOVERTEX )  // autovertex
          { target = Target3DList->Add(&VertexControl,x2,y2,1,1);
            if(target)
            { target->SizeLocked = true;
              p2 = (VertexClass*)(target->Object);
            }
          }
        }
        if(p1 && p2 && p1!=p2 &&
          p1->Layer == p2->Layer)
        { VisualLinkClass* vl = Target3DList->Add(New_Object,x1,y1,x2-x1,y2-y1);
          // no need to worry about limits
          // they will be reset with the object update at the next view draw
          if(vl)
          { FaceClass* face = (FaceClass*)(vl->Object);
            face->P1 = p1;
            p1->References++;
            face->P2 = p2;
            p2->References++;
            face->Texture = Action_Texture;
            vl->Indicate(screen);
            // but there's more points
            x = x2;
            y = y2;
            CrossHair_Limits(x1,y1,x2,y2);
            Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
            if(Target3DList->ObjectSnap)
              target = Target3DList->Find_Mouse_Object();
            else
              target = Target3DList->Find_Object(x2,y2);
            VertexClass* p3= NULL;
            if(target)
              LastLineVertex = p3 = (VertexClass*)(target->Object);
            else
            { if(MENU_ACTION_AUTOVERTEX )  // autovertex
              { target = Target3DList->Add(&VertexControl,x2,y2,1,1);
                if(target)
                { target->SizeLocked = true;
                  p3 = (VertexClass*)(target->Object);
                }
              }
            }
            if(p3 != p1 && p3 != p2 && p3->Layer == p2->Layer)
            { face->P3 = p3;
              p3->References++;
            }
            vl->Indicate(screen);
            // but there's one more point, or their might be
            x = x2;
            y = y2;
            CrossHair_Limits(x1,y1,x2,y2);
            Drag_CrossHairs(x, y, &x1, &y1, &x2, &y2,
                 ((Menu_View[SNAP_O].flags & D_SELECTED) != 0) | 6);
            if(Target3DList->ObjectSnap)
              target = Target3DList->Find_Mouse_Object();
            else
              target = Target3DList->Find_Object(x2,y2);
            VertexClass* p4 = NULL;
            if(target)
              LastLineVertex = p4 = (VertexClass*)(target->Object);
            else
            { if(MENU_ACTION_AUTOVERTEX )  // autovertex
              { target = Target3DList->Add(&VertexControl,x2,y2,1,1);
                if(target)
                { target->SizeLocked = true;
                  p4 = (VertexClass*)(target->Object);
                }
              }
            }
            if(p4 != p1 && p4 != p2 && p4 != p3 && p4->Layer == p2->Layer)
            { face->P4 = p4;
              p4->References++;
            } else {
              face->P4 = NULL;
            }
            target = vl;
          }
        }
      }
      else if(New_Object == &SolidControl)
      { // in this case must always use a new point as it might get moved
        target = Target3DList->Add(&VertexControl,x1,y1,1,1);
        if(target)
        { target->SizeLocked = true;
          p1 = (VertexClass*)(target->Object);
          VTX q2;
          Current_Visual->FlatToReal(&q2,x2,y2);
          fixed mx,my,mz,dx,dy,dz;
          mx = MIN(p1->Pos.X,q2.X);
          dx = ABS(p1->Pos.X-q2.X);
          my = MIN(p1->Pos.Y,q2.Y);
          dy = ABS(p1->Pos.Y-q2.Y);
          q2.Z = 0;
          mz = MIN(p1->Pos.Z,q2.Z);   // solids always flat to z at moment
          dz = ABS(p1->Pos.Z-q2.Z);   // can always alter in edit
          p1->Pos.Set(mx,my,mz);
          VisualLinkClass* vl = Target3DList->Add(New_Object,mx,my,dx,dy);
          if(vl)
          { SolidClass* solid = (SolidClass*)(vl->Object);
            if(solid)
            { solid->P = p1;
              p1->References++;
              solid->CreateVolume(dx,dy,dz);
              // p2->References++;  - p2 not referenced so can be discarded
              vl->Indicate(screen);
            }
          }
        }
      }
      else   // use default add method - dangerous if object requires VERTEX
        target = Target3DList->Add(New_Object,x1,y1,x2-x1+1,y2-y1+1);
        // currently poor protection in ControlTypes for NULL Vertex
      if(target)  // an object was created
        Target3DList->SelectOnly(target);
      // restore include types for normal usage
      Current_Visual->IncludeTypes = temptypes;
    }
    return D_REDRAW;
}


/******************************************/
/********** The menu functions ************/
/******************************************/



int Snaper(void)
{
  Target3DList->Snap_Selected();
  return D_REDRAW;
}


/* a simple 'about' box */
int About(void)
{
    billalert(" About Edit3D V1.0 (released under GNU General Pulic Licence)",
    "Copyright (C) 2004 Robert Parker (begun 2001)",
    "using Allegro "ALLEGRO_VERSION_STR" and Bgui V2.0. This instance compiled using "ALLEGRO_PLATFORM_STR,
    "&Ok", NULL, 'o', 0);
    return D_O_K;
}

int Check_New(void)     // called from object.cpp
{
  if(Menu_Action)  // remove existing action down to the first seperator
  { for(int n=0; Menu_Action[n].text && Menu_Action[n].text[0]; n++)
      Menu_Action[n].flags = 0;
  }
  Menu_Action[0].flags |= D_SELECTED;
  if(MENU_ACTION_ADD)
    Menu_right_click[0].child = Menu_new; // found in object.cpp
  return D_OK;
}

int Action_Checker(void)
{
  if(Menu_Action)
  { for(int n=0; Menu_Action[n].text; n++)
      Menu_Action[n].flags = 0;
  }
  active_menu->flags ^= D_SELECTED;
  if(MENU_ACTION_EDIT)
    Menu_right_click = Menu_right_click_edit;
  else
    Menu_right_click = Menu_right_click_norm;
  return D_OK;
}

int Auto_Vertex_Checker(void)
{ active_menu->flags ^= D_SELECTED;
/*  No need but may return in future
  if(active_menu->flags & D_SELECTED)
    (active_menu+1)->flags = 0;  // deselect Snap_Vertex
*/
  return D_OK;
}

int Snap_Vertex_Checker(void)
{ active_menu->flags ^= D_SELECTED;
/*  No need, but may return in future
  if(active_menu->flags & D_SELECTED)
    (active_menu-1)->flags = 0; // deselect Auto_Vertex
*/
  if(Target3DList)
    Target3DList->ObjectSnap = ((active_menu->flags & D_SELECTED) != 0);
  return D_OK;
}

int Find_All_Checker(void)
{ active_menu->flags ^= D_SELECTED;
  FindAll = ((active_menu->flags & D_SELECTED) != 0);
  return D_OK;
}

/* checks/unchecks the 'use default font' menu item */
int Font_Checker(void)
{
  active_menu->flags ^= D_SELECTED;

  if(FontFile && !(Menu_Options[3].flags & D_SELECTED))
    font = (FONT*)FontFile[0].dat;
  else
    font = &_default_font;

  editor[1].w = 0;
  SEND_MESSAGE(&editor[1], MSG_START, 0);

  return D_REDRAW;
}

int Gui_Color_Setter()
{
  int colors[3];
  colors[0] = Back_fg;
  colors[1] = Back_mg;
  colors[2] = Back_bg;
  char* back_str[] = { "Foreground","Midground","Background" };
  if(Color_Setter(colors,3,back_str))
  { Back_fg = colors[0];
    Back_mg = colors[1];
    Back_bg = colors[2];
    return D_REDRAW;
  }
  return D_OK;
}

/* changes the GUI global settings */
int Gui_Setter(void)
{
    char bl[30];
    int res;

    DIALOG gui_dlg[] =
    {
       /* (proc)            (x)  (y)  (w)  (h)  (fg) (bg) (key) (flags) (d1) (d2) (dp)                   (dp2)           (dp3) */
       { d_billwin_proc,    0,   0,   248, 116, 0,   0,   0,    0,      0,   0,   (void*)" Target settings",    NULL,           NULL },
       { d_billbutton_proc, 10,  20,  80,  16,  0,   0,   13,   D_EXIT, 0,   0,   (void*)"Color",               NULL,           NULL },
       { d_billbutton_proc, 100, 20,  80,  16,  0,   0,   13,   D_EXIT, 0,   0,   (void*)"Restore",             NULL,           NULL },
       { d_billcheck_proc,  8,   60,  164, 8,   0,   0,   0,    0,      0,   0,   (void*)"Focus follows mouse", NULL,           NULL },
       { d_billtext_proc,   8,   40,  144, 8,   0,   0,   0,    0,      0,   0,   (void*)"Font baseline",       NULL,           NULL },
       { Xedit_proc,        160, 36,  80,  16,  0,   0,   0,    0,      9,   0,   bl,                           NULL,           NULL },
       { d_billbutton_proc, 60,  92,  48,  16,  0,   0,   13,   D_EXIT, 0,   0,   (void*)"OK",                  NULL,           NULL },
       { d_billbutton_proc, 112, 92,  72,  16,  0,   0,   27,   D_EXIT, 0,   0,   (void*)"Cancel",              NULL,           NULL },
       { NULL,              0,   0,   0,   0,   0,   0,   0,    0,      0,   0,   NULL,                         NULL,           NULL }
    };

    usprintf(bl, "%d", Back_baseline);

    if(Back_mouse_focus)
      gui_dlg[9].flags = D_SELECTED;
    else
      gui_dlg[9].flags = 0;
    
    // Do the dialog
    set_dialog_color(gui_dlg, gui_fg_color, gui_bg_color);
    centre_dialog(gui_dlg);
    res = popup_dialog(gui_dlg, 5);

    // Read data back
    if(res == 1)
      Gui_Color_Setter();
    else if(res == 2)
      Set_Editor_Gui_State();
    else if(res == 6)
    { Back_baseline = ustrtol(bl, NULL, 0);
      Back_mouse_focus = gui_dlg[4].flags & D_SELECTED;
    }
    Set_Back_Gui_State();  // won't actually effect BGUI proceedures
    return D_REDRAW;
}

int Color_Selected()
{
  int cf = Back_fg;
  int cm = Back_mg;
  int cb = Back_bg;
  // Back colors are used for defaults
  Gui_Color_Setter();

  BEGIN_3D_LIST
    if(target->Selected)
      target->Control->SetColor(Back_fg,Back_mg,Back_bg);
  END_3D_LIST

  Back_fg = cf;
  Back_mg = cm;
  Back_bg = cb;

  return D_REDRAW;
}




/**********************************************/
/******* Source code handling functions *******/
/**********************************************/


/* get a copy of a string, allocating the memory */
char *Get_Copy(char *str)
{
    char *res = (char*)malloc(NCHAR);

    if(res)
      ustrcpy(res, str);

    return res;
}



/******************************************/
/******* The main editor functions ********/
/******************************************/


/* restore editor settings */
void Set_Editor_Gui_State(void)
{
    if(FontFile && !(Menu_Options[3].flags & D_SELECTED))
      font = (FONT*)FontFile[0].dat;
    else
      font = &_default_font;

    gui_fg_color = makecol(0,   0,   0);    // black
    gui_mg_color = makecol(127, 127, 127);  // grey
    gui_bg_color = makecol(255, 255, 255);  // white
    gui_font_baseline = 0;
    gui_mouse_focus = 1;

    Back_font = font;
    Back_fg = gui_fg_color;
    Back_mg = gui_mg_color;
    Back_bg = gui_bg_color;
    Back_baseline = gui_font_baseline;
    Back_mouse_focus = gui_mouse_focus ^ D_SELECTED;
}

void Set_Back_Gui_State(void)
{
    font = Back_font;
    gui_fg_color = Back_fg;
    gui_mg_color = Back_mg;
    gui_bg_color = Back_bg;
    gui_font_baseline = Back_baseline;
    gui_mouse_focus = Back_mouse_focus ^ D_SELECTED;
}

void (*Notify_Advance)(void*) = NULL;

void Mouse_Search_Notify(void* pntr)
{ if(pntr == Mouse_Target)
    Mouse_Target = NULL;
  if(pntr == Something_Found)
    Something_Found = NULL;
  if(pntr == Something_Selected)
    Something_Selected = NULL;
  if(Notify_Advance)        // go to the next notify where ever it may be
    Notify_Advance(pntr);
}

char StatusStr[80];

void Update_Mouse()
{   // update the mouse cursor
  static int oldmx, oldmy;
    if(Editor_Player->mouse_obj == 0)
    {
      BITMAP *new_pointer = _mouse_pointer;
      // mouse focus point
      int fx = 0;
      int fy = 0;
      int mx = mouse_x;
      int my = mouse_y;

      int mchoice = 0;
      if(mx > Visual[0].X2 && mx < Visual[1].X1)
        mchoice |= 1;
      if(my > Visual[0].Y2 && my < Visual[2].Y1)
        mchoice |= 2;
      switch(mchoice)
      { case 1:
          new_pointer = resize_pointer_h;
          fx = 5;
          fy = 3;
          break;
        case 2:
          new_pointer = resize_pointer_v;
          fx = 3;
          fy = 5;
          break;
        case 3:
          new_pointer = resize_pointer_d1;
          fx = 5;
          fy = 5;
          break;
        default:
          if(my > ViewsY1 && my < ViewsY2 && mx > ViewsX1 && mx < ViewsX2)
          { // search for objects in view space
            if(Mouse_Target)   // continue previous search
              Mouse_Target = (VisualLinkClass*)(Mouse_Target->NextLink());
            if(!Mouse_Target)  // start or restart search
              Mouse_Target = (VisualLinkClass*)(Target3DList->FirstLink());
            // only look for vertex
            // only look for vertex from current block and layer
            // possibly have some special arrangement for allowing extras
            if(Mouse_Target &&
                ((Mouse_Target->Control == &VertexControl) || MENU_ACTION_EDIT)
              )
            { if(Mouse_Target->Selected)
                Something_Selected = Mouse_Target;
              if(FindAll) {
                if(Mouse_Target->Inside(mx,my))
                  Something_Found = Mouse_Target;
              } else {
                if(((VisualLink3DClass*)Mouse_Target)->Inside(mx,my,
                      fixtoi(Current_Visual->Depth)))
                  Something_Found = Mouse_Target;
              }
            }
            // select approriate cursor
            if(MENU_ACTION_ADD ||
               MENU_ACTION_EDIT)   // add or paste modes
            { if(Something_Found)
                new_pointer = selection_pointer2;
              else
                new_pointer = crosshair_pointer;
              fx = 3;
              fy = 3;
            }
            else if(MENU_ACTION_ZOOM)  // zoom mode
            { new_pointer = magglass_pointer;
              fx = 3;
              fy = 3;
            }
            else if(MENU_ACTION_EDIT)  // edit mode
            { if(Something_Selected && Something_Selected->Selected &&
                  !((key_shifts & KB_CTRL_FLAG) || (key_shifts & KB_SHIFT_FLAG)))
                new_pointer = crosshair_pointer;
              else if(Something_Found)
                new_pointer = selection_pointer2;
              else
                new_pointer = selection_pointer1;
              fx = 3;
              fy = 3;
            }
            // if required - display mouse text positions
            if(mx != oldmx || my != oldmy)
            { if(Status_Disp)   // defined in DrawRect.cpp but uses PositionStatus in View.cpp
              { oldmx = mx;
                oldmy = my;
                int sm = Status_Disp_Mode;
                Status_Disp_Mode = (Status_Disp_Mode/3)*3;  // no relative or polar relevant
                scare_mouse();
                if(Something_Found)
                { int id = FindIndex(Something_Found->Object);
                  if(Something_Found->Control == &VertexControl)
                  { StatusVTX = &(((VertexClass*)Something_Found->Object)->Pos);
                    sprintf(StatusStr,"VERTEX %d",id);  // edit.cpp
                  } else if(Something_Found->Control == &LineControl) {
                    sprintf(StatusStr,"LINE %d",id);
                  } else if(Something_Found->Control == &SplineControl) {
                    sprintf(StatusStr,"SPLINE %d VECTOR %d",id,SplineControl.Vector);
                  } else {
                    sprintf(StatusStr,"UNKNOWN %d",id);
                  }
                  StatusMsg = StatusStr;
                } else {
                  StatusVTX = NULL;
                  StatusMsg = NULL;
                }
                Status_Disp(screen,mx,my,0,0);
                unscare_mouse();
                Status_Disp_Mode = sm;
              }
              Something_Found = NULL;
            }
          }
          break;
      }

/* all this only applies to dragging edges
      VisualLinkClass* t = Target3DList->Find_Mouse_Object();
      if(t)   // yes object found
      {
        if(mx < t->X+3)
        {
          if(my < t->Y+3)
            new_pointer = resize_pointer_d1;
          else if(my >= t->Y+t->H-3)
            new_pointer = resize_pointer_d2;
          else
            new_pointer = resize_pointer_h;
        }
        else if(mx >= t->X+t->W-3)
        {
          if(my < t->Y+3)
            new_pointer = resize_pointer_d2;
          else if(my >= t->Y+t->H-3)
            new_pointer = resize_pointer_d1;
          else
            new_pointer = resize_pointer_h;
        }
        else if(my < t->Y+3 || my >= t->Y+t->H-3)
          new_pointer = resize_pointer_v;
      }
*/
      if(mouse_sprite != new_pointer && new_pointer)
      {
        scare_mouse();
        set_mouse_sprite(new_pointer);
        set_mouse_sprite_focus(fx,fy);
        if(new_pointer != _mouse_pointer)
          set_mouse_sprite_focus(new_pointer->w/2, new_pointer->h/2);
      }
      unscare_mouse();
    }
}

BITMAP* dummybmp = NULL;

/* editor initialization */
char Initialized = FALSE;
void Init_Editor(void)
{
  if(!Initialized)
  {
    Initialized = TRUE;

    FontFile = load_datafile("font.dat");
    if(!FontFile)
      Menu_Options[3].flags = D_DISABLED;
    else
      Menu_Options[3].flags = 0;
    Register_Object_Set((ObjectControlClass**)ControlClasses);
    Menu_Action[0].child = Menu_right_click[0].child = Menu_new; // found in object.cpp
    Menu_Action[1].flags |= D_SELECTED;  // zoom mode
    CheckMother = Check_New;  // pass Menu_new a function to check Add option
    Help_File_FID = "Edit3D.HLP";
    Init_Grid();
    Init_Mouse();

    // tool bar
    editor[2].dp = magglass_pointer;
    editor[2].w = editor[2].h = 10;
    editor[2].flags ^= D_HIDDEN;
    ViewsY1 = 34;
    Target3DList = &TemporaryList;   // current edit target - Target3DList lives in vislst3d.cpp
    TargetList = &TemporaryList;     // to retain 2D functions - TargetList ,, ,, visulist.cpp
    Link_Notify = Mouse_Search_Notify;
    // Target list MUST be set before Init_Views()
    Init_Views();

  }
}

/* editor cleanup */
void Deinit_Editor(void)
{
    int i;
    
    if(!Initialized)
      return;
    Initialized = FALSE;
    DeInit_Views();
    Deinit_Mouse();
    unload_datafile(FontFile);
}

/* execute the dialog editor */
void run_editor(void)
{
  show_mouse(screen);
  Init_Editor();
  clear_keybuf();
  Set_Editor_Gui_State();
  Editor_Player = init_dialog(editor, -1);

  while(update_dialog(Editor_Player))
  { Update_Mouse();
    // do own stuff
  }

  shutdown_dialog(Editor_Player);
  // Set_Back_Gui_State();
  Editor_Player = NULL;
  Deinit_Editor();
}

// all the following were transferred from init.cpp as it was being a pain


/* This is a function that will initialize all that is needed by our custom
 * procs, and will take care of run-time initialized params (like BITMAPs).
 * We can change the gui_* variables of allegro, and the font too, they will
 * affect the edited dialog (but _not_ the editor).
 */

void bgui_procs_init(void)
{
    /* If you want the edited dialog (_not_ the editor) to use my thin font,
     * just uncomment these lines.
     */

    /*
    if((datafile = load_datafile("font.dat")) != NULL)
	font = datafile[0].dat;
    */
    bill_init();
//    bill_init_smallpal(48);  // after, NOT before bill_init
// as long as all the bill colors (greys plus dark blue) are available, this is
// not necessary
    set_gui_mode(1);
}

/* RGB -> color mapping table. Not needed, but speeds things up */
RGB_MAP RGB_table;

/* The main function of the editor. Here we just need to call our init
 * function, and register_proc_set().
 */
//int main(int argc, char *argv[])
void main()
{
  int argc = 0;
  char* argv[] = {"Hello" };
  
  int i;
  int bpp = 0;    // will load config or revert to 8 bit if not changed by argument
  int w   = 0;    // ditto to 640
  int h   = 0;    // ditto to 400
  char *path = NULL;
  char *name = NULL;

  set_uformat(U_ASCII);  // or U_UTF8 - either are usually 8 bits at least for 0 - 127
  set_config_file("config.cfg");  
  allegro_init();

    // processing of command line arguments
  for(i=1; i<argc; i++)
  {
    if((argv[i][0] == '-') || (argv[i][0] == '/')) // alternative arg markers
    { switch(argv[i][1])
      { case 'w':
          w = atoi(argv[i]+2);
          break;
        case 'h':
          h = atoi(argv[i]+2);
          break;
        case 'b':
          bpp = atoi(argv[i]+2);
          break;
      }
    }
    else
    {
      if(!path)
        path = argv[i];
      else
        name = argv[i];
    }
  }

  // setting up the editor
  install_keyboard();
  install_mouse();
  install_timer();

  Screen_Mode = new AllegroScreenClass(w,h,bpp);

  PALETTE pal;
  get_palette(pal);
  Rainbow_Palette(pal);
  set_pallete(pal);
  create_rgb_table(&RGB_table, pal, NULL); // print_progress);
  rgb_map = &RGB_table;

  // gui colors not actually used by bgui proceedures
  gui_fg_color = makecol(0, 0, 0);
  gui_bg_color = makecol(255, 255, 255);
  Init_EGAColors();
  // no need to create small pallet for bgui, the required colors are in rainbow_pallette
  bgui_procs_init();
  // Adding support for custom procs
  //register_proc_set(bgui_procs, "bgui");

  // Launching the editor
  // if(path)
  //   load_target(path, name);  // replace target with what ever your editiing
  run_editor();
  delete Screen_Mode;
 // return 0;
}

END_OF_MAIN();
