/*
 *    Implementation of Quixo Game, (c) Maciej Kobuszewski
 *    files: dialogs.c, dialogs.h  -  menu dialogs
 *           draw.c, draw.h  -  drawing windows
 *           board.c, board.h  -  calculating moves etc.
 *           ai.c, ai.h  -  AI of computer player
 *           quixo.c, quixo.h  -  main file
 */

#include <allegro.h>
#include <math.h>
#include "quixo.h"
#include "draw.h"
#include "board.h"
#include "dialogs.h"
#include "ai.h"

/*  GLOBAL VARIABLES  */
int exit_program = 0;                   /* set this to exit the program */
int back = 0;
int selected = -1;
int who = WHO_STARTS;
int one_player_mode = -1;
int first = 1;
int game_over = 0;
int block_board = 0;

void clear_kbd(int k)
{
    do
    {
    	rest(1);
    	poll_keyboard();
    }
    while (key[k]);

    clear_keybuf();
}

int get_command_button()
{
    /* check the position of the mouse with respect to the command buttons
     return 0 for back, -1 for nothing
     back */
    if (mouse_x <= BUTTON_BACK_X + BUTTON_BACK_W && mouse_x >= BUTTON_BACK_X)
    {
        if (mouse_y <= BUTTON_BACK_Y + BUTTON_BACK_H && mouse_y >= BUTTON_BACK_Y) 
        {
        	return 0;
        }
    }
    /* miss */
    return -1;
}

/* 0 : nothing happened
   1 : a button was clicked and dealt with */
int handle_buttons()
{
    int b;
    
    /* see if a button was clicked */
    b = get_command_button();

    switch (b)
    {
	case 0:
	    /* back */
	    back = 1;
	    break;
	default:
	    /* missed all buttons */
	    return 0;
	    break;
    }

    return 1;
}

int get_mouse_pos()
{
    int x, y;
    
    if(mouse_x - BOARD_X > 0)
       x = (int)floor((mouse_x - BOARD_X)/BLOCK_SIZE);
    else
       x = -1;
    if(mouse_y - BOARD_Y > 0)    
       y = (int)floor((mouse_y - BOARD_Y)/BLOCK_SIZE);
    else 
       y = -1;

    if (x < 0 || x > 4 || y < 0 || y > 4) 
    {
    	x=-1;
    	y=-1;
    }
   	
    return y*5+x;
} 

int get_move() {
    int pos = -1;
    back = 0;
    draw_screen();
    while (!(mouse_b & 1))
    {
        poll_keyboard();
        if (key[KEY_ESC])
        {
            if(game_over) 
            {
                game_over = 0;
                clear_kbd(KEY_ESC);
            }
            else
                back = 1;
            return -1;
        }
        /* wait patiently */
        rest(1);
    }
    /* mouse was clicked */
    if (handle_buttons())
    {
    	/* a command button was clicked and dealt with
    	   wait until mouse is released */
    	while (mouse_b & 1) {
            draw_screen();
        }
   	    return -1;
    }
    if(!block_board) 
    {
        /* get the move coordinates */
        pos = get_mouse_pos();
    
        /* wait until it is released */
        while (mouse_b & 1)
    	rest(1);
    	;
    	return pos;
        return -1;
    }
}

int main_loop() {
    int pos;
    if(check_gameover() && !block_board) {
        game_over = 1;
        block_board = 1;
    }
    if(one_player_mode && who == COMPUTER_PLAYER)
    {
        alpha_beta(DEPTH, -10000, 10000, board, who);
        move(temp_pos, &temp_spos, board, who);
        who = 3-who;
    } 
    else
    {
        pos = get_move();
        select_block(pos);
        if(validate_move(pos, selected, who, board))
        {
            move(pos, &selected, board, who);
            who = 3-who;
        }
    }
    draw_screen();
    
    /* check if a key was pressed. This is necessary to determine if the
       menu was requested during a computer turn (otherwise you can't quit
       in a 2 computer game). Because the key may have been pressed
       earlier, use readkey to detect it. Because MANY keys may have been
       pressed, clear the from the buffer before going into the menu :) */
    if (keypressed())
    {
    	if (readkey() >> 8 == KEY_ESC)
    	{
    	    clear_kbd(KEY_ESC);
    	    return 0;
    	}
    }
	if (back)
	{
	    clear_kbd(KEY_ESC);
	    return 0;
	}
    draw_screen();
    return 1;
}

int main(int argc, char *argv[])
{
   int ret;

   /* initialise everything */
   if (allegro_init() != 0)
      return 1;
   install_keyboard(); 
   install_mouse();
   install_timer();

   set_color_depth(32);
   if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, SCREEN_WIDTH, SCREEN_HEIGHT, 0, 0) != 0) {
	 set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
	 allegro_message("Unable to set any graphic mode\n%s\n", allegro_error);
	 return 1;
   }
   
   /* init gfx */
   if (!init_graphics())
      return 0;
   
   /* draw the dialog */
   do_main_dialog();
   
   while (1)
   {
       /* see if we should stop */
       if (exit_program)
           return 0;
      
       /* do nothing */
       ret = main_loop();
       
       /* menu requested? */
       if (!ret)
       {
           clear_keybuf();
           do_main_dialog();
       }
   }
   
   return 0;
}
END_OF_MAIN()
