/* sfh - shit from heaven
 * Copyright (C) 2002 David A. Capello
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <allegro.h>
#include <allegro/internal/aintern.h>
#include "blood.h"
#include "car.h"
#include "data.h"
#include "game.h"
#include "level.h"
#include "linklist.h"
#include "mymath.h"
#include "people.h"
#include "player.h"
#include "scorer.h"



#define WALKING        1
#define LOOKING        2
#define IDLING         4
#define CROSSING       8
#define WANT_IDLE      16
#define WANT_CROSS     32
#define LOCK_CAR       64



typedef struct VICTIM
{
  int flags;
  int time, ani_time;
  int x, y;
  int dx, dy;
  int old_x, old_y;
  fixed angle;
  fixed old_angle;
  int hits;
} VICTIM;

static LINKLIST *people;
static int more_people_time;

static void get_victim_info(VICTIM *victim, int *x, int *y, int from_old);
static int segment_line(fixed x, fixed y, fixed x1, fixed y1, fixed x2, fixed y2);
static int point_in_poly(fixed x, fixed y, fixed xs[4], fixed ys[4], int vertices);



void init_people(void)
{
  people = create_linklist(sizeof(VICTIM));
  more_people_time = 0;
}



void shutdown_people(void)
{
  delete_linklist(people);
}



static void move_victim(VICTIM *victim)
{
#define MOVE(to_x, to_y) \
  tile = get_level_tile(victim->x+(to_x), victim->y+(to_y)); \
  if (!(tile & HAS_PERSON)) {                                \
    set_level_tile(victim->x, victim->y, FALSE);             \
    victim->x += (to_x);                                     \
    victim->y += (to_y);                                     \
    set_level_tile(victim->x, victim->y, TRUE);              \
    return;                                                  \
  }

  int x, tile = 0;

  /* remove the victim (he is outside) */
  if ((victim->x < 0) || (victim->x >= LEVEL_W) ||
      (victim->y < 0) || (victim->y >= LEVEL_H)) {
    if (victim->flags & LOCK_CAR)
      unlock_autos();

    remove_linkitem(people, victim);
    return;
  }

  /* old position */
  victim->old_x = victim->x;
  victim->old_y = victim->y;
  victim->old_angle = victim->angle;

  /* is walking */
  if (victim->flags & WALKING) {
    /* want idle */
    if ((victim->flags & WANT_IDLE) &&
        (get_level_tile(victim->x, victim->y) & IDLE_TILE) &&
        ((rand() & 7) == 1)) {
      victim->flags ^= WANT_IDLE;
      victim->flags ^= WALKING;
      victim->flags |= IDLING;
      victim->ani_time = 0;
      return;
    }

    /* want cross */
    if ((victim->flags & WANT_CROSS) &&
        (get_level_tile(victim->x, victim->y) & CROSS_TILE) &&
        (get_auto() == 0) &&
        ((rand() & 7) == 1)) {
      lock_autos();

      victim->flags ^= WANT_CROSS;
      victim->flags ^= WALKING;
      victim->flags |= CROSSING | LOCK_CAR;
      victim->dx = (victim->x == 1)? +1: -1;
      return;
    }

    /* normal walk */
    MOVE(victim->dx, victim->dy);

    /* change the side */
    x = ((victim->x == 0) || (victim->x == LEVEL_W-2))? +1:
        ((victim->x == 1) || (victim->x == LEVEL_W-1))? -1: victim->dx;
    MOVE(x, victim->dy);

    /* try other things to avoid breaks */
    victim->flags |= WANT_CROSS;
  }
  /* is idle */
  else if (victim->flags & IDLING) {
    if (victim->ani_time > FPS*2) {
      victim->flags ^= IDLING;
      victim->flags ^= WALKING;
      victim->ani_time = 0;
    }
  }
  /* is crossing */
  else if (victim->flags & CROSSING) {
    /* x-direction */
    x = victim->dx;

    /* the victim become the other sidewalk */
    tile = get_level_tile(victim->x+x, victim->y);
    if (tile & CROSS_TILE) {
      if (victim->flags & LOCK_CAR) {
        victim->flags ^= LOCK_CAR;
        unlock_autos();
      }
      victim->flags ^= CROSSING;
      victim->flags |= WALKING;
      victim->dx = 0;
    }

    /* move the victim */
    MOVE(x, 0);
    MOVE(x, victim->dy);
    MOVE(x, -victim->dy);
  }
}



void update_people(void)
{
  VICTIM *victim, *next;

  for (victim=getfirst_linkitem(people); victim; victim=next) {
    next = getnext_linkitem(people, victim);

    /* collision with some car */
    if (victim->flags & CROSSING) {
      BITMAP *body = the_bitmaps[BODY1_1_PCX];
      fixed xs[4], ys[4];
      int x1, y1, x2, y2;
      int xp, yp;
      int ret;
  
      get_victim_info(victim, &xp, &yp, TRUE);
  
      _rotate_scale_flip_coordinates(
        itofix(body->w),
        itofix(body->h),
        itofix(xp),
        itofix(yp),
        itofix(body->w/2),
        itofix(body->h/2),
        victim->angle,
        ftofix(0.8),
        ftofix(0.8),
        FALSE, FALSE, xs, ys);
  
      x1 = fixtoi(MIN(xs[0], MIN(xs[1], MIN(xs[2], xs[3]))));
      y1 = fixtoi(MIN(ys[0], MIN(ys[1], MIN(ys[2], ys[3]))));
      x2 = fixtoi(MAX(xs[0], MAX(xs[1], MAX(xs[2], xs[3]))));
      y2 = fixtoi(MAX(ys[0], MAX(ys[1], MAX(ys[2], ys[3]))));

      if (hit_auto(x1, y1, x2, y2)) {
        if (victim->flags & LOCK_CAR)
          unlock_autos();
        else {
          add_score(xp, yp, 10000);
          add_player_score(10000);
        }

        /* insert the blood */
        add_blood_squirt(xp, yp, 64.0);
        /* kill this victim */
        set_level_tile(victim->x, victim->y, FALSE);
        remove_linkitem(people, victim);
        continue;
      }
    }

    /* move this victim */
    if (!(victim->flags & LOOKING)) {
      if (victim->flags & (WALKING | CROSSING)) {
        victim->ani_time++;
        if (victim->ani_time > FPS/2) {
          victim->ani_time = 0;
          move_victim(victim);
        }
      }
      else if (victim->flags & IDLING) {
        victim->ani_time++;
        move_victim(victim);
      }
    }
    else {
      victim->time++;
      if (victim->time > FPS) {
        victim->time = 0;
        victim->flags ^= LOOKING;
        victim->hits = 0;
      }
    }
  }

  /* insert more people */
  more_people_time++;
  if (more_people_time > FPS) {
    more_people_time = 0;
    add_people();
  }
}



void draw_people(BITMAP *bmp, int first)
{
  BITMAP *body, *head;
  VICTIM *victim;
  int x, y;

  for (victim=getfirst_linkitem(people); victim;
       victim=getnext_linkitem(people, victim)) {
    if (!(victim->flags & LOOKING)) {
      if (victim->flags & (WALKING | CROSSING)) {
        if ((victim->old_x != victim->x) ||
            (victim->old_y != victim->y)) {
          if (victim->ani_time < FPS*1/8)
            body = the_bitmaps[BODY1_1_PCX];
          else if (victim->ani_time < FPS*2/8)
            body = the_bitmaps[BODY1_2_PCX];
          else if (victim->ani_time < FPS*3/8)
            body = the_bitmaps[BODY1_1_PCX];
          else
            body = the_bitmaps[BODY1_3_PCX];
        }
        else
          body = the_bitmaps[BODY1_1_PCX];

        head = the_bitmaps[HEAD1_N_PCX];
      }
      else if (victim->flags & IDLING) {
        body = the_bitmaps[BODY1_1_PCX];
        head = the_bitmaps[HEAD1_N_PCX];
      }
    }
    else {
      body = the_bitmaps[BODY1_1_PCX];
      head = the_bitmaps[HEAD1_L_PCX];
    }

    get_victim_info(victim, &x, &y, TRUE);

    if (first) {
      BITMAP *shadow = create_bitmap(body->w, body->h);

      text_mode(0);
      draw_character(shadow, body, 0, 0, makecol(0, 0, 0));

      x = GSCREEN_W/2 + (x - GSCREEN_W/2) * 0.95;
      y = GSCREEN_H/2 + (y - GSCREEN_H/2) * 0.95;

      pivot_scaled_sprite(bmp, shadow, x, y,
        body->w/2, body->h/2, victim->angle, ftofix(0.78));

      destroy_bitmap(shadow);
    }
    else {
      pivot_scaled_sprite(bmp, body, x, y,
        body->w/2, body->h/2, victim->angle, ftofix(0.8));

      pivot_scaled_sprite(bmp, head, x, y,
        head->w/2, head->h/2, victim->angle, ftofix(0.8));
    }
  }
}



void add_people(void)
{
  int x_to_test[4] = { 0, 1, LEVEL_H-2, LEVEL_H-1 };
  int y_to_test[2] = { 0, LEVEL_H-1 };
  int x_avail[8];
  int y_avail[8];
  VICTIM *victim;
  int c, count;
  int x, y;

  count = 0;
  for (y=0; y<2; y++)
    for (x=0; x<4; x++) {
      if (~(get_level_tile(x_to_test[x], y_to_test[y]) & HAS_PERSON)) {
        x_avail[count] = x_to_test[x];
        y_avail[count] = y_to_test[y];
        count++;
      }
    }

  if (count < 1)
    return;

  victim = insert_linkitem(people);

  victim->time = victim->ani_time = 0;
  victim->flags = WALKING
                | ((rand() & 1)? WANT_CROSS: 0)
                | ((rand() & 1)? WANT_IDLE: 0);

  c = rand() % count;

  victim->x = x_avail[c];
  victim->y = y_avail[c];
  victim->dx = 0;
  victim->dy = (y_avail[c] == 0)? 1: -1;
  victim->old_x = victim->x;
  victim->old_y = (y_avail[c] == 0)? -1: LEVEL_H;
  victim->hits = 0;

  get_victim_info(victim, &x, &y, FALSE);
  victim->old_angle = victim->angle;

  set_level_tile(victim->x, victim->y, TRUE);
}



void hit_people(int x, int y)
{
  BITMAP *body = the_bitmaps[BODY1_1_PCX];
  VICTIM *victim;
  fixed xs[4], ys[4];
  int xp, yp;

  for (victim=getfirst_linkitem(people); victim;
       victim=getnext_linkitem(people, victim)) {
    get_victim_info(victim, &xp, &yp, TRUE);

    _rotate_scale_flip_coordinates(
      itofix(body->w),
      itofix(body->h),
      itofix(xp),
      itofix(yp),
      itofix(body->w/2),
      itofix(body->h/2),
      victim->angle,
      ftofix(0.8),
      ftofix(0.8),
      FALSE, FALSE, xs, ys);

    if (point_in_poly(itofix(x), itofix(y), xs, ys, 4)) {
      int score = (!victim->hits)? 100: 500*victim->hits;

      add_score(xp, yp, score);
      add_player_score(score);

      victim->time = 0;
      victim->flags |= LOOKING;
      victim->hits++;

      if (victim->flags & LOCK_CAR) {
        victim->flags ^= LOCK_CAR;
        unlock_autos();

        /* insert (in force way) a car to kill this victim */
        add_auto(!(victim->x >= LEVEL_W/2));
      }
    }
  }
}



static void get_victim_info(VICTIM *victim, int *x, int *y, int from_old)
{
  *x = LEFT_CORNER
     + (victim->old_x*TILE_W)
     + ((victim->x*TILE_W)-(victim->old_x*TILE_W)) * victim->ani_time / (FPS/2)
     + TILE_W/2;

  *y = TOP_CORNER
     + (victim->old_y*TILE_H)
     + ((victim->y*TILE_H)-(victim->old_y*TILE_H)) * victim->ani_time / (FPS/2)
     + TILE_H/2;

  if (victim->flags & WALKING) {
    victim->angle = ftofix(128.0*atan2(victim->dy, victim->dx)/M_PI-64.0);
  }
  else if (victim->flags & IDLING) {
    if (victim->x == 0)
      victim->angle = ftofix(+64.0);
    else
      victim->angle = ftofix(-64.0);
  }
  else if (victim->flags & CROSSING) {
    victim->angle = ftofix(128.0*atan2(0, victim->dx)/M_PI-64.0);
  }

  /* make the range from old to new */
  if (from_old) {
    victim->angle =
      fixadd(victim->old_angle,
             fixsub(victim->angle,
                    victim->old_angle) * victim->ani_time / (FPS/2));
  }
}



static int segment_line(fixed x, fixed y, fixed x1, fixed y1, fixed x2, fixed y2)
{
  int min_x, max_x;

  if (x1 == x2)
    return FALSE;

  min_x = MIN(x1, x2);
  max_x = MAX(x1, x2);

  if ((x >= min_x) && (x <= max_x) &&
      (y <= fixadd(y1,
                   fixdiv(fixmul(fixsub(y2, y1),
                                 fixsub(x, x1)),
                          fixsub(x2, x1)))))
    return TRUE;
  else
    return FALSE;
}



static int point_in_poly(fixed x, fixed y, fixed xs[4], fixed ys[4], int vertices)
{
  int c, count = 0;

  for (c=0; c<vertices-1; c++)
    if (segment_line(x, y, xs[c], ys[c], xs[c+1], ys[c+1]))
      count++;

  if (segment_line(x, y, xs[c], ys[c], xs[0], ys[0]))
    count++;

  return (count & 1)? TRUE: FALSE;
}




