#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <allegro.h>
#include "vm.h"

THREAD *create_thread(const char *name)
{
  THREAD *thread=(THREAD*)malloc(sizeof(THREAD));
  thread->name=strdup(name);
  thread->x=0;
  thread->x=0;
  thread->data=0;
  thread->ip=0;
  thread->sp=0;
  thread->code[0]=INSN_END;
  return thread;
}

void destroy_thread(THREAD *thread)
{
  free(thread->name);
  free(thread);
}

static void thread_error(THREAD *thread,const char *message)
{
  static char culprit[256];
  sprintf(culprit,"insn %d",thread->ip/2);
  alert(thread->name,message,culprit,"OK","",13,27);
  fprintf(stderr,"%s: %s\n",thread->name,message);
}

static void push(THREAD *thread,int imm)
{
  if (thread->sp==256) {
    thread_error(thread,"Stack overflow");
    return;
  }
  thread->stack[thread->sp++]=imm;
}

static int pop(THREAD *thread)
{
  if (thread->sp==0) {
    thread_error(thread,"Stack underflow");
    return 0;
  }
  return thread->stack[--thread->sp];
}

static void settop(THREAD *thread,int imm)
{
  thread->stack[thread->sp-1]=imm;
}

static int gettop(THREAD *thread)
{
  if (thread->sp==0) {
    thread_error(thread,"Stack underflow");
    return 0;
  }
  return thread->stack[thread->sp-1];
}

int do_insn(THREAD *thread,int time)
{
  int end=0;
  int i1,i2,i3;
  const int insn=thread->code[thread->ip];
  const int param=thread->code[thread->ip+1];
  thread->ip+=2;
  switch (insn) {
    case INSN_NOP:
      break;
    case INSN_END:
      thread->sp=0;
      thread->ip=0;
      end=1;
      break;
    case INSN_YIELD:
      end=1;
      break;
    case INSN_PUSHTIME:
      push(thread,time);
      break;
    case INSN_PUSHX:
      push(thread,thread->x);
      break;
    case INSN_PUSHY:
      push(thread,thread->y);
      break;
    case INSN_POPX:
      thread->x=pop(thread);
      break;
    case INSN_POPY:
      thread->y=pop(thread);
      break;
    case INSN_PUSH:
      push(thread,param);
      break;
    case INSN_POP:
      pop(thread);
      break;
    case INSN_ADD:
      i1=pop(thread);
      i2=pop(thread);
      push(thread,i1+i2);
      break;
    case INSN_SUB:
      i1=pop(thread);
      i2=pop(thread);
      push(thread,i2-i1);
      break;
    case INSN_MUL:
      i1=pop(thread);
      i2=pop(thread);
      push(thread,i1*i2);
      break;
    case INSN_DIV:
      i1=pop(thread);
      i2=pop(thread);
      if (i1==0) {
        thread_error(thread,"Division by zero");
        break;
      }
      push(thread,i2/i1);
      break;
    case INSN_MOD:
      i1=pop(thread);
      i2=pop(thread);
      if (i1==0) {
        thread_error(thread,"Modulo by zero");
        break;
      }
      push(thread,i2%i1);
      break;
    case INSN_SIN:
      i1=pop(thread);
      i2=pop(thread);
      push(thread,fixtoi(i1*fsin(itofix(i2))));
      break;
    case INSN_COS:
      i1=pop(thread);
      i2=pop(thread);
      push(thread,fixtoi(i1*fcos(itofix(i2))));
      break;
    case INSN_RAND:
      i1=pop(thread);
      push(thread,i1?rand()%i1:0);
      break;
    case INSN_AJZ:
      i1=pop(thread);
      if (!i1) {
        thread->ip=param*2;
      }
      break;
    case INSN_RJZ:
      i1=pop(thread);
      if (!i1) {
        thread->ip+=(param-1)*2;
      }
      break;
    case INSN_PUSHD:
      push(thread,thread->data);
      break;
    case INSN_POPD:
      thread->data=pop(thread);
      break;
    case INSN_PUSHIP:
      push(thread,(thread->ip-2)/2);
      break;
    case INSN_PEEK:
      i1=pop(thread);
      push(thread,thread->code[i1*2]);
      push(thread,thread->code[i1*2+1]);
      break;
    case INSN_POKE:
      i1=pop(thread);
      i2=pop(thread);
      i3=pop(thread);
      thread->code[i3*2]=i2;
      thread->code[i3*2+1]=i1;
      break;
    case INSN_PUSHW:
      push(thread,SCREEN_W);
      break;
    case INSN_PUSHH:
      push(thread,SCREEN_H);
      break;
    case INSN_NEG:
      i1=pop(thread);
      push(thread,-i1);
      break;
    default:
      thread_error(thread,"Unknown insn");
      break;
  }
  return end;
}

void run_thread(THREAD *thread,int time,int max_insns)
{
  while (max_insns--) {
    if (do_insn(thread,time)) return;
  }

  thread_error(thread,"Preempting runaway thread");
}
